#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: admin.py 10449 2017-11-07 06:22:13Z David $
#
# Copyright (c) 2012 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: David $
# $Date: 2017-11-07 14:22:13 +0800 (Tue, 07 Nov 2017) $
# $Revision: 10449 $

from django import forms
from django.contrib import admin
from django.utils.translation import ugettext as _

from emencia.django.newsletter.models import Newsletter
from emencia.django.newsletter.admin import NewsletterAdmin

from editor.TinyMCEWidget import CustomTinyMCEWidget

from selectable.forms import AutoCompleteWidget, AutoComboboxWidget

from Theophrastus.lookups import SenderLookup, ReplyLookup

class NewsletterTinyMCEForm(forms.ModelForm):

    def __init__(self, *args, **kwargs):
        
        newsletterObj = Newsletter.objects.all().order_by('-id')

        if newsletterObj:
            sender = newsletterObj[0].header_sender
            reply = newsletterObj[0].header_reply
        else:
            sender = getattr(
                settings,
                'NEWSLETTER_DEFAULT_HEADER_SENDER',
                "Nuwa<service@tracedig.com>"
            )
            reply = getattr(
                settings,
                'NEWSLETTER_DEFAULT_HEADER_REPLY',
                "Nuwa<service@tracedig.com>"
            )

        super(NewsletterTinyMCEForm, self).__init__(*args, **kwargs)
        self.fields['header_sender'].initial = sender
        self.fields['header_reply'].initial = reply

    content = forms.CharField(
        widget=CustomTinyMCEWidget(attrs={'cols': 150, 'rows': 80}),
        label=_("Content"))
    
    
    header_sender = forms.CharField(
        label=_('sender'),
        widget=AutoComboboxWidget(lookup_class=SenderLookup),
        required=True,
    )

    header_reply = forms.CharField(
        label=_('reply to'),
        widget=AutoComboboxWidget(lookup_class=ReplyLookup),
        required=True,
    )

    def clean_header_reply(self):
        """
        Validate reply value.
        """
        reply = self.cleaned_data["header_reply"]
        try:
            reply.encode()
        except UnicodeEncodeError:
            raise forms.ValidationError(
                _("Incorrect value for reply, it can only include English "
                "word, number and symbol."))
        return reply

    class Meta:
        model = Newsletter
        fields = '__all__'

class NewsletterTinyMCEAdmin(NewsletterAdmin):

    list_display = ('title', 'mailing_list', 'server', 'status',
                    'formatedSendingDate', 'formatedCreationDate',
                    'formatedModificationDate',
                    'historic_link', 'statistics_link')

    def formatedSendingDate(self, obj):
        return obj.sending_date.strftime("%Y-%m-%d %H:%M:%S")
    formatedSendingDate.short_description = _('sending date')

    def formatedCreationDate(self, obj):
        return obj.creation_date.strftime("%Y-%m-%d %H:%M:%S")
    formatedCreationDate.short_description = _('creation date')

    def formatedModificationDate(self, obj):
        return obj.modification_date.strftime("%Y-%m-%d %H:%M:%S")
    formatedModificationDate.short_description = _('modification date')

    class Media:
        try:
            js = [
                'js/Newsletter.js',
            ]
        except AttributeError:
            pass

    form = NewsletterTinyMCEForm

admin.site.unregister(Newsletter)
admin.site.register(Newsletter, NewsletterTinyMCEAdmin)

from emencia.django.newsletter.models import Contact
from emencia.django.newsletter.admin import ContactAdmin

class ContactTheophrastusAdmin(ContactAdmin):
    list_display = (
        'email', 'first_name', 'last_name', 'tags', 'tester', 'subscriber',
        'valid', 'total_subscriptions', 'formatedCreationDate',
        'related_object_admin')

    def formatedCreationDate(self, obj):
        return obj.creation_date.strftime("%Y-%m-%d %H:%M:%S")
    formatedCreationDate.short_description = _('creation date')

admin.site.unregister(Contact)
admin.site.register(Contact, ContactTheophrastusAdmin)
