#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Util.py 11858 2020-01-12 11:45:03Z Lavender $
#
# Copyright (c) 2018 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $ (last)
# $Date: 2020-01-12 20:45:03 +0900 (週日, 12 一月 2020) $
# $Revision: 11858 $

import sys
import six
import hashlib
import urllib

from collections import OrderedDict

from django.conf import settings

from Iuppiter.DjangoUtil import DJANGO_VERSION

if sys.version_info.major == 3:
    from urllib.parse import urlencode
else:
    from urllib import urlencode

from Iuppiter.Encoding import utf8

# ​https://github.com/vinta/python-pay2go

def toBytes(s):
    return s if six.PY2 else s.encode()

class Pay2GoMPGAPI(object):

    def __init__(self, hashKey, hashIv, sandbox, *args, **kwargs):
        self.hashKey = hashKey
        self.hashIv = hashIv
        if sandbox:
            self.ApiUrl = 'https://ccore.spgateway.com/MPG/mpg_gateway'
        else:
            self.ApiUrl = 'https://core.spgateway.com/MPG/mpg_gateway'

        self.requiredFields = (
            'Amt',
            'MerchantID',
            'MerchantOrderNo',
            'TimeStamp',
            'Version',
        )

    def buildFormData(self, data):
        def generateCheckValue(data):
            orderedDict = OrderedDict()

            orderedDict['HashKey'] = self.hashKey
            for field in self.requiredFields:
                orderedDict[field] = data[field]
            orderedDict['HashIV'] = self.hashIv

            checkValue = hashlib.sha256(
                urlencode(orderedDict).encode('utf-8')).hexdigest().upper()

            return checkValue

        formData = dict(data)
        formData['CheckValue'] = generateCheckValue(data)

        return formData

    def createPayment(self, data):
        payment = Payment(
            formAction=self.ApiUrl,
            formData=self.buildFormData(data)
        )

        return payment

CATHAPBK_SERVICE_URL = \
    'https://sslpayment.uwccb.com.tw/EPOSService/Payment/OrderInitial.aspx'

from django.template.loader import get_template
from django.template import Context

class CathaybkAPI(object):

    def __init__(self, merchantId, cubkey):
        self.merchantId = merchantId
        self.cubkey = cubkey

    def generateCheckValue(self, data):
        checkValue = hashlib.md5(toBytes(
            self.merchantId + data['ORDERNUMBER'] + 
            str(data['AMOUNT']) + self.cubkey)).hexdigest()
        return checkValue

    def buildXml(self, data):
        temp = get_template('shop/payment/Cathaybk.xml')
        if DJANGO_VERSION >= 11100: # django >= 1.11
            context = {
                'data': data,
                'checkValue': self.generateCheckValue(data),
                'merchantId': self.merchantId,
            }
        else:
            context = Context({
                'data': data,
                'checkValue': self.generateCheckValue(data),
                'merchantId': self.merchantId,
            })
        xml = temp.render(context)
        return xml

    def buildForm(self, data):
        xml = self.buildXml(data)
        temp = get_template('shop/payment/Cathaybk.html')
        if DJANGO_VERSION >= 11100: # django >= 1.11
            context = {
                'xml': xml,
                'apiUrl': CATHAPBK_SERVICE_URL,
            }
        else:
            context = Context({
                'xml': xml,
                'apiUrl': CATHAPBK_SERVICE_URL,
            })
        form = temp.render(context)
        return form
        


class Payment(object):

    def __init__(self, formAction, formData, *args, **kwargs):
        self.formAction = formAction
        self.formData = formData

class ECPayShopShipment(object):
    def __init__(self, hashKey, hashIv, sandbox, *args, **kwargs):
        self.hashKey = hashKey
        self.hashIv = hashIv
        if sandbox:
            self.ApiUrl = 'https://vendor-stage.ecpay.com.tw'
        else:
            self.ApiUrl = 'https://vendor.ecpay.com.tw'

def createCheckValue(data):
    data = OrderedDict(data)
    data = OrderedDict(sorted(data.items()))

    orderedDict = OrderedDict()
    orderedDict['HashKey'] = settings.ECPAY_API_HASH_KEY
    for field in data:
        orderedDict[field] = data[field]

    orderedDict['HashIV'] = settings.ECPAY_API_HASH_IV

    dataList = []
    for k, v in orderedDict.items():
        dataList.append("%s=%s" % (k, v))
    dataStr = u"&".join(dataList)
    
    if six.PY2:
        encodeStr = urllib.urlencode({'data': utf8(dataStr),})[5:]
        checkValue = hashlib.md5(encodeStr.lower()).hexdigest().upper()
    else:
        encodeStr = urllib.parse.urlencode({'data': utf8(dataStr),})[5:]
        checkValue = hashlib.md5(encodeStr.lower().encode()).hexdigest().upper()
    return checkValue