#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Decorator.py 12426 2020-07-26 06:36:50Z Bear $
#
# Copyright (c) 2020 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Bear $ (last)
# $Date: 2020-07-26 15:36:50 +0900 (週日, 26 七月 2020) $
# $Revision: 12426 $
"""
Define decorator related functions.
"""
import collections

def decorator(dec):
    """
    Transform a defined decorator to be a well behavior decorator which
    preserve the decorated function name, docstring, signature...etc.
    See section dealing-with-third-party-decorators-new-wrapper at
    http://micheles.googlecode.com/hg/decorator/documentation.html

    @param decorator A defined decorator.
    @return A well behavior decorator which its functionality is the same as
            decorator argument.
    """
    from decorator import FunctionMaker

    def wrapper(f):
        f = f if not hasattr(f, 'im_class') else f.__func__
        return FunctionMaker.create(f, 'return decorated(%(signature)s)',
                                    dict(decorated=dec(f)), undecorated=f)
    return wrapper

def removeDecorator(decorated, name=""):
    """
    Remove decorated function to original function.

    @param decorated Decorated function.
    @param name The original function name.
    @return Original function.
    """
    if name == "":
        name = decorated.__name__

    # Get the closure of function.
    # The closure stores binded variables in function.
    # Decorator function must have stored original function in its closure.
    closure = decorated.__closure__

    # Michele Simionato's decorator module stores original function in
    # func_globals with '_func_' key.
    globals = decorated.__globals__
    if closure:
        for cell in closure:
            f = cell.cell_contents
            if isinstance(f, collections.Callable):
                varname = f.__name__
                if varname == name:
                    if f.__closure__:
                        return removeDecorator(f, name)
                    else:
                        return f
                else:
                    r = removeDecorator(f, name)
                    if r == f:
                        continue
                    else:
                        return r
    # Decorated by decorator module.
    elif '_func_' in globals:
        if isinstance(globals['_func_'], collections.Callable):
            return removeDecorator(globals['_func_'], name)

    return decorated
