#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Model.py 11267 2018-11-23 08:23:13Z Kevin $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Kevin $ (last)
# $Date: 2018-11-23 17:23:13 +0900 (週五, 23 十一月 2018) $
# $Revision: 11267 $
"""
Django model utilities.
"""

import six

import logging

logger = logging.getLogger(__name__)

from Iuppiter.Decorator import decorator

from django.db import transaction as dj_transaction

# For backward compatibility.
# In django 1.2.3 transaction didn't support nesting,
# but the problem has been solved in django 1.6.
# Please see http://code.djangoproject.com/ticket/2227 for more detail.
transaction = decorator(dj_transaction.atomic)

import gc

from django.db import models


class ServiceManager(models.Manager):
    """
    Our own default manager to provide some extra features for service based
    architecture.
    """

    def create(self, _obj=None, **kws):
        """
        Create object.

        @param _obj The model's instance.
        @param **kws Model's attribute to create.
        @return A model instance.
        """
        if _obj:
            _obj.save()
            return _obj
        else:
            return super(self.__class__, self).create(**kws)

    def update(self, queryOrObj, **changes):
        """
        Update object.

        @param queryOrObj Object or the query of db.
        @param **changes the changes of object(s).
        @return A list of model id instance that updated.
        """
        
        if queryOrObj and isinstance(queryOrObj, models.Model):
            if changes:
                for k, v in changes.items():                
                    setattr(queryOrObj, k, v)
            queryOrObj.save()
            result = [queryOrObj.id]
        elif queryOrObj and isinstance(queryOrObj, dict):
            if changes:                
                querySet = self.filter(**queryOrObj)
                querySet.update(**changes)
                result = [obj.id for obj in querySet]
            else:
                raise RuntimeError(
                "when queryOrObj is a dict changes cannot be none.")
        else:
            raise RuntimeError(
                "queryOrObj must be a model instance or a dict.")        
        
        return result

    def delete(self, idOrObj=None, **kws):
        """
        Delete object data by given id or other conditions.
    
        @param idOrObj Object or its id.
        @param **kws Keyword arguments for conditions.
        @return A list of model instance that deleted (the id of objects will
                be remained).
        """
        # id
        if idOrObj and isinstance(idOrObj, int):
            result = [idOrObj]
            queryset = self.filter(id=idOrObj, **kws)
            queryset.delete()
        # obj
        elif idOrObj and isinstance(idOrObj, models.Model):
            result = [idOrObj.id]
            idOrObj.delete()
        elif idOrObj != None:
            raise RuntimeError(
               "idOrObj argument must be int or model instance.")
        else:
            queryset = self.filter(**kws)
            result = [obj.id for obj in queryset]
            queryset.delete()
    
        return result

    def checkExist(self, *args, **kws):
        """
        Check whether object exists.

        @param *args Q conditions.
        @param **kws Conditions.
        @return True if object exists, else false.
        """
        return self.filter(*args, **kws).exists()

    def get(self, errorOnNotFound=True, *args, **kws):
        """
        Get object by given arguments.

        @param errorOnNotFound If true, raise ObjectDoesNotExist exception
                               while it can't find any result.
        @param *args Arguments.
        @param **kws Keyword arguments.
        @return Object.
        """
        try:
            return super(self.__class__, self).get(*args, **kws)
        except self.model.DoesNotExist:
            if errorOnNotFound:
                raise
            else:
                return None

    def byIds(self, ids):
        """
        Find objects by given id list.

        @param ids List of object ids.
        @return List of objects.
        """
        return [obj for obj in list(self.in_bulk(ids).values())]

    def getIterator(self, queryset, chunksize=1000):
        """
        Get memory efficient iterator to iterate query set.
        Please see:
        http://www.mellowmorning.com/2010/03/03/
        django-query-set-iterator-for-really-large-querysets/
        for more details.

        @param queryset QuerySet.
        @param chunksize Chunk size to load data at once.
        @note That the implementation of the iterator does not support
              ordered query sets.
        """
        try:
            lastpk = queryset.order_by('-pk')[0].pk
        except IndexError: # Empty?
            return queryset
        else:
            def do(queryset):
                pk = 0
                queryset = queryset.order_by('pk')
                while pk < lastpk:
                    for row in queryset.filter(pk__gt=pk)[:chunksize]:
                        pk = row.pk
                        yield row
                    gc.collect()
            return do(queryset)


from django.db.models import base

class MetaBaseModelMixin(base.ModelBase):
    """
    Metaclass for BaseModelMixin to overwrite default manager of model.
    """
    if six.PY2:
        def __new__(cls, name, bases, attrs):
            # Skip ModelBase.__new__ to prevent duplicate
            # initializing from ModelBase.
            if 'BaseModelMixin' in [
                b.__name__ for b in bases if hasattr(b, '__name__')]:
                return type.__new__(cls, name, bases, attrs)
            else:
                return super(MetaBaseModelMixin, cls).__new__(
                    cls, name, bases, attrs)

    def __init__(self, *args, **kws):
        """
        Constructor.
        To set default manager to our own service manager.

        @param *args Arguments.
        @param **kws Keyword arguments.
        """
        if hasattr(self, 'objects') and type(self.objects) == models.Manager:
            self.add_to_class('objects', ServiceManager())

@six.add_metaclass(MetaBaseModelMixin)
class BaseModelMixin(object):
    """
    The BaseModelMixin is a mixin to provide basic implemention for all
    django models.
    """

import time
import datetime

import django

if django.__version__ == '1.8':
    from django.db.models.fields.subclassing import SubfieldBase

    @six.add_metaclass(SubfieldBase)
    class PreciseDateTimeField(models.DateTimeField):
        """
        This field stores the timestamp information as millisecond to avoid
        the lacking of standard implementation for timestamp field.
        """

        def __init__(self, *args, **kws):
            """
            Constructor.
            """
            super(PreciseDateTimeField, self).__init__(*args, **kws)

        def toSeconds(self, value):
            """
            Convert datetime object to seconds.

            @param value Datetime object.
            @return Seconds (float).
            """
            if not isinstance(value, datetime.datetime):
                raise ValueError('Not a valid datetime object.')

            seconds = time.mktime(value.timetuple())
            seconds += value.microsecond / 1000000.0
            return seconds

        def fromSeconds(self, value):
            """
            Convert float seconds to datetime object.

            @param value Seconds.
            @return Datetime object.
            """
            return datetime.datetime(
                microsecond=int((value - int(value)) * 1000000),
                *time.localtime(value)[0:6])

        def get_db_prep_save(self, value, connection):
            if not value:
                return models.Field.get_db_prep_save(self, value,
                                                     connection=connection)
            return self.toSeconds(value)

        def to_python(self, value):
            if value is None:
                return value

            if isinstance(value, datetime.datetime):
                return value

            if isinstance(value, float):
                return self.fromSeconds(value)
            else:
                raise ValueError('Not a valid value to be assigned to this field.')

        def get_prep_lookup(self, lookup_type, value):
            if lookup_type == 'range':
                value = [self.toSeconds(v) for v in value]

            return models.Field.get_prep_lookup(self, lookup_type, value)

        def get_prep_value(self, value):
            return value

        def get_db_prep_value(self, value):
            return value

        def get_internal_type(self):
            return 'FloatField'
