#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: ThreadPool.py 9427 2015-09-15 08:01:29Z Judy $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Judy $ (last)
# $Date: 2015-09-15 16:01:29 +0800 (週二, 15 九月 2015) $
# $Revision: 9427 $

from threadpool import makeRequests, WorkRequest, WorkerThread
from threadpool import NoResultsPending, NoWorkersAvailable
from threadpool import ThreadPool as _ThreadPool

class ThreadPool(object):
    def __init__(self, numWorkers, queueSize=0, resultQueueSize=0,
                 pollTimeout=5):
        """
        Create a thread pool.

        .. warning:
            If you set both ``queueSize`` and ``resultQueueSize`` to
            ``!= 0`` there is the possibilty of a deadlock, when the results
            queue is not pulled regularly and too many jobs are put in the work
            requests queue.
            To prevent this, always set ``timeout > 0`` when calling
            ``ThreadPool.putRequest()`` and catch ``Queue.Full`` exceptions.

        @param numWorkers The count of child threads.
        @param queueSize If ``queueSize > 0`` the size of the work
                         *request queue* is limited and the thread pool blocks
                         when the queue is full and it tries to put more work
                         requests in it (see ``putRequest`` method), unless
                         you also use a positive ``timeout`` value for
                         ``putRequest``.
        @param resultQueueSize If ``resultQueueSize > 0`` the size of the
                               *results queue* is limited and the worker
                               threads will block when the queue is full and
                               they try to put new results in it.
        @param pollTimeout The timeout for poll.
        """
        self.threadPool = _ThreadPool(
            numWorkers, q_size=queueSize, resq_size=resultQueueSize,
            poll_timeout=pollTimeout)

    def createWorkers(self, numWorkers, pollTimeout=5):
        """
        Add num_workers worker threads to the pool.

        @param numWorkers The count of child threads.
        @param pollTimeout Sets the interval in seconds (int or float) for how
                           often threads should check whether they are
                           dismissed, while waiting for requests.
        """
        self.threadPool.createWorkers(numWorkers, poll_timeout=pollTimeout)

    def dismissWorkers(self, numWorkers, doJoin=False):
        """
        Tell num_workers worker threads to quit after their current task.
        """
        self.threadPool.dismissWorkers(numWorkers, do_join=doJoin)

    def joinAllDismissedWorkers(self):
        """
        Perform Thread.join() on all worker threads that have been dismissed.
        """
        self.threadPool.joinAllDismissedWorkers()

    def putRequest(self, request, block=True, timeout=None):
        """
        Put work request into work queue and save its id for later.
        """
        self.threadPool.putRequest(request, block=block, timeout=timeout)

    def poll(self, block=False):
        """
        Process any new results in the queue.
        """
        self.threadPool.poll(block=block)

    def wait(self):
        """
        Wait for results, blocking until all have arrived.
        """
        self.threadPool.wait()
