#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: UtilTest.py 10960 2018-04-03 05:48:20Z Kevin $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Kevin $ (last)
# $Date: 2018-04-03 14:48:20 +0900 (週二, 03 四月 2018) $
# $Revision: 10960 $

"""
Test all utilities.
"""

import unittest
import os
import platform
import zipfile
import six
from Iuppiter import Util

class UtilTest(unittest.TestCase):
    """
    Test case for Util.
    """

    def testExtendUnique(self):
        """
        Test for extendUnique function.
        """
        self.assertEqual(Util.extendUnique((1, 2, 2, 3), (2, 4, 5)), (
            1, 2, 3, 4, 5
        ))

        self.assertEqual(Util.extendUnique((1, 2, 2, 3), (2, 4, 5), False), (
            1, 2, 2, 3, 4, 5
        ))

        self.assertEqual(Util.extendUnique(["a", "b", "c"], ["d", "e", "a"]), [
            "a", "b", "c", "d", "e"
        ])

        self.assertEqual(
            Util.extendUnique(["a", "b", "b"], ("e", "e", "b"), False), [
            "a", "b", "b", "e"
        ])

        self.assertEqual(
            Util.extendUnique(["a", "b", "c"], ("e", "f", "g"), False,
                              referenceKey="b", addBefore=False), [
            "a", "b", "e", "f", "g", "c"
        ])
        
        self.assertEqual(
            Util.extendUnique(["a", "b", "c"], ("e", "f", "g"), False,
                              referenceKey="b", addBefore=True), [
            "a", "e","f", "g", "b", "c" 
        ])
        
    def testEscapeUrlFragment(self):
        """
        Test for escapeUrlFragment function.
        """
        urls = [
            'http://ecshweb.pchome.com.tw/search/v1/#!mac/Scope=all&CurPage=1&'
            'SortParm=RNK&SortOrder=DC',
            '',
            'http://naga.servehttp.com/trac/mercurius/ticket/2037#comment:7',
            'http://ecshweb.pchome.com.tw:80/search/v1/#!mac/Scope=all&CurPage'
            '=1&SortParm=RNK&SortOrder=DC',
            'http://ecshweb.pchome.com.tw/search/v1/?send=true#!mac/Scope=all&'
            'CurPage=1&SortParm=RNK&SortOrder=DC',
            'http://ecshweb.pchome.com.tw:93/search/v1/?send=true#!mac/Scope=a'
            'll&CurPage=1&SortParm=RNK&SortOrder=DC',
        ]

        newUrls = [
            'http://ecshweb.pchome.com.tw/search/v1/?_escaped_fragment_=mac/Sc'
            'ope=all%26CurPage=1%26SortParm=RNK%26SortOrder=DC',
            '',
            'http://naga.servehttp.com/trac/mercurius/ticket/2037#comment:7',
            'http://ecshweb.pchome.com.tw:80/search/v1/?_escaped_fragment_=mac'
            '/Scope=all%26CurPage=1%26SortParm=RNK%26SortOrder=DC',
            'http://ecshweb.pchome.com.tw/search/v1/?send=true&_escaped_fragme'
            'nt_=mac/Scope=all%26CurPage=1%26SortParm=RNK%26SortOrder=DC',
            'http://ecshweb.pchome.com.tw:93/search/v1/?send=true&_escaped_fra'
            'gment_=mac/Scope=all%26CurPage=1%26SortParm=RNK%26SortOrder=DC',
        ]

        for (n, url) in enumerate(urls):
            r = Util.escapeUrlFragment(url)
            self.assertEqual(r, newUrls[n])

    def testRglob(self):
        """
        Test for rglob function.
        """
        self.assertFalse(Util.rglob('.', '*.never'))
        utilTests = Util.rglob(os.path.abspath(os.path.dirname(__file__)),
                               'UtilTest.py*')
        # #py3 In Python 3 .pyc files will be moved to __pycache__ folder, then
        # there should be only one file found (.py)
        # https://docs.python.org/3/whatsnew/
        # 3.2.html#pep-3147-pyc-repository-directories
        if six.PY2:
            self.assertTrue(len(utilTests) > 1)
        else:
            pyc = [filename for filename in utilTests
                if filename.endswith('pyc')]
            if pyc:
                self.assertTrue(len(utilTests) > 1)
            else:
                self.assertTrue(len(utilTests) == 1)
        
        self.assertTrue(os.path.abspath(__file__) in utilTests)

        pyFiles = Util.rglob(os.path.dirname(os.path.abspath(__file__)), 
                             '*.py', True)
        self.assertTrue(len(pyFiles) > 1)
        for p in pyFiles:
            self.assertTrue(p[0] == '"' and p[-1] == '"')

    def testRun(self):
        """
        Test for run function.
        """
        # test echo
        if platform.system() == "Linux":
            out = Util.run("echo", "abc", shell=False).strip()
        else:
            out = Util.run("echo", "abc", shell=True).strip()        
        self.assertEqual(out, b'abc')
        
        # test copy
        # #py3 In python 3, __file__ variable is like
        # D:/Mercurius/Iuppiter/Iuppiter/tests/UtilTest.py,
        # not legal path like
        # D:\\Mercurius\\Iuppiter\\Iuppiter\\tests\\UtilTest.py in windows.

        copied = os.path.abspath(__file__ + '2')

        if os.path.exists(copied):
            if platform.system() == "Linux":
                out = Util.run('rm', '"%s"' % copied, shell=False)
            else:
                out = Util.run('del', '"%s"' % copied, shell=True)
            self.assertFalse(os.path.exists(copied))
        
        if platform.system() == "Linux":
            out = Util.run('cp', '"%s" "%s"' % (__file__, copied),
                           shell=False)
        else:
            out = Util.run('copy', '"%s" "%s"' % (
                os.path.abspath(__file__), copied), shell=True)
        self.assertTrue(os.path.exists(copied))

        # test delete
        if platform.system() == "Linux":
            out = Util.run('rm', '"%s"' % copied, shell=False)
        else:
            out = Util.run('del', '"%s"' % copied, shell=True)
        self.assertFalse(os.path.exists(copied))

        import subprocess
        
        # test subprocess
        try:
            if platform.system() == "Linux":
                out = Util.run("echo", "abc", shell=False, expectedReturnCode=1)
            else:
                out = Util.run("echo", "abc", shell=True, expectedReturnCode=1)
            self.fail('subprocess.CalledProcessError should be raised.')
        except subprocess.CalledProcessError:
            # This should be ok, because expectedReturnCode is None.
            if platform.system() == "Linux":
                out = Util.run("echo", "abc", 
                               shell=False, expectedReturnCode=None)
            else:
                out = Util.run("echo", "abc", 
                               shell=True, expectedReturnCode=None)

        # All in first argument case.
        import sys
        myEnv = os.environ.copy()

        if os.name == 'nt':
            myEnv['PATH'] = (os.path.dirname(sys.executable) +
                             ';' + myEnv['PATH'])
        else:
            myEnv['PATH'] = (os.path.dirname(sys.executable) +
                             ':' + myEnv['PATH'])

        out = Util.run('python -c "import os; ' 
                       'print(os.path.dirname(os.path.abspath(os.__file__)))"',
                       shell=True, env=myEnv)
        osdir = os.path.dirname(os.path.abspath(os.__file__))
        
        self.assertEqual(out.strip(), osdir.encode())

    def testArchiveZip(self):
        """
        Test for archiveZip function.
        """
        # File.
        Util.archiveZip(__file__, 'ArchiveZip-Test-1.zip')

        import zipfile
        h = zipfile.ZipFile('ArchiveZip-Test-1.zip', "r")
        try:
            self.assertTrue(os.path.split(__file__)[-1] in h.namelist())
        finally:
            h.close()
            os.remove('ArchiveZip-Test-1.zip')

        Util.archiveZip(__file__, 'ArchiveZip-Test-2.zip', fullPath=True)
        h = zipfile.ZipFile('ArchiveZip-Test-2.zip', "r")
        try:
            name = __file__
            if os.name == 'nt':
                name = name.replace('\\', '/')
                if name.find(':/') != -1:
                    name = name[name.find(':/') + 2:]
            if platform.system() == "Linux":
                name = name.lstrip("/")
            self.assertTrue(name in h.namelist())
        finally:
            h.close()
            os.remove('ArchiveZip-Test-2.zip')

        # Folder.
        folder = os.path.join(os.path.abspath(os.path.dirname(__file__)))
        Util.archiveZip(folder, 'ArchiveZip-Test-3.zip')
        h = zipfile.ZipFile('ArchiveZip-Test-3.zip', "r")
        try:
            self.assertTrue(os.path.split(__file__)[-1] in h.namelist())
            self.assertTrue('extension/__init__.py' in h.namelist())
            self.assertTrue('models/__init__.py' in h.namelist())
        finally:
            h.close()
            os.remove('ArchiveZip-Test-3.zip')

        Util.archiveZip(os.path.join(folder, 'TestData', 'Patch'), 
                        'ArchiveZip-Test-4.zip',
                        fullPath=True)
        h = zipfile.ZipFile('ArchiveZip-Test-4.zip', "r")
        try:
            prefix = folder
            if os.name == 'nt':
                prefix = prefix.replace('\\', '/')
                prefix = prefix[prefix.find(':/') + 2:]
            if platform.system():
                prefix = prefix.lstrip("/")

            self.assertTrue(prefix + 
                '/TestData/Patch/PatchTestPackage/'
                'PatchTestModule.py.patch.xml' in 
                h.namelist())
#             self.assertTrue(prefix + '/i18n/templatetags/__init__.py' in
#                             h.namelist())
        finally:
            h.close()
            os.remove('ArchiveZip-Test-4.zip')

    def testExtractZip(self):
        """
        Test for extractZip function.
        """
        import shutil

        if os.path.exists('ArchiveTest'):
            shutil.rmtree('ArchiveTest')

        # We should not make this test depend on archiveZip function, but
        # it is too trivial to zip a file from scratch for testing.
        try:
            Util.archiveZip(__file__, 'ArchiveZip-Test-1.zip')
            Util.extractZip('ArchiveZip-Test-1.zip', 'ArchiveTest')
        finally:
            os.remove('ArchiveZip-Test-1.zip')

        self.assertTrue(os.path.exists(
            os.path.join('ArchiveTest', os.path.split(__file__)[-1])))

        # Folder.
        folder = os.path.join(os.path.dirname(__file__), '..', 'test')
        try:
            Util.archiveZip(folder, 'ArchiveZip-Test-2.zip')
            Util.extractZip('ArchiveZip-Test-2.zip',
                            os.path.join('ArchiveTest', 'extractUtil'))
        finally:
            os.remove('ArchiveZip-Test-2.zip')

        self.assertTrue(os.path.exists(
            os.path.join('ArchiveTest', 'extractUtil', '__init__.py')))
        self.assertTrue(os.path.exists(
            os.path.join('ArchiveTest', 'extractUtil', 'Runner.py')))

        if os.path.exists('ArchiveTest'):
            shutil.rmtree('ArchiveTest')

    def testCopyTree(self):
        """
        Test for copyTree function.
        """
        import shutil

        here = os.path.dirname(os.path.abspath(__file__))
        dest = os.path.join(here, 'copy_tree_test')
        empty = os.path.join(here, 'empty')

        svn = os.path.exists(os.path.join(here, '.svn'))

        if os.path.exists(dest):
            shutil.rmtree(dest)
        if os.path.exists(empty):
            shutil.rmtree(empty)

        os.makedirs(empty)
        os.makedirs(os.path.join(empty, 'branch1'))
        os.makedirs(os.path.join(empty, 'branch2'))
        os.makedirs(os.path.join(empty, 'branch2', 'depth'))

        try:
            output = Util.copyTree(here, dest)

            f = os.path.join(dest, 'UtilTest.py')
            self.assertTrue(f in output)
            self.assertTrue(os.path.exists(f))

            f = os.path.join(dest, '__init__.py')
            self.assertTrue(f in output)
            self.assertTrue(os.path.exists(f))

            f = os.path.join(dest, 'models', '__init__.py')
            self.assertTrue(f in output)
            self.assertTrue(os.path.exists(f))

            skips = 1 if svn else 0

            # no copy_tree_test itself and .svn.
            self.assertEqual(len(os.listdir(here)) - 1 - skips,
                             len(os.listdir(dest)))
            # no .svn
            self.assertEqual(
                len(os.listdir(os.path.join(empty, 'branch2'))),
                len(os.listdir(os.path.join(dest, 'empty', 'branch2'))))
        finally:
            if os.path.exists(dest):
                shutil.rmtree(dest)
            if os.path.exists(empty):
                shutil.rmtree(empty)

    def testClassForName(self):
        """
        Test for classForName function.
        """
        iuppiter = Util.classForName('Iuppiter')
        import Iuppiter

        self.assertEqual(iuppiter, Iuppiter)

        ospath = Util.classForName('os.path')

        self.assertEqual(ospath, os.path)

        this = Util.classForName('Iuppiter.tests.UtilTest.UtilTest')
        self.assertEqual(type(this), type(UtilTest))

        this = Util.classForName('Iuppiter.tests.UtilTest')
        self.assertEqual(type(this), type(unittest))

        this = Util.classForName('Iuppiter.tests')
        self.assertEqual(type(this), type(unittest))

        try:
            Util.classForName('xx.xx.xx')
            self.fail('xx.xx.xx should not be a valid import statement.')
        except ImportError:
            pass

        # UNICODE string.
        this = Util.classForName(u'Iuppiter.tests.UtilTest.UtilTest')
        self.assertEqual(type(this), type(UtilTest))

    def testIsolationRef(self):
        """
        Test for IsolationRef class.
        """
        t = unittest.TestCase
        r = Util.IsolationRef(unittest)
        self.assertEqual(r.TestCase, unittest.TestCase)
        r.TestCase = 1
        self.assertEqual(r.TestCase, 1)
        self.assertEqual(unittest.TestCase, t)

    def testLRUDecorator(self):
        """
        Test for lruCache decorator.
        """
        # Simply test for performance only.

        def fib(n):
            if n < 2:
                return n
            return fib(n - 1) + fib(n - 2)

        import time
        c = time.clock()
        self.assertEqual(fib(30), 832040)
        e = time.clock() - c
        print(('fib elapsed:', e * 1000, 'ms.'))

        @Util.lruCache(500)
        def lruFib(n):
            if n < 2:
                return n
            return lruFib(n - 1) + lruFib(n - 2)

        c2 = time.clock()
        self.assertEqual(lruFib(30), 832040)
        e2 = time.clock() - c2
        print(('lruFib elapsed:', e2 * 1000, 'ms.'))

        self.assertTrue(e2 < e / 100.0) # At least 100 times faster.

    def testPreferredSort(self):
        """
        Test for preferredSort function.
        """

        self.assertEqual(Util.preferredSort(
            ["Apple", "Banana", "Cherry", "Peach"],
            ["Banana", "Peach"]),
            ["Banana", "Peach", "Apple", "Cherry"],
        )

        self.assertEqual(Util.preferredSort(
            ["Apple", "Banana", "Cherry", "Peach"],
            ["Banana", "Peach", "NOT IN"]),
            ["Banana", "Peach", "Apple", "Cherry"],
        )

        self.assertEqual(Util.preferredSort(
            [1, 2, 3, 4, 5],
            [4, 3]),
            [4, 3, 1, 2, 5],
        )

        self.assertEqual(Util.preferredSort(
            [5, 5, 5, 4, 3, 2, 1],
            [3, 2], reverse=True, inPlace=False),
            list(reversed([3, 2, 5, 5, 5, 4, 1])),
        )

    def testBuildReplacedZipBytes(self):
        zipFileName = os.path.join(
            os.path.abspath(os.path.dirname(__file__)),
            'TestData', '#3010', 'ZipFile.zip')

        dirName = os.path.join(
            os.path.abspath(os.path.dirname(__file__)),
            'TestData', '#3010', 'ReplacingDirectory')

        def assertContent(resultFile, data=None):
            with zipfile.ZipFile(resultFile, 'r') as f:

                self.assertLess(
                    sum([zinfo.compress_size for zinfo in f.filelist]),
                    sum([zinfo.file_size for zinfo in f.filelist])
                )

                for key, val in data.items():
                    if key == 'big_text':
                        continue
                    self.assertEqual(f.read(key), val)

            os.remove(os.path.abspath(resultFile))

        # Answer
        answer = {
            'a.txt': b'replace a.txt',
            'b.txt': b'bbbbb',
            'new.txt': b'new',
            'Dir1/c.txt': b'replace c.txt',
            'Dir1/d.txt': b'ddddd',
            'Dir1/new1.txt': b'new1',
            'Dir2/new2.txt': b'new2'
        }

        # case 1
        resultFile = Util.buildZipByDir(
            dirName, zipFileName, outputName='test.zip')
        assertContent(resultFile, data=answer)

        # case 2
        resultFile = Util.buildZipByDir(
            dirName, zipFileName, {'test.txt': b'test'})
        answer['test.txt'] = b'test'
        assertContent(resultFile, data=answer)

        # case 3
        resultFile = Util.buildZipByDir(
            dirName, zipFileName, {'test.txt': b'test'}, outputName='test.zip')
        assertContent(resultFile, data=answer)


if __name__ == '__main__':
    unittest.main()
