#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: admin.py 10982 2018-04-16 08:14:06Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2018-04-16 16:14:06 +0800 (週一, 16 四月 2018) $
# $Revision: 10982 $

import os
import datetime
import hashlib

from django.contrib import admin
from django import forms
from django.conf import settings
from django.contrib.admin.templatetags.admin_static import static
from django.forms import Media

# 介面拖拉上傳

# ------------------------------------------------------------------------------
from Zephyrus.template_editor.models import Static, Template

TEXT_SUB_FILE_NAME = [
    'txt', 'css', 'js', 'html', 'htm',
]

IMAGE_SUB_FILE_NAME = [
    'jpg', 'png', 'jpeg', 'bmp', 'gif', 'tif',
]


class FolderFrom(forms.ModelForm):
    class Meta(object):
        model = Static
        exclude = ('parent', 'type', 'name')

class TextFrom(forms.ModelForm):
    content = forms.CharField(widget=forms.Textarea, required=False)

    class Meta(object):
        model = Static
        exclude = ('parent', 'type', 'name')

class ImageFrom(forms.ModelForm):
    image = forms.ImageField(required=False)

    class Meta(object):
        model = Static
        exclude = ('parent', 'type', 'name')

class BaseFileAdmin(admin.ModelAdmin):
    change_list_template = 'template_editor/change_list.html'
    change_form_template = 'template_editor/change_form.html'

    def get_form(self, request, obj=None, **kwargs):
        if not obj:
            # add Form 新增能夠上傳檔案來新增 static 檔案
            return super(BaseFileAdmin, self).get_form(request, obj, **kwargs)

        # is folder
        if obj.type == 'folder':
            return FolderFrom

        # is file

        # -create path
        fullPath = obj.absolutePath

        # -get file form
        subFileName = obj.name.split('.')[-1].lower()

        if subFileName in TEXT_SUB_FILE_NAME:
            form = TextFrom
            with open(fullPath, 'rb') as f:
                data = f.read()
            form.base_fields['content'].initial = data
        elif subFileName in IMAGE_SUB_FILE_NAME:
            form = ImageFrom
        else:
            form = FolderFrom

        return form

    def change_view(self, request, object_id, form_url='', extra_context=None):
        extra_context = {}
        obj = self.model.objects.get(id=object_id)

        extra_context['filePath'] = obj.path

        # use checksum
        now = str(datetime.datetime.now()).encode('utf-8')
        extra_context['timeHash'] = hashlib.md5(now).hexdigest()

        subFileName = obj.name.split('.')[-1].lower()

        if subFileName in TEXT_SUB_FILE_NAME:
            extra_context['isImg'] = False
        elif subFileName in IMAGE_SUB_FILE_NAME:
            extra_context['isImg'] = True
        else:
            extra_context['isImg'] = False

        return super(BaseFileAdmin, self).change_view(
            request, object_id, form_url=form_url, extra_context=extra_context)

    def response_change(self, request, obj):
        if obj.type == 'file':
            subFileName = obj.name.split('.')[-1].lower()

            if subFileName in TEXT_SUB_FILE_NAME:
                # 能支援上傳修改檔案
                content = request.POST.get('content')
                
                fullPath = obj.absolutePath
                with open(fullPath, 'wb') as f:
                    data = f.write(content.encode('utf-8'))
            elif subFileName in IMAGE_SUB_FILE_NAME:
                # 圖片轉檔
                image = request.FILES.get('image')
                if image:
                    data = image.read()
                    image.close()

                    fullPath = obj.absolutePath

                    with open(fullPath, 'wb') as f:
                        data = f.write(data)

        return super(BaseFileAdmin, self).response_change(request, obj)

class StaticAdmin(BaseFileAdmin):

    def changelist_view(self, request, extra_context=None):
        roots = [node for node in Static.objects.all() if node.is_root_node()]
         
        extra_context = {
            'roots': roots,
            'isStatic': True,
        }
        
        return super(StaticAdmin, self).changelist_view(
            request,  extra_context=extra_context)

class TemplateAdmin(BaseFileAdmin):

    def changelist_view(self, request, extra_context=None):
        roots = [node for node in Template.objects.all() if node.is_root_node()]

        extra_context = {
            'roots': roots,
            'isStatic': False,
        }
        
        return super(TemplateAdmin, self).changelist_view(
            request,  extra_context=extra_context)

admin.site.register(Static, StaticAdmin)
admin.site.register(Template, TemplateAdmin)