#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: DistUtil.py 13434 2022-04-26 09:25:17Z Jason $
#
# Copyright (c) 2022 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Jason $ (last)
# $Date: 2022-04-26 17:25:17 +0800 (Tue, 26 Apr 2022) $
# $Revision: 13434 $
"""
Distribution utilities.
"""

import sys
import os
import platform

# Our extension location mapping layout.
EXTENSION_DIR_MAPPING = {
    '32bit': {
        'win32': 'Win32\\%s',
        'linux2': 'Linux_x86_%s',
    },
    '64bit': {
        'win32': 'Win64\\%s',
        'linux2': 'Linux_x64_%s',
    },
}

PATH = "."

def getSitePackagesDir():
    """
    Get location of site_packages for current python installation.

    @return Full path of site_packages.
    """
    # Better solution:
    # http://stackoverflow.com/questions/122327/
    # how-do-i-find-the-location-of-my-python-site-packages-directory
    try:
        from distutils.sysconfig import get_python_lib
        return get_python_lib()
    except ImportError:
        pass
    
    # if distribute installed, the pkg_resources will inside distribute, 
    # not correct path.
    import pkg_resources
    sitePackagesDir = os.path.dirname(pkg_resources.__file__)
    if not os.path.isdir(sitePackagesDir):
        sitePackagesDir = os.path.dirname(sitePackagesDir)
        
    # Test it.
    if os.path.exists(os.path.join(sitePackagesDir, 'Iuppiter')):
        return sitePackagesDir
        
    standardModulePath = os.path.dirname(os.path.abspath(os.__file__))
    if os.path.dirname(sitePackagesDir) == standardModulePath:
        return sitePackagesDir
    
    sitePackagesDir = os.path.join(standardModulePath, 'site-packages')
    if os.path.exists(sitePackagesDir):
        return sitePackagesDir
    
    try:
        import django            
        djangoInit = os.path.abspath(django.__file__)
        sitePackagesDir = os.path.dirname(os.path.dirname(djangoInit))
    except ImportError: 
        # Iuppiter might or might not tight with django, so this is possible.
        import Iuppiter
        iuppiterInit = os.path.abspath(__file__)            
        sitePackagesDir = os.path.dirname(os.path.dirname(iuppiterInit))

    return sitePackagesDir

def getExtensionPath(base='.', mode='Release', module=''):
    """
    Get extension location based on our extension layout convention.

    @param base Base directory.
    @param mode Build mode (Debug or Release).
    @param module Module name. Empty if you want to get directory path only.
    """
    bits = platform.architecture()[0]
    
    p = os.path.abspath(base)
    dir = os.path.join(p, EXTENSION_DIR_MAPPING[bits][sys.platform] % mode)
    if module:
        ext = ''
        if sys.platform.startswith('win'):
            ext = 'pyd'
        else:
            ext = 'so'

        dir = os.path.join(dir, '%s.%s' % (module, ext))

    return dir

import shutil

def supportReinstall(package):
    """
    Support python setup.py reinstall command.

    @param package Package name.
    """
    if 'reinstall' in sys.argv:        
        # In higher version of setuptools, pkg_resources.get_distribution won't
        # always get package from site-packages first, if there is .egg-info 
        # folder here, it will get this path.
        sitePackagesDir = getSitePackagesDir()
        eggInfo = '%s.egg-info' % package
        if (os.path.abspath(os.curdir) != sitePackagesDir and 
            os.path.exists(eggInfo)):
            shutil.rmtree(eggInfo)
            if os.path.exists('build'):
                shutil.rmtree('build')
            if os.path.exists('dist'):
                shutil.rmtree('dist')                
        
        # First time the pkg_resources imported, it will cache the working set
        # so we must remove .egg-info folder before import it.
        # setuptools will setup the working set,
        # so setuptools must be import after pkg_resources
        
        import pkg_resources        
        try:
            d = pkg_resources.get_distribution(package)
            if d:
                d = os.path.join(d.location, package)
                if os.path.exists(d):
                    shutil.rmtree(d)
        except pkg_resources.DistributionNotFound:
            pass
            
        # If there are the same package name but older versions...
        # See #2761.
        dirs = os.listdir(sitePackagesDir)
        for d in dirs:
            if d.startswith('%s-' % package):
                shutil.rmtree(os.path.join(sitePackagesDir, d))

        # Clean up 'build' folder.
        d = 'build'
        if os.path.exists(d):
            shutil.rmtree(d)

        sys.argv[sys.argv.index('reinstall')] = 'install'

# Python2's open function has no encoding option but we need it,
# so we import io.open to handle it
# see details: https://docs.python.org/2/library/io.html#io.open
if sys.version_info.major == 2:
    from io import open

def getRequirements(requirementFile='REQUIREMENTS.txt'):
    """
    Get project requirements.
    Read requirements file and return a list of requirements.
    @param requirementFile The package requirements file.
                           Default: REQUIREMENTS.txt
    @return requirements, extraRequirements list.
    """

    requirements = []
    extraRequirements = []

    if not os.path.exists(requirementFile):
        return requirements, extraRequirements

    with open(requirementFile, 'r', encoding='utf-8') as f:
        line = f.readline()
        while line:
            line = line.rstrip(" ")
            line = line.lstrip(" ")
            line = line.strip("\n")
            if line.startswith("#") == False and line != "":
            # pip vcs support
            # https://pip.pypa.io/en/stable/reference/pip_install/#vcs-support
                if line.startswith("http://"):
                    splits = line.split(" ")
                    extraRequirements.append(splits[0])          
                elif line.startswith("https://"):
                    splits = line.split(" ")
                    extraRequirements.append(splits[0])  
                elif line.startswith("git+"):    
                    splits = line.split(" ")
                    extraRequirements.append(splits[0])
                elif line.startswith("hg+"):    
                    splits = line.split(" ")
                    extraRequirements.append(splits[0])
                elif line.startswith("svn+"):    
                    splits = line.split(" ")
                    extraRequirements.append(splits[0])
                elif line.startswith("bzr+"):    
                    splits = line.split(" ")
                    extraRequirements.append(splits[0])
                else:
                    # To eliminate the comment.
                    if "#" in line:
                        splits = line.split('#')
                        line = splits[0]                    
                    line = line.lstrip(" ")
                    line = line.rstrip(" ")
                    requirements.append(line)           
            line = f.readline()
    return requirements, extraRequirements

import urllib.request as urllib_request

def setup(name,
          requirementsFile="REQUIREMENTS.txt",
          testsRequire="REQUIREMENTS_TEST.txt",  # list or filename
          extrasRequire=None,  # list or filename
          extraRequirements=None,
          **kws):
    """
    Setup package and install all requirements.
    Using setuptools.setup to setup the package, and use pip to install the 
    requirements.
    
    @param name Package name.
    @param requirementsFile The requirements.txt that include all the package 
                            requirements.
    @param testsRequire The package requirements that will be installed
                        when use "python setup.py test", this param should be
                        a file name or a list of requirements' names.
    @param extrasRequire A dictionary mapping names of “extras”
                        (optional features of your project) to
                        a file name or a list of requirements' names.
                        And the user can use "pip install [Packagename][extra]"
                        to install other distributions to support the features.
    @param pipOption Extra option while using pip install.
    @param extraRequirements Requirements that cannot install through pip 
                             install, e.g. single py file or special terminal 
                             command.
                             extraRequirements can be a single py file, 
                             a py file link or a terminal command.
                             extraRequirements must be a str or str list.
                             Usage of extraRequirements:
                             1) Install an exist py file:
                                extraRequirement="singletonmixin.py"
                             2) Install py file through py file link:
                                extraRequirement=
                                "http://www.garyrobinson.net/singletonmixin.py"
                             3) Install package with special terminal cmd:
                                extraRequirement=
                                "pip install --upgrade --trusted-host abc.org"
                             4) Install multiple extra requirements:
                                extraRequirement=[
                                    "singletonmixin.py",
                                    "http://www.garyrobinson.net/abc.py",
                                    "pip install --upgrade --trusted-host a.org"
                                ]
    @param **kws Keyword arguments of setuptools.setup, e.g. description,
                 version, license, author, author_email, url, etc.
    """   
    supportReinstall(name)
    
    import pip
    from setuptools import setup as setuptoolsSetup
    from setuptools import find_packages
    from pkg_resources import packaging
    
    
    _createManifestFile()         
    sitePackagesDir = getSitePackagesDir()   

    projectRootDir = os.path.abspath(os.path.curdir)
      
    packages = find_packages()
    requirements, _extraRequirements = getRequirements(requirementsFile)

    # testsRequire

    testsRequireList = []
    if testsRequire:
        if isinstance(testsRequire, str):
            testsRequireList, extras = getRequirements(testsRequire)
            if extras:
                raise NotImplementedError(
                    ('{} are not supported.\n'
                     'See more details: \n'
                     'https://code.nuwainfo.com/'
                     'trac/mercurius/ticket/2933#comment:25').format(extras))
        elif isinstance(testsRequire, list):
            testsRequireList = testsRequire
        else:
            raise TypeError('"testRequire" must be a string or list')

    # extrasRequire

    extrasRequireDict = {}
    if extrasRequire:
        if isinstance(extrasRequire, dict):
            for mode, extraRequire in extrasRequire.items():
                if isinstance(extraRequire, str):
                    extrasRequireDict[mode], extras = \
                        getRequirements(extraRequire)
                    if extras:
                        raise NotImplementedError(
                            ('{} are not supported.\n'
                             'See more details: \n'
                             'https://code.nuwainfo.com/'
                             'trac/mercurius/ticket/2933#comment:25'
                             ).format(extras))
                elif isinstance(extraRequire, list):
                    extrasRequireDict[mode] = extraRequire
                else:
                    raise TypeError('"extraRequire" must be a string or list')
        else:
            raise TypeError('"extrasRequire" must be a dict')

    kws.setdefault('name', name)
    kws.setdefault('zip_safe', False)
    kws.setdefault('include_package_data', True) 
    kws.setdefault('packages', packages)
    kws.setdefault('install_requires', requirements)
    kws.setdefault('tests_require', testsRequireList)
    kws.setdefault('extras_require', extrasRequireDict)
    
    def _getExtraRequirement(command):
        if command.endswith(".py"):
            splitedStr = command.split("/")
            for s in splitedStr:
                if ".py" in s:            
                    fileName = s
                    packageName = fileName.strip(".py")             
            if command.startswith("http"):        
                print("downloading %s..." % packageName)
                urllib_request.urlretrieve(command, fileName)
                print("installing %s..." % packageName)
                shutil.copy(fileName, sitePackagesDir)
                os.remove(fileName)
            else:
                print("installing %s..." % packageName)
                shutil.copy(command, sitePackagesDir)         
        else:       
            print("processing %s..." % command)
            os.system(command)
                     
    if 'install' in sys.argv or 'bdist_wheel' in sys.argv:
        for url in _extraRequirements:       
            os.system('pip install "{url}"'.format(url=url))
        if extraRequirements:
            if isinstance(extraRequirements, str):
                _getExtraRequirement(extraRequirements)
            elif isinstance(extraRequirements, list):
                for extra in extraRequirements:
                    _getExtraRequirement(extra)
            else:
                raise RuntimeError(
                    "extraRequirements argument must be " 
                    "str or list instance.")
                   
    setuptoolsSetup(**kws)    
    
    
    # patch
    if 'install' in sys.argv or 'bdist_wheel' in sys.argv:
        patchDir = os.path.join(projectRootDir, 'Patch')
        if os.path.exists(patchDir):    
            from Iuppiter import Patch
            Patch.patch(patchDir, strict=False)           
       
def _createManifestFile():
    """
    Create MANIFEST.in on the root of project.
    A MANIFEST.in file can define the list of files to include in the 
    distribution built by the sdist or bdist command.
    """
  
    projectRootDir = os.path.abspath(os.getcwd())   
    manifestFile = os.path.join(projectRootDir, 'MANIFEST.in')
    
    from setuptools import find_packages
    projectName = find_packages()[0]
    
    if os.path.exists(manifestFile):
        _file = open(manifestFile, 'r')
        if _file.readline() == u"# AUTOMATIC CREATED\n":
            _file.close()
            os.remove(manifestFile)
            with open(manifestFile, 'w+') as _file:
                _file.write(u"# AUTOMATIC CREATED\n")
                _file.write(u"graft Patch \n")
                _file.write(u"include *.txt \n")
                _file.write(u"recursive-include %s * \n" % projectName)
                _file.write(u"recursive-exclude %s *.pyc \n" % projectName)
        else:
            _file.close()
    else:
        with open(manifestFile, 'w+') as _file:
            _file.write(u"# AUTOMATIC CREATED\n")
            _file.write(u"graft Patch \n")
            _file.write(u"include *.txt\n")
            _file.write(u"recursive-include %s *\n" % projectName)
            _file.write(u"recursive-exclude %s *.pyc" % projectName)
        
def getScript(name):
    """
    Get the location of script execute file.
    
    @param name Script name. 
    @return script Return the Location of script if exist.
                   If the script not exist, return None.
    """
    
    pythonDir = os.path.dirname(sys.executable)
    script = os.path.join(pythonDir, name)
    if os.path.exists(script):
        return script
    else:
        if platform.system() == "Windows":
            script = os.path.join(pythonDir, "%s.exe" % name)
            if os.path.exists(script):
                return script
            else:
                scriptDir = os.path.join(pythonDir, "Scripts")
                script = os.path.join(scriptDir, "%s.exe" % name)
                if os.path.exists(script):
                    return script
        print("script %s not exist." % script)
        return None

# Generic cprint and colored functions.
try:
    import colorama
    import termcolor

    colorama.init()

    cprint = termcolor.cprint
    colored = termcolor.colored
except ImportError:
    def cprint(text, *args, **kws):
        print(text)

    colored = lambda text, *args, **kws: text
