#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Function.py 10932 2018-03-26 07:56:30Z Kevin $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Kevin $ (Last)
# $Date: 2018-03-26 15:56:30 +0800 (Mon, 26 Mar 2018) $
# $Revision: 10932 $

"""
Function utility functions definition.
"""

import types
import inspect
import six

def getInfo(func):
    """
    Get function information as a dictionary.
    Source: decorator 2.3.2 module.
            http://www.phyast.pitt.edu/~micheles/python/documentation.html

    @return An info dictionary containing:
        - name (the name of the function : str)
        - argnames (the names of the arguments : list)
        - defaults (the values of the default arguments : tuple)
        - signature (the signature : str)
        - doc (the docstring : str)
        - module (the module name : str)
        - dict (the function __dict__ : str)

    >>> def f(self, x=1, y=2, *args, **kw): pass

    >>> info = getinfo(f)

    >>> info["name"]
    'f'
    >>> info["argnames"]
    ['self', 'x', 'y', 'args', 'kw']

    >>> info["defaults"]
    (1, 2)

    >>> info["signature"]
    'self, x, y, *args, **kw'
    """
    assert inspect.ismethod(func) or inspect.isfunction(func)
    regargs, varargs, varkwargs, defaults = inspect.getargspec(func)
    argnames = list(regargs)
    if varargs:
        argnames.append(varargs)
    if varkwargs:
        argnames.append(varkwargs)
    signature = inspect.formatargspec(regargs, varargs, varkwargs, defaults,
                                      formatvalue=lambda value: "")[1:-1]
    return dict(name=func.__name__, argnames=argnames, signature=signature,
                defaults=func.__defaults__, doc=func.__doc__,
                module=func.__module__, dict=func.__dict__,
                globals=func.__globals__, closure=func.__closure__)

from decorator import decorator

def enhance(callable, wrapper):
    """
    Enhance a function or method by given wrapper and keep its signature.
    This is a kind of function (callable objects) mixin.

    @param callable Function or method.
    @param wrapper Wrapper function (first argument must be callable).
    @return Enhanced callable.
    """
    # Decorator module doesn't support functor... :(
    if not inspect.isroutine(wrapper) and hasattr(wrapper, '__call__'):
        def _functor(method, *args, **kws):
            return wrapper(method, *args, **kws)
        return enhance(callable, _functor)

    c2 = callable
    isMethod = False
    if inspect.ismethod(c2):
        c2 = callable.__func__
        isMethod = True

    dec = decorator(wrapper)
    newCallable = dec(c2)

    if isMethod:
        if six.PY2:
            newCallable = types.MethodType(
                newCallable, callable.im_self, callable.im_class)
        else:
            newCallable = types.MethodType(newCallable, callable.__self__)
        return newCallable
    else:
        return newCallable