#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: geni18nstatic.py 11620 2019-06-09 15:41:21Z Trista $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Trista $ (last)
# $Date: 2019-06-09 23:41:21 +0800 (Sun, 09 Jun 2019) $
# $Revision: 11620 $



from django.core.management.base import BaseCommand, CommandError
from optparse import make_option

class Command(BaseCommand):

    help = "Generate static files with i18n."

    def add_arguments(self, parser):

        parser.add_argument('-a', '--app-name',  type=str, help="App name",
                            dest="app", default="")

        parser.add_argument('--catalog', dest='catalog', default='django',
            action='store',
            type=str,
            help='Catalog name.')


    # Validation is called explicitly each time the server is reloaded.
    requires_model_validation = False

    def handle(self, app='', *args, **options):
        if args:
            raise CommandError('Usage is geni18nstatic %s' % self.args)

        catalog = options.get('catalog')

        import os
        from django.utils.translation.trans_real import to_locale
        from django.conf import settings

        curdir = os.path.abspath(os.path.curdir)

        info = []
        apps = settings.INSTALLED_APPS
        if app:
            if app in apps:
                apps = [app]
            else:
                raise CommandError('Invalid app name: %s' % app)

        for app in apps:
            path = app.replace('.', os.sep)
            info.append((app, path))
            # We recursively search for locale to append it to ready for
            # process info list.
            for root, dirs, files in os.walk(path):
                if 'locale' in dirs:
                    info.append((app, root))
                if '.svn' in dirs:
                    dirs.remove('.svn')  # Don't visit .svn directories.

        if os.path.exists(os.path.join(curdir, 'locale')):
            info.append(('.', curdir))

        import polib
        import re

        jsRE = re.compile('(.*/)?static/.*\.js')
        gettextRE = re.compile("(n?gettext)(\s*\(.*?\))", re.DOTALL)

        from django.views import i18n
        import hashlib

        for lang in settings.LANGUAGES:
            entries = {}
            for app, path in info:
                if os.path.exists(os.path.join(path, "locale")):
                    locale = to_locale(lang[0])
                    p = os.path.join(path, "locale", locale, "LC_MESSAGES",
                                     '%s.po' % catalog)

                    if not os.path.exists(p):
                        print(('WARNNING: %s does not exist, '
                               'please do i18n command first.') % p)
                        continue

                    po = polib.pofile(p)
                    for entry in po:
                        for file, line in entry.occurrences:
                            entries.setdefault(file, {
                                'appPath': path, 
                                'elist': []
                            })['elist'].append((entry, po))

            from Iuppiter import Util
            import io as StringIO
            
            from Iuppiter.DjangoUtil import DJANGO_VERSION
                
                
            if DJANGO_VERSION >= 11100:
                from django.utils.html import escapejs as q
            if DJANGO_VERSION < 11100:
                from django.utils.text import javascript_quote as q

            for file, finfo in list(entries.items()):
                # javascript
                appPath = finfo['appPath']
                elist = finfo['elist']
                           
                if jsRE.match(file):
                    plural = None
                    path = Util.path(file)
                    if not os.path.exists(path):
                        path = os.path.normpath(os.path.join(appPath, file))
                    with open(path, "rb") as f:
                        c = f.read()
                    key = hashlib.md5(file).hexdigest()
                    js = StringIO.StringIO()
                    msgs = StringIO.StringIO()
                    msgs.write("var catalog_%s = {" % key)
                    msgsv = []
                    for i, (entry, po) in enumerate(elist):
                        if 'Plural-Forms' in po.metadata and not plural:
                            pluralForm = po.metadata['Plural-Forms']
                            plural = ([el.strip()
                                       for el in pluralForm.split(';')
                                       if el.strip().startswith('plural=')][0].
                                       split('=',1)[1])

                        if entry.msgid_plural and entry.msgstr_plural:
                            n = len(entry.msgstr_plural)
                            v = ["'%s'" % q(entry.msgstr_plural[str(i)])
                                 for i in range(n)]

                            msgsv.append("'%s': [%s]" %
                                         (entry.msgid, ','.join(v)))

                        elif entry.msgid and entry.msgstr:
                                v = q(entry.msgstr)
                                msgsv.append("'%s': '%s'" %
                                    (q(entry.msgid),
                                     v if v else q(entry.msgid)))

                    msgs.write(','.join(msgsv))
                    msgs.write("};\n")

                    if plural:
                        js.write((
                            "function pluralidx_%s(n) { "
                                "return pluralidx(n, '%s'); "
                            "}") % (key, plural))
                    else:
                        js.write((
                            "function pluralidx_%s(count) { "
                                "return (count == 1) ? 0 : 1; "
                            "}") % key)

                    js.write("\n")
                    js.write(msgs.getvalue())
                    msgs.close()

                    # gettext, ngettext.
                    js.write((
                        "function gettext_%(key)s(msgid) { "
                            "return gettext(catalog_%(key)s, msgid); "
                        "}") % {'key': key})
                    js.write("\n")

                    js.write((
                        "function ngettext_%(key)s(singular, plural, count) { "
                            "return ngettext(catalog_%(key)s, "
                                "pluralidx_%(key)s, singular, plural, count); "
                        "}") % {'key': key})
                    js.write("\n")

                    # Write original javascript content.
                    js.write(gettextRE.sub(r"\1_%s\2" % key, c))

                    name = "%s_%s.js" % (''.join(path.split('.')[:-1]),
                                         locale.lower().replace('_', '-'))
                    print('Writing: %s ...' % name)
                    with open(Util.path(name), "wb") as f:
                        f.write(js.getvalue())

                    js.close()
