#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 12332 2020-06-09 12:15:58Z Lavender $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Lavender $ (last)
# $Date: 2020-06-09 20:15:58 +0800 (Tue, 09 Jun 2020) $
# $Revision: 12332 $
"""
Test case utilities.
"""

import os
import time
import requests

# We can't directly import Iuppiter.Configure, because that will cause error
# when we run nose test (ImportError).
try:
    import settings
except ImportError:
    try:
        # Run from RunTests.
        from Iuppiter.Configure import settings
    except ImportError:
        # Normal Iuppiter test case.
        from Iuppiter import DefaultSettings as settings

rootDir = os.path.dirname(settings.__file__)
originalDir = os.path.abspath((os.path.curdir))

def cycleGet(url, waitSeconds=3, waitTimes=10):
    tryTime = 0
    while tryTime < waitTimes: # 測 waitTimes 次
        try:
            res = requests.get(url)
            if res.status_code == 200:
                break
        except Exception as e:
            pass
        tryTime += 1
        time.sleep(waitSeconds)
    return res

def chdirToRootDir():
    """
    Change current directory to project's root directory.
    """
    print(('chdir ', rootDir))
    os.chdir(rootDir)

# email receive test
from imaplib import IMAP4, IMAP4_SSL
from email.utils import parseaddr, parsedate_tz
from datetime import datetime, timedelta
import email

def _parseEmail(messageObj):
    """
    Parse email message as a dictionary.
    Reference: http://fanli7.net/a/bianchengyuyan/Python/20140122/464043.html
    
    @param messageObj An email message object.
                      String and file can be transformed into message object 
                      with the following API: 
                      email.message_from_string(string)
                      email.message_from_file(file)                      
                      See more details in the doc:
                      https://docs.python.org/2/library/email.parser.html
    @return An email information dictionary containing:
            - subject (The subject of the email : str)
            - sender (The sender of the email : str)
            - receiver (The receiver of the email : str)
            - cc (cc of the email : str)
            - sentTime (The sending time of email in UTC : datetime obj)
            - content (The body content of the email : str)
            - messageObj (The messageObj of the email : obj)
    """
       
    subject = messageObj.get('subject')   
    header = email.Header.Header(subject)
      
    sender = parseaddr(messageObj.get('from'))[1]
    receiver = parseaddr(messageObj.get('to'))[1]
    cc = parseaddr(messageObj.get_all('cc'))[1]
    sentTimeTuple = parsedate_tz(messageObj.get('date'))
    sentTime = datetime(*sentTimeTuple[0:6])
    
    content = None
    for body in messageObj.walk():
        if not body.is_multipart(): 
            content = body.get_payload(decode=True)
      
    return {"subject": subject,
            "sender": sender,
            "receiver": receiver,
            "cc": cc,
            "sentTime": sentTime,
            "content": content,
            "messageObj": messageObj}

def getEmails(host, userName, password, port=993, useSSL=True, 
              inbox='INBOX', searchCmd='ALL'):
    """
    Get emails information via imap.
    
    @param host Email host server.
    @param userName Email username.    
    @param password Email password.
    @param port Email connect port, default to 993.
    @param useSSL Default to True.
                  If you don't want to use ssl to connect email,
                  set useSSL to False.
    @param inbox The mailbox you are going to select.
                 The default inbox is 'INBOX'.
    @param searchCmd The email searching command.
                     Notice: the command should follow the format:
                     '(keyword "argument")'
                     e.g. '(SINCE "01-Jan-2012")'
                          '(BEFORE "01-Jan-2012")'
                          '(FROM "sender")'
                          See more searching keyword on the doc:
                          http://tools.ietf.org/html/rfc3501.html#page-103
                     The default searching command is 'ALL'.
    @return (connStatus, emails) A tuple include connect status and emails.
            connStatus The status of email connecting.
                       The status can be "OK", "NO", or "BAD".
            emails An email information dictionary containing:
                  - status (The status of the email : str)
                  - subject (The subject of the email : str)
                            (The status can be "OK", "NO", or "BAD".)
                  - sender (The sender of the email : str)
                  - receiver (The receiver of the email : str)
                  - cc (cc of the email : str)
                  - sentTime (The sending time of email in UTC : datetime obj)
                  - content (The body content of the email : str)
                  - messageObj (An email message object : obj)
                     (see more on 
                      https://docs.python.org/2/library/email.parser.html)
    """
    emails = []
    try:
        # login
        mail = (IMAP4_SSL if useSSL else IMAP4)(host, port)    
        mail.login(userName, password)
        
        # get emails
        mail.select(inbox)
        connStatus, mailList = mail.search(None, searchCmd)
        if not mailList[0] == '':        
            mailList = mailList[0].split(" ")
            for m in mailList:
                fetchStatus, fetchData = mail.fetch(m, "(RFC822)")
                messageObj = email.message_from_string(fetchData[0][1])
                
                mailInfo = _parseEmail(messageObj)
                mailInfo.setdefault('status', fetchStatus)        
                emails.append(mailInfo)
        else:
            connStatus = "MAIL NOT FOUND"
    finally:
        mail.close()
        mail.logout()    
    return connStatus, emails

import time

def testEmailRecieve(host, userName, password, sender, port=993, useSSL=True, 
                     inbox='INBOX', extraSearchCmd='', 
                     sentMinute=5, maxWait=10, poolingInterval=30,
                     subject=None, receiver=None, cc=None, content=None, 
                     testCase=None):
    """
    Test for email receive.
    
    The function will search for mails which receive time between now and 
    now minus sentMinute. sentMinute is default to 5 minutes.
    
    If you set the testCase, this function will do some basic assertion, 
    including email receive status, email status and email sender.
    The assertion of subject, receiver, cc and content will be execute after 
    you set the value of them.
                    
    @param host Email host server.
    @param userName Email username.    
    @param password Email password.
    @param password Email sender.
    @param port Email connect port, default to 993.
    @param useSSL Default to True.
                  If you don't want to use ssl to connect email,
                  set useSSL to False.
    @param inbox The mailbox you are going to select.
                 The default inbox is 'INBOX'.
    @param extraSearchCmd The email extra searching command.
                          The base searching command is '(FROM "sender")'. 
                          The extraSearchCmd will add after the base command.
                          Notice: the command should follow the format:
                          '(keyword "argument")'
                          e.g. '(SINCE "01-Jan-2012")'
    @param sentMinute Email receive time for searching.
                      This function will search for emails which receive time 
                      between now and now minus sentMinute.
    @param maxWait Max waiting time for searching emails. (unit: minutes)
    @param poolingInterval Pooling interval of searching emails. (unit: seconds)
    @param subject The subject of the email going to assert.
    @param receiver The receiver of the email going to assert.
    @param cc Email Cc of the email going to assert.
    @param content The body content of the email going to assert.
    @param testCase The test case of unittest.
    @return (status, emails) A tuple include status and emails.
            status The status of email recieve.
                   The status can be "OK", "MAIL NOT FOUND", "NO", or "BAD".
            emails A list of email information dictionary.
                   An email information dictionary containing:
                   - status (The status of the email : str)
                            (The status can be "OK", "NO", or "BAD".)
                   - subject (The subject of the email : str)
                   - sender (The sender of the email : str)
                   - receiver (The receiver of the email : str)
                   - cc (cc of the email : str)
                   - sentTime (The sending time of email in UTC : datetime obj)
                   - content (The body content of the email : str)
                   - messageObj (An email message object : obj)
                     (see more on 
                      https://docs.python.org/2/library/email.parser.html)
                      
    @attention If you use gmail for email receive testing, please ensure you 
               have enable the security authority and IMAP.               
               gmail security authority setting:
               https://www.google.com/settings/security/lesssecureapps               
               gmail IMAP setting:
               https://support.google.com/mail/
               troubleshooter/1668960?hl=zh-Hant#ts=1665018
    """
    
    emails = []
    searchCmd = '(FROM "{sender}")'.format(sender=sender) + extraSearchCmd    
    now = datetime.utcnow()    
    compareTime = now - timedelta(minutes=sentMinute)
    maxTime = now + timedelta(minutes=maxWait)
    
    while now < maxTime and not emails:       
        status, _emails = getEmails(host, userName, password, port, 
                                    useSSL, inbox, searchCmd)
               
        # check sent time    
        if _emails:
            for mail in _emails:
                sentTime = mail.get('sentTime')
                if sentTime > compareTime:
                    emails.append(mail)
            if not emails:
                status = "MAIL NOT FOUND"
        
        time.sleep(poolingInterval)
        now = datetime.utcnow()
    
    if testCase:
        testCase.assertEqual(status, "OK")
        for e in emails:
            testCase.assertEqual(e['status'], "OK")
            testCase.assertEqual(e['sender'], sender)
            if subject:
                testCase.assertEqual(e['subject'], subject)
            if receiver:
                testCase.assertEqual(e['receiver'], receiver)
            if cc:
                testCase.assertEqual(e['cc'], cc)
            if content:
                testCase.assertTrue(content in e['content'])
    return status, emails
