#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: JsonTest.py 7437 2012-04-23 06:31:53Z Yachu $
#
# Copyright (c) 2009 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Yachu $
# $Date: 2012-04-23 14:31:53 +0800 (週一, 23 四月 2012) $
# $Revision: 7437 $

import six
import sys

import test as util

util.chdirToRootDir()

import unittest

from django.test import TransactionTestCase

from Iuppiter import Json

class A:
    def __init__(self):
        self.a = 1
        self.b = "xx"
        self.c = [1, 2, 3]

    def void(self):
        pass

    def __eq__(self, other):
        if self.a == other.a and self.b == other.b and self.c == other.c:
            return True
        else:
            return False

class B(A):
    def __init__(self):
        self.d = A()

    def __eq__(self, other):
        if self.d == other.d:
            return True
        else:
            return False

class C(object):
    c = 3

    def __init__(self):
        self.a = 1
        self.b = 2

    def __eq__(self, other):
        if self.a == other.a and self.b == other.b and self.c == other.c:
            return True
        else:
            return False

class JsonTest(TransactionTestCase):
    """
    Test case for Json utilities.
    """

    def normalize(self, s):
        """
        Replace result string to make different test environment no difference.

        @param s Json string.
        @return Normalized string.
        """
        return s.replace('null', 'None').replace('Iuppiter.test.', 'test.')

    def testSerializer1(self):
        """
        Test for serializer case 1.
        """
        i = 1
        j = 2.01
        k = "string"
        if six.PY2:
            l = long(99999999)
        else:
            l = 99999999
        m = [1, 2, 3]
        n = (1, 2, 3)
        o = {'w': 1, 'x': 2, 'y': 3, 'z': 4}

        import json
        s = Json.Serializer()
        self.assertEqual(s.serialize(i), json.dumps(i))
        self.assertEqual(s.serialize(j), json.dumps(j))
        self.assertEqual(s.serialize(l), json.dumps(l))
        self.assertEqual(s.serialize(m), json.dumps(m))
        self.assertEqual(s.serialize(n), json.dumps(n))
        self.assertEqual(s.serialize(o), json.dumps(o))

        self.assertEqual(s.deserialize(s.serialize(i)), i)
        self.assertEqual(s.deserialize(s.serialize(j)), j)
        self.assertEqual(s.deserialize(s.serialize(k)), k)
        self.assertEqual(s.deserialize(s.serialize(l)), l)
        self.assertEqual(s.deserialize(s.serialize(m)), m)
        # Tuple will be serialized as a list.
        self.assertEqual(s.deserialize(s.serialize(n)), [1, 2, 3])
        self.assertEqual(s.deserialize(s.serialize(o)), o)

    def testSerializer2(self):
        """
        Test for serializer case 2.
        """
        from datetime import datetime
        now = datetime.now()

        s = Json.Serializer()
        t = (now.year, now.month, now.day, now.hour, now.minute, now.second,
             now.microsecond)
        self.assertEqual(eval(s.serialize(now)), eval(
            ('{' +
             '"obj": "new Date(Date.UTC(%d,%d,%d,%d,%d,%d,%d))", ' +
             '"__pythonclass__": "datetime.datetime", ' +
             '"year": %d, ' +
             '"month": %d, ' +
             '"day": %d, ' +
             '"hour": %d, ' +
             '"minute": %d, ' +
             '"second": %d, ' +
             '"microsecond": %d' +
            '}') % (t + t)))

        obj = s.deserialize(s.serialize(now))
        self.assertEqual(obj, now)

    def testSerializer3(self):
        """
        Test for serializer case 3.
        """
        s = Json.Serializer()
        b = B()
        bjson = s.serialize(b)

        self.assertEqual(eval(self.normalize(bjson)),
            eval(self.normalize(('{"__pythonclass__": "Iuppiter.tests.JsonTest.B", ' +
            '"d": {' +
                   '"a": 1, ' +
                   '"b": "xx", ' +
                   '"c": [1, 2, 3], ' +
                   '"__pythonclass__": "Iuppiter.tests.JsonTest.A", ' +
                 '}' +
            '}')))
        )

        b2 = s.deserialize(bjson)
        self.assertEqual(b2.d.a, 1)
        self.assertEqual(b2.d.b, "xx")
        self.assertEqual(b2.d.c, [1, 2, 3])
        self.assertEqual(b2, b)

        a = A()
        ajson = s.serialize(a)
        list = [1, 2, b, 3, [4, 5, a], (6, 7, {'8':b}, 9)]
        # Let tuple to be list.
        list2 = [1, 2, b, 3, [4, 5, a], [6, 7, {'8':b}, 9]]
        json = s.serialize(list)

        self.assertEqual(json,
            '[1, 2, ' + bjson +
            ', 3, [4, 5, ' + ajson +
            '], [6, 7, {"8": ' + bjson +
            '}, 9]]'
        )

        list3 = s.deserialize(json)
        self.assertEqual(list2, list3)

    def testSerializer4(self):
        """
        Test for serializer case 4.
        """
        def aToObjDict(obj):
            dic = {}
            dic['a'] = obj.a
            dic['b'] = obj.b
            dic['c'] = obj.c
            dic['z'] = str(obj.a) + obj.b
            return dic

        def objDictToA(dic):
            a = A()
            a.a = dic['a']
            a.b = dic['b']
            a.c = dic['c']
            return a

        s = Json.Serializer()
        s.register(A, aToObjDict, objDictToA)

        a = A()
        ajson = s.serialize(a)

        self.assertEqual(eval(self.normalize(ajson)),
            eval(self.normalize(
                '{"a": 1, "c": [1, 2, 3], "b": "xx", "z": "1xx", ' +
                '"__pythonclass__": "Iuppiter.tests.JsonTest.A"}')))

        a2 = s.deserialize(ajson)
        self.assertEqual(a2, a)

        b = B()
        bjson = s.serialize(b)
        self.assertEqual(eval(self.normalize(bjson)),
            eval(self.normalize(('{"__pythonclass__": "Iuppiter.tests.JsonTest.B", ' +
                                 '"d": ' + ajson +
                                 '}'))))

        b2 = s.deserialize(bjson)
        self.assertEqual(b2, b)

    def testSerializer5(self):
        """
        Test for serializer case 5.
        """
        s = Json.Serializer()
        c = C()
        cjson = s.serialize(c)

        c2 = s.deserialize(cjson)
        self.assertEqual(c2.a, 1)
        self.assertEqual(c2.b, 2)
        self.assertEqual(c2.c, 3)
        self.assertEqual(c2, c)

    def testUnicode(self):
        """
        Test for serializer for unicode string.
        """
        s = Json.Serializer()
        u = u"unicode string"

        j = s.serialize(u)
        self.assertEqual(j, '"unicode string"')

        du = s.deserialize(j)
        self.assertEqual(du, u)

    def testClass(self):
        """
        Test for #651.
        """
        s = Json.Serializer()
        self.assertEqual("{}", s.serialize(C))

    def testRegex(self):
        """
        Test for serialize regular expression object.
        """
        import re

        # re.compile use unicode string
        # see https://code.nuwainfo.com/trac/mercurius/ticket/3075#comment:16
        regex = re.compile(u'[-+]?[0-9]+\.?[0-9]*')
        s = Json.Serializer()
        json = s.serialize(regex)
        
        if sys.version_info.major >= 3 and sys.version_info.minor >= 7:
            self.assertEqual(eval(json.replace('null', 'None')),
                eval(('{"__pythonclass__": "re.Pattern", ' +
                '"pattern": "[-+]?[0-9]+\.?[0-9]*"' +
                '}').replace('null', 'None')))
        else:
            self.assertEqual(eval(json.replace('null', 'None')),
                eval(('{"__pythonclass__": "_sre.SRE_Pattern", ' +
                '"pattern": "[-+]?[0-9]+\.?[0-9]*"' +
                '}').replace('null', 'None')))

        regex2 = s.deserialize(json)
        self.assertEqual(regex2, regex)
        
    def testSortedDict(self):
        """
        Test for serialize SortedDict object.
        """
        import django

        import django

        if django.__version__ != '1.8':
            return

        from django.utils.datastructures import SortedDict

        d = SortedDict()
        d['a'] = 1
        d['b'] = 2
        d['aa'] = 3
        d['bb'] = 4
        d['nested'] = SortedDict({'z': 1, 'w': 2, 'x': 3})

        s = Json.serializer
        json = s.serialize(d)
        dd = s.deserialize(json)

        k = list(d.keys())
        v = list(d.values())

        kk = list(dd.keys())
        vv = list(dd.values())

        self.assertEqual(kk, k)
        self.assertEqual(vv, v)

if __name__ == '__main__':
    unittest.main()
