#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: models.py 11172 2018-07-16 09:31:00Z Lavender $
#
# Copyright (c) 2018 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $ (last)
# $Date: 2018-07-16 18:31:00 +0900 (週一, 16 七月 2018) $
# $Revision: 11172 $

import os
import hashlib

from django.db import models
from django.conf import settings
from django.utils.translation import ugettext_lazy as _
from django.db.models import Max

from mptt.models import MPTTModel, TreeForeignKey

from Iuno.template_editor import (
    BACKUP_DIR_NAME, BACKUP_FILE_DIR, STATIC_DIR, TEMPLATE_DIR)

FOLDER = 1
FILE = 2

TYPE_CHOICES = (
    (FOLDER, 'File'),
    (FILE, 'Folder'),
)

class Version(models.Model):
    dirName = models.CharField(max_length=32) # md5(path)
    fileName = models.CharField(max_length=30)

    number = models.PositiveIntegerField() 
    checksum = models.CharField(max_length=32) # MD5(file data)

    conflict = models.BooleanField(default=False)

    preVersion = models.ForeignKey(
        'self',
        related_name='previous',
        on_delete=models.CASCADE,
        null=True,
        default=None,
    )
    nextVersion = models.ForeignKey(
        'self',
        related_name='next',
        on_delete=models.CASCADE,
        null=True,
        default=None,
    )

    @property
    def backupPath(self):
        path = os.path.join(
            BACKUP_DIR_NAME, self.dirName, self.fileName)
        return path

    @property
    def absoluteBackupPath(self):
        path = self.backupPath
        # FIXME: settings.TEMPLATES[0]['DIRS'][0] maybe undefined
        fullPath = os.path.join(STATIC_DIR, path)
        return fullPath

class VersionControlMixin(object):
    def getVersion(self, versionNumber=None):
        if self.type == FILE:
            path = self.path
            dirName = hashlib.md5(path).hexdigest()

            if Version.objects.filter(dirName=dirName).exists():
                if versionNumber:
                    return Version.objects.get(
                                dirName=dirName, number=versionNumber)
                else:
                    maxVersion = Version.objects.filter(
                        dirName=dirName).aggregate(Max('number'))['number__max']
                    return Version.objects.get(
                                dirName=dirName, number=maxVersion)
            else:
                return None
        else:
            return None

    @property
    def version(self):
        return self.getVersion()

class Static(MPTTModel, VersionControlMixin):
    type = models.IntegerField(choices=TYPE_CHOICES)
    name = models.CharField(max_length=50)
    parent = TreeForeignKey(
        'self', 
        on_delete=models.CASCADE, 
        null=True, 
        blank=True, 
        related_name='children')

    class MPTTMeta:
        order_insertion_by = ['name']

    @property
    def path(self):
        path = ''

        ancestors = self.get_ancestors()
        for ancestor in ancestors:
            path = os.path.join(path, ancestor.name)
        path = os.path.join(path, self.name)

        return path

    @property
    def absolutePath(self):
        path = self.path
        
        # FIXME: deploy maybe unwork. 
        fullPath = os.path.join(STATIC_DIR, path)

        return fullPath

    #FIXME: 最好資料夾回傳 None，沒附檔名的檔案回傳 ''
    @property
    def subFileName(self):
        if self.type == FOLDER:
            return 'folder'
        subFileName = self.name.split('.')[-1].lower()
        if subFileName == self.name.lower():
            return None
        return subFileName

    def __str__(self):
        return "%s" % (self.name,)

class Template(MPTTModel, VersionControlMixin):
    type = models.IntegerField(choices=TYPE_CHOICES)
    name = models.CharField(max_length=50)
    parent = TreeForeignKey(
        'self', 
        on_delete=models.CASCADE, 
        null=True, 
        blank=True, 
        related_name='children')

    class MPTTMeta:
        order_insertion_by = ['name']

    @property
    def path(self):
        path = ''

        ancestors = self.get_ancestors()
        for ancestor in ancestors:
            path = os.path.join(path, ancestor.name)
        path = os.path.join(path, self.name)

        return path

    @property
    def absolutePath(self):
        path = self.path
        
        # FIXME: settings.TEMPLATES[0]['DIRS'][0] maybe undefined
        fullPath = os.path.join(TEMPLATE_DIR, path)
        
        return fullPath

    @property
    def subFileName(self):
        if self.type == FOLDER:
            return 'folder'
        subFileName = self.name.split('.')[-1].lower()
        if subFileName == self.name.lower():
            return None
        return subFileName

    def __str__(self):
        return "%s" % (self.name,)

    
    