#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 11977 2020-03-16 18:37:48Z Lavender $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $ (last)
# $Date: 2020-03-17 02:37:48 +0800 (週二, 17 三月 2020) $
# $Revision: 11977 $

import os
import sys
import warnings
import logging

# from django.utils.translation import ugettext_lazy as _

# django.core.exceptions.AppRegistryNotReady: The translation infrastructure 
# cannot be initialized before the apps registry is ready. 
# Check that you don't make non-lazy gettext calls at import time.

from Iuppiter.Util import extendUnique
from Iuppiter.DjangoUtil import (
    extendInstalledApps, DJANGO_VERSION, extendMiddlewareClasses)

from .Version import ZEPHYRUS_VERSION

def createLogger(name, fileHandlerName):
    logger = logging.getLogger(name)
    logger.setLevel(logging.DEBUG)
    handler = logging.StreamHandler()
    handler.setFormatter(
        logging.Formatter("%(asctime)s - %(levelname)s - %(message)s"))
    logger.addHandler(handler)
    fileHandler = logging.FileHandler(fileHandlerName)
    fileHandler.setLevel(logging.WARNING)
    logger.addHandler(fileHandler)
    return logger

def update(settingsDict, defaults, config=None):
    """
    Update settings by defaults if the key is not in settings.
    
    @param settingsDict Settings dict.
    @param defaults Defaults dict to provide default value.
    @param config Reserved.
    """
    for k, v in defaults.items():
        if k not in settingsDict:
            settingsDict[k] = v

def warnWithout(k, settingsDict, defaults):
    """
    Prompt warning message for without certain key in settings
    while doing attachSettings.
    
    @param k Settings key.
    @param settingsDict Settings dict.
    @param defaults Defaults dict to provide default value.
    """
    if k not in settingsDict:
        warnings.warn("'%s' %s %s" % (
            colored(k, 'yellow', attrs=['bold']),
            colored(" not set in settings.py, it will be set as ", 'white',
                    attrs=['bold']),
            colored(defaults[k], 'cyan', attrs=['bold'])))

def attachSettings(settingsDict):
    if 'ZEPHYRUS_BLOG_ENABLE' in settingsDict:
        ZEPHYRUS_BLOG_ENABLE = settingsDict['ZEPHYRUS_BLOG_ENABLE']
    else:
        ZEPHYRUS_BLOG_ENABLE = False


    if 'ZEPHYRUS_OSCAR_ENABLE' in settingsDict:
        ZEPHYRUS_OSCAR_ENABLE = settingsDict['ZEPHYRUS_OSCAR_ENABLE']
    else:
        ZEPHYRUS_OSCAR_ENABLE = False

    #=========================================================================
    # Django plugins.
    ADDING_INSTALLED_APPS = [
        'djangocms_history',
        'djangocms_forms',
        'djangocms_file',
        'djangocms_picture',
        'Zephyrus.boilerplate',
        'Zephyrus.landing',
        'Zephyrus.facebook',
        'solo',
        'Zephyrus.instagram',
        'Zephyrus.line',
        'Zephyrus.comment',
        'Zephyrus.statistics',
        'drip',
        'Zephyrus.autoresponder',
        'robots',
        'django_gravatar',
        'captcha',
        'Zephyrus.map',
    ]
    
    if ZEPHYRUS_BLOG_ENABLE:
        ADDING_INSTALLED_APPS += [
            'aldryn_apphooks_config',
            'aldryn_categories',
            'aldryn_common',
            'aldryn_newsblog',
            'aldryn_people',
            'aldryn_translation_tools',
            'parler',
            'sortedm2m',
            'taggit',
            'haystack',
            
            'django.contrib.redirects',
            'Zephyrus.blog_extension',
        ]

    if ZEPHYRUS_OSCAR_ENABLE:
        ADDING_INSTALLED_APPS += [
            'Zephyrus.cms_oscar',
        ]
    
    installed = settingsDict['INSTALLED_APPS']
    keys = [k for k in installed if k.startswith('djangocms_')]
    extendInstalledApps(settingsDict, ADDING_INSTALLED_APPS,
                        key=list(keys)[-1])
    
    #=========================================================================
    # Middleware
    middlewares = [
        'Zephyrus.statistics.middleware.StatisticsMiddleware',
    ]
   
    
    extendMiddlewareClasses(settingsDict, middlewares)
    
    #=========================================================================
    # Django CMS Ckeditor.
    DEFAULT_CKEDITOR_SETTINGS = {
        'CKEDITOR_SETTINGS': {
            'language': '{{ language }}',
            'toolbar_CMS': [
                ['cmsplugins',],
                
                ['Source', '-', 'Save', 'NewPage', 'DocProps', 'Preview', 
                 'Print', '-', ' Templates'],
                ['Cut', 'Copy', 'Paste', 'PasteText', 'PasteFromWord', '-', 
                 'Undo', 'Redo'],
                ['Find', 'Replace', '-', 'SelectAll', '-', 'SpellChecker', 
                 'Scayt'],
                ['Form', 'Checkbox', 'Radio', 'TextField', 'Textarea', 'Select', 
                'Button', 'ImageButton', 'HiddenField'],
                ['Bold', 'Italic', 'Underline', 'Strike', 'Subscript', 
                'Superscript', '-', 'RemoveFormat'],
                ['NumberedList', 'BulletedList', '-', 'Outdent', 'Indent', '-', 
                'Blockquote', 'CreateDiv', '-', 'JustifyLeft', 'JustifyCenter', 
                'JustifyRight', 'JustifyBlock', '-', 'BidiLtr', 'BidiRtl'],
                ['Link', 'Unlink', 'Anchor'],
                ['Image', 'Flash', 'Table', 'HorizontalRule', 'Smiley', 
                'SpecialChar', 'PageBreak', 'Iframe'],
                ['Styles', 'Format', 'Font', 'FontSize'],
                ['TextColor', 'BGColor'],
                ['Maximize', 'ShowBlocks', '-', 'About'],
            ],
            'toolbar_HTMLField': [         
                ['Source', '-', 'Save', 'NewPage', 'DocProps', 'Preview', 
                 'Print', '-', ' Templates'],
                ['Cut', 'Copy', 'Paste', 'PasteText', 'PasteFromWord', '-', 
                 'Undo', 'Redo'],
                ['Find', 'Replace', '-', 'SelectAll', '-', 'SpellChecker', 
                 'Scayt'],
                ['Form', 'Checkbox', 'Radio', 'TextField', 'Textarea', 'Select', 
                'Button', 'ImageButton', 'HiddenField'],
                ['Bold', 'Italic', 'Underline', 'Strike', 'Subscript', 
                'Superscript', '-', 'RemoveFormat'],
                ['NumberedList', 'BulletedList', '-', 'Outdent', 'Indent', '-', 
                'Blockquote', 'CreateDiv', '-', 'JustifyLeft', 'JustifyCenter', 
                'JustifyRight', 'JustifyBlock', '-', 'BidiLtr', 'BidiRtl'],
                ['Link', 'Unlink', 'Anchor'],
                ['Image', 'Flash', 'Table', 'HorizontalRule', 'Smiley', 
                'SpecialChar', 'PageBreak', 'Iframe'],
                ['Styles', 'Format', 'Font', 'FontSize'],
                ['TextColor', 'BGColor'],
                ['Maximize', 'ShowBlocks', '-', 'About'],
            ],
            'skin': 'moono-lisa',
        },
        'TEXT_ADDITIONAL_TAGS': (
            'iframe',
        ),
        'TEXT_ADDITIONAL_ATTRIBUTES': (
            'scrolling', 'allowfullscreen', 'frameborder'
        ),
    }
    update(settingsDict, DEFAULT_CKEDITOR_SETTINGS)

    #=========================================================================
    # Django CMS Snippe.
    DEFAULT_SNIPPE_SETTINGS = {
        'DJANGOCMS_SNIPPET_SEARCH': True,
    }
    update(settingsDict, DEFAULT_SNIPPE_SETTINGS)

    #=========================================================================
    # Zephyrus Landing.
    LANDING_FORM_SETTINGS = [
        ('djangocms_forms/form_template/default.html', 'Default'),
        ('landing/LandingForm.html', 'Landing Form'),
    ]

    LANDING_CMS_TEMPLATES = [
        ('landing/ThankYou.html', 'Landing Thank You Page'),
    ]

    if 'DJANGOCMS_FORMS_TEMPLATES' in settingsDict:
        formTemplates = settingsDict['DJANGOCMS_FORMS_TEMPLATES']
    else:
        formTemplates = LANDING_FORM_SETTINGS

    if 'CMS_TEMPLATES' in settingsDict:
        cmsTemplates = settingsDict['CMS_TEMPLATES']
    else:
        cmsTemplates = LANDING_CMS_TEMPLATES


    settingsDict['DJANGOCMS_FORMS_TEMPLATES'] = extendUnique(
        formTemplates, LANDING_FORM_SETTINGS)
    settingsDict['CMS_TEMPLATES'] = extendUnique(
        cmsTemplates, LANDING_CMS_TEMPLATES)

    #=========================================================================
    # Zephyrus blog
    if ZEPHYRUS_BLOG_ENABLE:
        BASE_DIR = settingsDict['BASE_DIR']
        if settingsDict['DEBUG'] or sys.platform.startswith("win32"):
            HAYSTACK_SETTINGS = {
                'HAYSTACK_CONNECTIONS': {
                    'default': {
                        'ENGINE': (
                            'haystack.backends.whoosh_backend.WhooshEngine'),
                        'PATH': os.path.join(BASE_DIR, 'whoosh_index'),
                        # 這個在執行 rebuild index 會爆炸
                        'EXCLUDED_INDEXES': [
                            'aldryn_people.search_indexes.PeopleIndex'
                        ],
                    },
                },
            }
        else:
            try:
                import xapain
                HAYSTACK_SETTINGS = {
                    'HAYSTACK_CONNECTIONS': {
                        'default': {
                            'ENGINE': 'xapian_backend.XapianEngine',
                            'PATH': os.path.join(BASE_DIR, 'xapian_index'),
                            # 這個在執行 rebuild index 會爆炸
                            'EXCLUDED_INDEXES': [
                                'aldryn_people.search_indexes.PeopleIndex'],
                        },
                    },
                }
            except Exception as e:
                HAYSTACK_SETTINGS = {
                    'HAYSTACK_CONNECTIONS': {
                        'default': {
                            'ENGINE': (
                                'haystack.backends.whoosh_backend.WhooshEngine'
                            ),
                            'PATH': os.path.join(BASE_DIR, 'whoosh_index'),
                            # 這個在執行 rebuild index 會爆炸
                            'EXCLUDED_INDEXES': [
                                'aldryn_people.search_indexes.PeopleIndex'
                            ],
                        },
                    },
                }

        if not 'HAYSTACK_CONNECTIONS' in settingsDict:
            update(settingsDict, HAYSTACK_SETTINGS)
        else:
            for key in settingsDict['HAYSTACK_CONNECTIONS']:
                value = settingsDict['HAYSTACK_CONNECTIONS'][key]
                if 'EXCLUDED_INDEXES' in value:
                    value['EXCLUDED_INDEXES'].append(
                        'aldryn_people.search_indexes.PeopleIndex')
                else:
                    value['EXCLUDED_INDEXES'] = [
                        'aldryn_people.search_indexes.PeopleIndex',
                    ]
            

    
