#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Service.py 10932 2018-03-26 07:56:30Z Kevin $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Kevin $ (last)
# $Date: 2018-03-26 15:56:30 +0800 (Mon, 26 Mar 2018) $
# $Revision: 10932 $

from inspect import isclass
"""
Define utility functions and service facade base class for all individual
app service.
"""
import types
import functools

import six
import singletonmixin

class ServiceGetter(singletonmixin.Singleton):
    """
    A singleton facade object to get all registered services.
    It provides getting, registering and unregistering functionality for
    service subsystem.
    """

    def __init__(self):
        """
        Constructor.
        """
        # Initialize singleton mixin.
        super(ServiceGetter, self).__init__()
        self.services = {}

    def get(self, name):
        """
        Get service instance.

        @param name Service name.
        """
        instance = self.services[name]

        if type(instance) is MetaBaseService:
            instance = instance.getInstance()
            self.services[name] = instance

        return instance

    def isRegistered(self, name):
        """
        Is registered or not.

        @param name Service name.
        @return True if service is registered.
        """
        return name in self.services

    def register(self, service, name=""):
        """
        Register a service.

        @param service Service class.
        @param name Short name to get this service, this name will be used to
                    be a get method name (getXXX). If the name is empty string
                    (default), it will try to use service name as name.
        @return True if success.
        """
        if not name:
            name = service.__name__

        if name in self.services:
            return False

        # Register in dict, but not initialize it yet.
        self.services[name] = service

        if six.PY2:
            getMethod = types.MethodType(
                functools.partial(self.get.__func__, name=name),
                self, ServiceGetter)
        else:
            getMethod = types.MethodType(
                functools.partial(self.get.__func__, name=name),
                self)

        setattr(self, 'get%s' % name, getMethod)

        return True

    def unregister(self, service=None, name=""):
        """
        Unregister a service.

        @param service Service class.
        @param name Service name.
        @return True if success.
        """
        if not name:
            name = service.__name__

        if name not in self.services:
            return False

        del self.services[name]

        exec(("del self.get%s" % name), locals())

        return True

# Instantiate it as a cache instance.
serviceGetter = ServiceGetter.getInstance()

class MetaBaseService(singletonmixin.MetaSingleton):
    """
    Metaclass for BaseService to automatically register service to
    serviceGetter.
    """

    def __init__(cls, name, bases, attrs):
        if name != 'BaseService' and bases != (singletonmixin.Singleton,):
            serviceGetter.register(cls)
        super(MetaBaseService, cls).__init__(name, bases, attrs)

@six.add_metaclass(MetaBaseService)
class BaseService(singletonmixin.Singleton):
    """
    Base class of service.
    """

    def __init__(self, _models=None):
        """
        Constructor.

        @param _models The models (domain objects) that this service will
                       control.
        """
        # Initialize singleton mixin.
        super(BaseService, self).__init__()

        if _models:
            from django.db import models

            methods = (
                ("create%s", 'create'),
                ("update%s", 'update'),
                ("delete%s", 'delete'),
                ("check%sExist", 'checkExist'),
                ("findAll%ss", 'all'),
                ("find%ss", 'filter'),
                ("get%s", 'get'),
                ("find%ssByIds", 'byIds'),
            )

            bos = []
            modelsTuple = ()
            if isinstance(_models, (tuple, list)):
                modelsTuple = _models
            else:
                modelsTuple = (_models,)

            for i in modelsTuple:
                
                if isclass(i):
                    if not issubclass(i, models.Model):
                        raise RuntimeError("_models must be a model or models.")
                    bos.append(i)
                    continue
                
                for bo in dir(i):
                    bo = getattr(i, bo)
                    try:
                        if (issubclass(bo, models.Model) and
                            bo.__module__ == i.__name__):
                            bos.append(bo)
                    except:
                        continue

            from Iuppiter.Model import ServiceManager
            from Iuppiter.Function import getInfo

            for bo in bos:
                setattr(self.__class__, bo.__name__, bo)

                for name, delegate in methods:
                    name = name % bo.__name__

                    # Method existed, don't overwrite it.
                    if hasattr(self, name):
                        continue
                    
                    if not hasattr(bo, 'objects'):
                        continue
                    
                    defaultManagerLam = "    six.get_unbound_function(getattr( \
                                                 ServiceManager, __delegate__ \
                                             ))(__bo__.objects, %s)"
                    serviceManagerLam = "    getattr( \
                                                 __bo__.objects, __delegate__ \
                                             )(%s)"
                    # Handle default manager.
                    if not isinstance(bo.objects, ServiceManager):
                        
                        try:
                            method = getattr(ServiceManager, delegate)
                            
                            lambdaString = defaultManagerLam
                            
                            wrapperDist = {
                                'ServiceManager': ServiceManager,
                                '__bo__': bo,
                                '__delegate__': delegate,
                                'six': six,
                            }
                        except:
                            continue # Ignore unsupported method.
    

                    # ServiceManger
                    else:
                    
                        try:
                            method = getattr(bo.objects, delegate)
                            lambdaString = serviceManagerLam
                            wrapperDist = {
                                '__bo__': bo,
                                '__delegate__': delegate,
                            }
                        except:
                            continue # Ignore unsupported method.
    

                    infodict = getInfo(six.get_unbound_function(method))
                    s = infodict['signature'].split(',')
    
                    # We do not use 'method' variable directly because we want
                    # to make sure the reference to manager's method is the
                    # same.
                    code = (("lambda %s: " + 
                             lambdaString) %
                            (infodict['signature'],
                             '' if s[0] == 'self' and len(s) == 1
                                else ','.join(s[1:])))

                    f = functools.update_wrapper(eval(code, 
                                      wrapperDist), 
                                      six.get_unbound_function(method),
                                      assigned=('__module__', '__name__',
                                                '__doc__', 'func_defaults' 
                                                if six.PY2 else '__defaults__'))
                        
                    f.__name__ = name
                    # Only for future usage.
                    f.undecorated = six.get_unbound_function(method)

                    if six.PY2:
                        f = types.MethodType(f, None, self.__class__)

                    setattr(self.__class__, name, f)
