#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: i18n.py 12691 2021-01-14 12:04:58Z Lavender $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Lavender $ (last)
# $Date: 2021-01-14 20:04:58 +0800 (Thu, 14 Jan 2021) $
# $Revision: 12691 $

from django.core.management.base import BaseCommand, CommandError
from optparse import make_option

from Iuppiter.DistUtil import getScript

class Command(BaseCommand):

    help = "i18n with Babel commands."

    # Validation is called explicitly each time the server is reloaded.
    requires_model_validation = False

    def add_arguments(self, parser):

        parser.add_argument('-a', '--app-name',  type=str, help="App name",
                            dest="app", default="")

        parser.add_argument('--op', dest='op', default='po',
            action='store', choices=['po', 'mo', 'all'],
            help='Process operation (po or mo).')


        parser.add_argument('--catalog', dest='catalog', default='django',
            action='store',
            type=str,
            help='Catalog name.')

        parser.add_argument('--extract_args', dest='extract_args', default='',
            action='store',
            type=str,
            help='Arguments pass to pybabel extract.')

    def handle(self, app='', *args, **options):
        if args:
            raise CommandError('Usage is i18n %s' % self.args)

        op = options.get('op', True)
        catalog = options.get('catalog')
        extractArgs = options.get('extract_args')

        import os
        from django.utils.translation.trans_real import to_locale
        from django.conf import settings

        curdir = os.path.abspath(os.path.curdir)

        # Make babel is able to load custom extractor from current project.
        # TODO:
        pythonPath = ''
        if 'PYTHONPATH' in os.environ:
            pythonPath = os.environ['PYTHONPATH']
            pythonPath += os.pathsep
        pythonPath += curdir
        os.environ['PYTHONPATH'] = pythonPath
        
        pyBabelPath = getScript("pybabel")

        info = []
        apps = settings.INSTALLED_APPS
        if app:
            if app in apps:
                apps = [app]
            else:
                raise CommandError('Invalid app name: %s' % app)

        for app in apps:
            path = app.replace('.', os.sep)
            info.append((app, path))
            # We recursively search for babel.cfg to append it to ready for
            # process info list.
            for root, dirs, files in os.walk(path):
                if 'babel.cfg' in files:
                    info.append((app, root))
                if '.svn' in dirs:
                    dirs.remove('.svn')  # Don't visit .svn directories.
                    
        defaultBabelPath = os.path.join(
            os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))), 
            "babel.cfg"
        )
        
        def exeCmd(cmd, path='.'):
            print("cd %s" % path)
            os.chdir(path)
            if not os.path.exists("babel.cfg"):
                cmd = cmd.replace("babel.cfg", defaultBabelPath)
            print(cmd)
            os.system(cmd)
            print("cd %s" % curdir)
            os.chdir(curdir)

        def do(path, callback):
            if os.path.exists(os.path.join(path, "locale")):
                for lang in settings.LANGUAGES:
                    locale = to_locale(lang[0])
                    p = os.path.join(path, "locale", locale, "LC_MESSAGES")
                    callback(locals())

        def po():
            # Extract messages to .pot.
            extractCmd = ("%s extract -F babel.cfg -o locale/%s.pot ."
                          % (pyBabelPath, catalog))
            exeCmd(extractCmd)

            for app, path in info:
                if os.path.exists(os.path.join(path, "locale")):
                    exeCmd(extractCmd, path)

            # Create or update .po.
            cCmd = "%s init -D %s -i locale/%s.pot -d locale -l %s"
            uCmd = "%s update -D %s -i locale/%s.pot -d locale -l %s" 

            def updatePo(vars):
                if os.path.exists(os.path.join(vars['p'], "%s.po" % catalog)):
                    exeCmd(uCmd 
                           % (pyBabelPath, catalog, catalog, vars['locale']),
                           vars['path'])
                else:
                    exeCmd(cCmd % 
                           (pyBabelPath, catalog, catalog, vars['locale']),
                           vars['path'])

            do('.', updatePo)
            for app, path in info:
                do(path, updatePo)

        def mo():
            cmd = ("%s compile -D %%s -d locale -l %%s --statistics"
                   % pyBabelPath)

            def generateMo(vars):
                if os.path.exists(os.path.join(vars['p'], "%s.po" % catalog)):
                    c = cmd
                    # English related translation must be generated always.
                    if vars['locale'].startswith('en'):
                        c += ' -f'
                    exeCmd(c % (catalog, vars['locale']), vars['path'])

            do('.', generateMo)
            for app, path in info:
                do(path, generateMo)

        if op == 'po':
            po()
        elif op == 'mo':
            mo()
        elif op == 'all':
            po()
            mo()
        else:
            return
