#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: forms.py 11741 2019-10-08 07:46:25Z Lavender $
#
# Copyright (c) 2018 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $ (last)
# $Date: 2019-10-08 15:46:25 +0800 (週二, 08 十月 2019) $
# $Revision: 11741 $

import os
import hashlib
import datetime

from django import forms
from django.conf import settings
from django.utils.translation import gettext_lazy as _
from django.contrib import messages
from django.core.files.storage import default_storage
from django.core.files.base import ContentFile
from django.conf import settings

from oscar import VERSION
from oscar.core.loading import get_model, get_class, get_classes
from ckeditor_uploader.widgets import CKEditorUploadingWidget

(RelatedFieldWidgetWrapper,
 ) = get_classes(
    'dashboard.widgets',
    ('RelatedFieldWidgetWrapper',))

# product ckeditor
catalogueViews = get_class('dashboard.catalogue', 'views')

Product = get_model('catalogue', 'Product')
_ProductForm = get_class('dashboard.catalogue.forms', 'ProductForm')

class ProductForm(_ProductForm):
    class Meta:
        model = Product
        
        if not VERSION >= (2, 0, 0):
            fields = [
                'title', 'upc', 'description', 'is_discountable', 'structure']
        else:
            fields = [
                'title', 'upc', 'description', 
                'is_discountable', 'structure', 'is_public']
            
        if hasattr(settings, 'IUNO_SHOP_PRODUCT_DESCRIPTION_CKEDITOR'):
            if settings.IUNO_SHOP_PRODUCT_DESCRIPTION_CKEDITOR:
                widgets = {
                    'structure': forms.HiddenInput(),
                    'description': CKEditorUploadingWidget(),
                }
            else:
                widgets = {
                    'structure': forms.HiddenInput(),
                    'description': forms.Textarea(attrs={'class': 'wysiwyg'}),
                }
        else:
            widgets = {
                'structure': forms.HiddenInput(),
                'description': forms.Textarea(attrs={'class': 'wysiwyg'}),
            }
            
catalogueViews.ProductCreateUpdateView.form_class = ProductForm

# content block ckeditor
if not VERSION >= (2, 0, 0):
    # RawHTMLForm
    promotionsViews = get_class('dashboard.promotions', 'views')

    RawHTML = get_model('promotions', 'RawHTML')
    _RawHTMLForm = get_class('dashboard.promotions.forms', 'RawHTMLForm')

    class RawHTMLForm(_RawHTMLForm):
        class Meta:
            model = RawHTML
            fields = ['name', 'body']
            
            if hasattr(settings, 'IUNO_SHOP_CONTENT_BLOCK_CKEDITOR'):
                if settings.IUNO_SHOP_CONTENT_BLOCK_CKEDITOR:
                    widgets = {
                        'body': CKEditorUploadingWidget(),
                    }
                else:
                    widgets = {
                        'body': forms.Textarea(attrs={'class': 'wysiwyg'}),
                    }
            else:
                widgets = {
                    'body': forms.Textarea(attrs={'class': 'wysiwyg'}),
                }
                
    promotionsViews.CreateRawHTMLView.form_class = RawHTMLForm
    promotionsViews.UpdateRawHTMLView.form_class = RawHTMLForm
else:
    # RawHTMLForm
    pass
    
# Pages
pagesViews = get_class('dashboard.pages', 'views')

FlatPage = get_model('flatpages', 'FlatPage')
_PageUpdateForm = get_class('dashboard.pages.forms', 'PageUpdateForm')

class PageUpdateForm(_PageUpdateForm):
    class Meta(_PageUpdateForm.Meta):
        model = FlatPage
        fields = ('title', 'url', 'content')
        
        if hasattr(settings, 'IUNO_SHOP_PAGE_CKEDITOR'):
            if settings.IUNO_SHOP_CONTENT_BLOCK_CKEDITOR:
                widgets = {
                    'content': CKEditorUploadingWidget(),
                }
            else:
                widgets = {
                    'content': forms.Textarea(attrs={'class': 'wysiwyg'}),
                }
        else:
            widgets = {
                'content': forms.Textarea(attrs={'class': 'wysiwyg'}),
            }
            
pagesViews.PageCreateView.form_class = PageUpdateForm
pagesViews.PageUpdateView.form_class = PageUpdateForm


if VERSION >= (2, 0, 0):
    # option type
    from Iuno.shop.models import (
        OptionType, TYPE_CHOICES, TEXT, INTEGER, 
        BOOLEAN, FLOAT, DATE, DATETIME, OPTION, RICHTEXT, FILE, IMAGE
    )
    from Iuppiter.view import FileDragDropWidget

    Option = get_model('catalogue', 'Option')
    ProductAttribute = get_model('catalogue', 'ProductAttribute')
    AttributeOptionGroup = get_model('catalogue', 'AttributeOptionGroup')
    _OptionForm = get_class('dashboard.catalogue.forms', 'OptionForm')

    class OptionForm(_OptionForm):

        optionType = forms.ChoiceField(
            choices=TYPE_CHOICES, label=_("Option type"))
        optionGroup = forms.ModelChoiceField(
            queryset=AttributeOptionGroup.objects.all(), 
            label=_("Option group")
        )
        
        def __init__(self, *args, **kwargs):
            if "instance" in kwargs:
                instance = kwargs["instance"]
                if instance:
                    ot, created = OptionType.objects.get_or_create(
                        option=instance,
                        defaults={
                            'type': TEXT, 
                            'optionGroup': None,
                        }
                    )
                    kwargs["initial"] = {
                        "optionType": ot.type,
                        "optionGroup": ot.optionGroup,
                    }
               
            super().__init__(*args, **kwargs)
            
            self.fields["optionGroup"].required = False
            self.fields["optionGroup"].help_text = _("Select an option group")
            
            remote_field = ProductAttribute._meta.get_field(
                'option_group').remote_field
            self.fields["optionGroup"].widget = RelatedFieldWidgetWrapper(
                self.fields["optionGroup"].widget, remote_field)
                
        def save(self, commit=True):
            instance = super(OptionForm, self).save(commit=commit)
            type = self.cleaned_data['optionType']
            optionGroup = self.cleaned_data['optionGroup']
            
            ot, created = OptionType.objects.update_or_create(
                option=instance,
                defaults={
                    'type': type, 
                    'optionGroup': optionGroup,
                }
            )
            if commit:
                instance.save()
            return instance
                
        class Meta:
            model = Option
            fields = ['name', 'type']

    catalogueViews.OptionCreateUpdateView.form_class = OptionForm

    from oscar.apps.basket import views as basketViews
    from oscar.templatetags import basket_tags
    from oscar.forms.widgets import DatePickerInput, DateTimePickerInput

    _AddToBasketForm = get_class('basket.forms', 'AddToBasketForm')

    class AddToBasketForm(_AddToBasketForm):

        def _add_option_field(self, product, option):
            ot, created = OptionType.objects.get_or_create(
                option=option,
                defaults={
                    'type': TEXT, 
                    'optionGroup': None,
                }
            )
            kwargs = {
                'required': option.is_required,
                'label': option.name,
            }
            
            if ot.type == OPTION:
                # OPTION
                choice = []
                if ot.optionGroup:
                    choiceOptions = [
                        o.option 
                        for o in ot.optionGroup.options.all().order_by('id')]
                    
                    for c in choiceOptions:
                        choice.append((c, c))
                    
                kwargs.update({
                    'widget': forms.Select,
                    'choices': choice,
                })
                self.fields[option.code] = forms.ChoiceField(**kwargs)
            elif ot.type == INTEGER:
                # INTEGER
                self.fields[option.code] = forms.IntegerField(**kwargs)
            elif ot.type == BOOLEAN:
                # Boolean
                self.fields[option.code] = forms.BooleanField(**kwargs)
            elif ot.type == FLOAT:
                # FLOAT
                self.fields[option.code] = forms.FloatField(**kwargs)
            elif ot.type == RICHTEXT:
                # RICHTEXT
                kwargs.update({
                    'widget': forms.Textarea,
                })
                self.fields[option.code] = forms.CharField(**kwargs)
            elif ot.type == DATE:
                # DATE
                kwargs.update({
                    'widget': DatePickerInput,
                })
                self.fields[option.code] = forms.DateField(**kwargs)
            elif ot.type == DATETIME:
                # DATETIME
                kwargs.update({
                    'widget': DateTimePickerInput,
                })
                self.fields[option.code] = forms.DateTimeField(**kwargs)
            elif ot.type == FILE:
                # FILE
                kwargs.update({
                    'widget': forms.FileInput,
                })
                self.fields[option.code] = forms.FileField(**kwargs)
            elif ot.type == IMAGE:
                # IMAGE
                kwargs.update({
                    'widget': FileDragDropWidget
                })
                self.fields[option.code] = forms.ImageField(**kwargs)
            else:
                # TEXT
                self.fields[option.code] = forms.CharField(**kwargs)
                
    class SimpleAddToBasketForm(AddToBasketForm):
        def __init__(self, *args, **kwargs):
            super().__init__(*args, **kwargs)
            if 'quantity' in self.fields:
                self.fields['quantity'].initial = 1
                self.fields['quantity'].widget = forms.HiddenInput()
    
    BasketMessageGenerator = get_class('basket.utils', 'BasketMessageGenerator')
    
    class BasketAddView(basketViews.BasketAddView):
    
        form_class = AddToBasketForm 
        
        def form_valid(self, form): 
            # save file
            options = form.cleaned_options()
            
            for optionData in options:
                option = optionData['option']
                ot, created = OptionType.objects.get_or_create(
                    option=option,
                    defaults={
                        'type': TEXT, 
                        'optionGroup': None,
                    }
                )
                if ot.type in [FILE, IMAGE]:
                    f = self.request.FILES.get(option.code, None)
                    if f:
                        pathId = hashlib.md5(
                            ("%s%s" % (
                                str(f),
                                datetime.datetime.now(),
                            )).encode()).hexdigest()
                        mediaPath = os.path.join(
                            'OptionUpload', pathId, str(f))
                        uploadPath = os.path.join(
                            settings.MEDIA_ROOT, mediaPath)
                        if os.path.exists(uploadPath):
                            os.remove(uploadPath)
                        
                        path = default_storage.save(
                            mediaPath, 
                            ContentFile(f.read())
                        )
                        
                        optionData['value'] = mediaPath
                    else:
                        optionData['value'] = ''
                    
            # 以下複製的，因為要修改其中一段
            offers_before = self.request.basket.applied_offers()

            line, created = self.request.basket.add_product(
                form.product, form.cleaned_data['quantity'],
                options)

            messages.success(self.request, self.get_success_message(form),
                             extra_tags='safe noicon')

            # Check for additional offer messages
            BasketMessageGenerator().apply_messages(self.request, offers_before)

            # Send signal for basket addition
            self.add_signal.send(
                sender=self, product=form.product, user=self.request.user,
                request=self.request)

            return super(basketViews.BasketAddView, self).form_valid(form)
      
    #from oscar.apps.basket import apps
    basketViews.BasketAddView.form_class = AddToBasketForm 
    basketViews.BasketAddView.form_valid = BasketAddView.form_valid   
    #apps.BasketConfig.add_view = BasketAddView
    basket_tags.AddToBasketForm = AddToBasketForm
    basket_tags.SimpleAddToBasketForm = SimpleAddToBasketForm
    

