#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: generate_test_data.py 9694 2016-06-29 09:50:19Z Judy $
#
# Copyright (c) 2016 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Judy $
# $Date: 2016-06-29 17:50:19 +0800 (三, 29  6 2016) $
# $Revision: 9694 $

import datetime
import random
import string

def createStatTestData(contactNum=100, testContactNum=5, testDays=15,
                       sent=90, error=10, invalid=5, opened=80, openedOnSite=30,
                       linkOpened=60, unsubscription=10, bounce=5, complain=5,
                       reject=5, sendTest=90):
    """
    Create statistic test data.
    
    @param contactNum The amount of contact.
                      The default value is 100.    
    @param testContactNum The amount of test contact.
                          The default value is 5.  
    @param testDays The range of test day.
                    The default value is 15.   
    @param sent The percentage of sent email.
                The default value is 90.
                NOTICE: 'sent' plus 'error' should equal to 100.
    @param error The percentage of error email.
                 The default value is 10.
                 NOTICE: 'sent' plus 'error' should equal to 100.
    @param invalid The percentage of invalid email.
                   The default value is 5.
    @param opened The percentage of opened email.
                  The default value is 80.
                  NOTICE: 'opened' include 'opened unique', 'opened on site' and 
                  'link opened', so 'opened' should bigger than 'opened on site'
                  and 'link opened'.           
    @param openedOnSite The percentage of email opened on site.
                          The default value is 30.
    @param linkOpened The percentage of link opened email.
                       The default value is 60.
    @param unsubscription The percentage of unsubscribe email.
                          The default value is 10.
    @param bounce The percentage of bounce email.
                  The default value is 5.
    @param complain The percentage of complain email.
                    The default value is 5.
    @param reject The percentage of reject email.
                  The default value is 5.
    @param sendTest The percentage of success test email.
                    The default value is 90.
    """
    
    from Theophrastus.models import Contact, MailingList, Newsletter, \
                                    ContactMailingStatus
    
    def overwriteAutoNow(m):
        for field in m._meta.local_fields:
            if field.name == "creation_date":
                field.auto_now = False
                field.auto_now_add = False
            elif field.name == "modification_date":
                field.auto_now = False
                field.auto_now_add = False        
    
    def createContact(firstName, lastName, email, modifyTime, isTester=False, 
                      isSubscriber=True, isValid=True):
        newContact = Contact(email=email, first_name=firstName, 
                             last_name=lastName, subscriber=isSubscriber,
                             valid=isValid, tester=isTester, tags='',
                             creation_date=modifyTime,
                             modification_date=modifyTime)    
        overwriteAutoNow(newContact)          
        newContact.save()
        return newContact
    
    def createMailingList(subscriberList, modifyTime, name='testList'):
        newMailingList = MailingList(name=name, description='', 
                                     creation_date=modifyTime, 
                                     modification_date=modifyTime)
        overwriteAutoNow(newMailingList)  
        newMailingList.save()
        newMailingList.subscribers.add(*subscriberList)
        return newMailingList
    
    def createNewsletter(mailingList, testContactList, sendingTime, modifyTime,
                         title='testNewsletter', status=2):
        newNewsletter = Newsletter(title=title, status=status, slug=title,
                                   sending_date=sendingTime,
                                   mailing_list=mailingList,
                                   creation_date=modifyTime, 
                                   modification_date=modifyTime)
        overwriteAutoNow(newNewsletter)
        newNewsletter.save()
        newNewsletter.test_contacts.add(*testContactList)
        return newNewsletter
    
    def createContactStatus(newsletter, contact, status, createTime):
        newContactStatus = ContactMailingStatus(newsletter=newsletter,
                                                contact=contact,
                                                status=status, 
                                                creation_date=createTime)
        overwriteAutoNow(newContactStatus)
        newContactStatus.save()
        return newContactStatus
    
    timeStringFormat = '%Y-%m-%d %H:%M:%S'
    now = datetime.datetime.now()
    startTime = now - datetime.timedelta(days=testDays)
    startTimeStr = startTime.strftime(timeStringFormat)
    
    # create contacts
    contacts = []
    testContacts = []
    for c in range(contactNum):
        firstName = "c%d" % c
        lastName = "c%d" % c
        # random string:
        # https://taizilongxu.gitbooks.io/
        # stackoverflow-about-python/content/63/README.html
        email = "c%s@email.com" % ''.join(random.choice(
                string.ascii_uppercase + string.digits) for _ in range(8))
        modifyTime = startTimeStr
        isTester = False
        
        if c in range(testContactNum):
            isTester = True        
            _contact = createContact(firstName, lastName, email, 
                                     modifyTime, isTester)
            testContacts.append(_contact)
        else:
            _contact = createContact(firstName, lastName, email, 
                                     modifyTime, isTester)
        contacts.append(_contact)
    
    # create mailing list
    mailingList = createMailingList(contacts, startTimeStr)
    
    # create newsletter
    title = 'test_%s' % ''.join(random.choice(
             string.ascii_uppercase + string.digits) for _ in range(8))
    newsletter = createNewsletter(mailingList, testContacts, 
                                  startTimeStr, startTimeStr, title=title)
    
    # create status
    # --
    # SENT_TEST = -1
    # SENT = 0
    # ERROR = 1
    # INVALID = 2
    # OPENED = 4
    # OPENED_ON_SITE = 5
    # LINK_OPENED = 6
    # UNSUBSCRIPTION = 7
    # BOUNCE = 8
    # COMPLAIN = 9
    # REJECT = 10
    # --
    # -- sent and error
    contacts = random.sample(contacts, contactNum)
    sentNum = int(round(contactNum * sent / 100))
    errorNum = int(round(contactNum * error / 100))
    sentContacts = []
    for index, con in enumerate(contacts):
        if index < errorNum:
            createContactStatus(newsletter, con, ContactMailingStatus.ERROR, 
                                startTimeStr)
        else:
            createContactStatus(newsletter, con, ContactMailingStatus.SENT, 
                                startTimeStr)
            sentContacts.append(con)
    
    # -- invalid
    sentContacts = random.sample(sentContacts, len(sentContacts))
    invalidNum = int(round(sentNum * invalid / 100))
    validContacts = []
    time = startTime + datetime.timedelta(days=random.randint(0, 2))
    time = time.strftime(timeStringFormat)
    for index, con in enumerate(sentContacts):
        if index < invalidNum:
            createContactStatus(
                newsletter, con, ContactMailingStatus.INVALID, time)
        else:
            validContacts.append(con)
            
            
    # -- opened, opened on site and link opened
    openedNum = int(round(sentNum * opened / 100))
    openedOnSiteNum = int(round(sentNum * openedOnSite /100))
    linkOpenedNum = int(round(sentNum * linkOpened / 100))
   
    # ---- opened    
    openedContacts = random.sample(validContacts, openedNum)
    for contact in openedContacts:
        statusNum = random.randint(1, 5)
        for j in range(statusNum):
            time = startTime + datetime.timedelta(
                                        days=random.randint(0, testDays))
            time = time.strftime(timeStringFormat)
            createContactStatus(
                newsletter, contact, ContactMailingStatus.OPENED, time)
    # ---- opened on site
    openedOnSiteContacts = random.sample(openedContacts, openedOnSiteNum)
    for contact in openedOnSiteContacts:
        statusNum = random.randint(1, 5)
        for j in range(statusNum):
            time = startTime + datetime.timedelta(
                                        days=random.randint(0, testDays))
            time = time.strftime(timeStringFormat)
            createContactStatus(newsletter, 
                                contact, 
                                ContactMailingStatus.OPENED_ON_SITE, 
                                time)
    # ---- link opened
    linkedContacts = random.sample(openedContacts, linkOpenedNum)
    for contact in linkedContacts:
        statusNum = random.randint(1, 5)
        for j in range(statusNum):
            time = startTime + datetime.timedelta(
                                        days=random.randint(0, testDays))
            time = time.strftime(timeStringFormat)
            createContactStatus(newsletter, contact, 
                                ContactMailingStatus.LINK_OPENED, time)
    
    # -- unsubscribe
    unsubscriptionNum = int(round(contactNum * unsubscription /100))
    unsubscriptionContacts = random.sample(validContacts, unsubscriptionNum)
    for contact in unsubscriptionContacts:
        time = startTime + datetime.timedelta(days=random.randint(0, testDays))
        time = time.strftime(timeStringFormat)
        createContactStatus(newsletter, contact, 
                            ContactMailingStatus.UNSUBSCRIPTION, time)
    
    # -- bounce, reject and complain
    bounceNum = int(round(sentNum * bounce / 100))
    complainNum = int(round(sentNum * complain / 100))
    rejectNum = int(round(sentNum * reject / 100))
    
    bounceContact = []
    complainContact = []
    rejectContact = []
    
    validContacts = random.sample(validContacts, len(validContacts))
    for i in range(bounceNum):
        time = startTime + datetime.timedelta(days=random.randint(0, testDays))
        time = time.strftime(timeStringFormat)
        _contact = validContacts[i]
        createContactStatus(
            newsletter, _contact, ContactMailingStatus.BOUNCE, time)
        bounceContact.append(_contact)
    
    for i in range(complainNum):
        time = startTime + datetime.timedelta(days=random.randint(0, testDays))
        time = time.strftime(timeStringFormat)
        _contact = validContacts[i + bounceNum-1]
        createContactStatus(
            newsletter, _contact, ContactMailingStatus.COMPLAIN, time)
        complainContact.append(_contact)
    
    for i in range(rejectNum):
        time = startTime + datetime.timedelta(days=random.randint(0, testDays))
        time = time.strftime(timeStringFormat)
        _contact = validContacts[i + bounceNum-1 + complainNum - 1]
        createContactStatus(
            newsletter, _contact, ContactMailingStatus.REJECT, time)
        rejectContact.append(_contact)
        
    # send test
    time = startTime + datetime.timedelta(days=random.randint(0, testDays))
    successTestContactNum = int(round(testContactNum * sendTest / 100))
    successTestContact = random.sample(testContacts, successTestContactNum)    
    for test in successTestContact:
        createContactStatus(
            newsletter, test, ContactMailingStatus.SENT_TEST, time)     
    

from django.core.management.base import BaseCommand

class Command(BaseCommand):
    """
    Command for create statistic data.
       
    If you haven't set any option argument, the command will automatically 
    generate test data with default value and create a fixture JSON file which 
    named 'test.json'. You can setting option argument to create your own test
    data as well.
    
    A test server can be run with the test data we just created by doing
    following steps:
    
    1) Add a folder called 'fixture' in any app in your project.
    2) Move the JSON file into folder 'fixture'.
    3) Copy 'initial_data.json' from demo project into 'fixture'
    4) Run command 
       python manage.py testserver --addrport='YOURPORT' initial_data 'TESTDATA'
       TESTDATA would be the file name of test data. Then, a test server will 
       start.
       example: python manage.py testserver --addrport=7788 test initial_data
    
    See more on related ticket: 
    https://code.nuwainfo.com/trac/mercurius/ticket/2741
    
    @param contact_num The Amount of contact for testing.
                       The default is 100.
    @param test_contact_num The Amount of test contact for testing.
                            The default is 5.
    @param test_days The time range for testing.
                     The default is 15.
    @param create_random_data Value can be 'T' (True) or 'F' (False).
                              The default value is 'F'.
                              If value is 'T', we will randomly set status of 
                              test data.
    @param sent The percentage of sent email.
                The default value is 90.
                NOTICE: 'sent' plus 'error' should equal to 100.
    @param error The percentage of error email.
                 The default value is 10.
                 NOTICE: 'sent' plus 'error' should equal to 100.
    @param invalid The percentage of invalid email.
                   The default value is 5.
    @param opened The percentage of opened email.
                  The default value is 80.
                  NOTICE: 'opened' include 'opened unique', 'opened on site' and 
                  'link opened', so 'opened' should bigger than 'opened on site'
                  and 'link opened'.           
    @param opened_on_site The percentage of email opened on site.
                          The default value is 30.
    @param link_opened The percentage of link opened email.
                       The default value is 60.
    @param unsubscription The percentage of unsubscribe email.
                          The default value is 10.
    @param bounce The percentage of bounce email.
                  The default value is 5.
    @param complain The percentage of complain email.
                    The default value is 5.
    @param reject The percentage of reject email.
                  The default value is 5.
    @param sendTest The percentage of success test email.
                    The default value is 90.
    @param dumpdata The name of output file.
                    The default value is 'test.json'.
    """
    
    def add_arguments(self, parser):
        # optional arguments
        parser.add_argument('--contact_num',
            dest='contactNum',
            default=100,
            metavar='CONTACT_NUM',
            help='Amount of contact for testing.')
        parser.add_argument('--test_contact_num',
            dest='testContactNum',
            default=5,
            metavar='TEST_CONTACT_NUM',
            help='Amount of test contact for testing.')
        parser.add_argument('--test_days',
            dest='testDays',
            default=15,
            metavar='TEST_DAYS',
            help='Time range for testing.')
        parser.add_argument('--create_random_data',
            dest='randomData',
            default=False,
            metavar='CREATE_RANDOM_DATA',
            type=bool,
            help="""Automatic create statistic data, 
                    value can be T (True) or F (False).""")
        parser.add_argument('--sent',
            dest='sent',
            default=90,
            help='Percentage of sent email.')
        parser.add_argument('--error',
            dest='error',
            default=10,
            help='Percentage of error email.')
        parser.add_argument('--invalid',
            dest='invalid',
            default=5,
            help='Percentage of invalid email.')
        parser.add_argument('--opened',
            dest='opened',
            default=80,
            help='Percentage of opened email.')
        parser.add_argument('--opened_on_site',
            dest='openedOnSite',
            default=30,
            metavar='OPENED_ON_SITE',
            help='Percentage of opened on site email.')
        parser.add_argument('--link_opened',
            dest='linkOpened',
            default=60,
            metavar='LINK_OPENED',
            help='Percentage of link opened email.')
        parser.add_argument('--unsubscription',
            dest='unsubscription',
            default=10,
            help='Percentage of unsubscription email.')
        parser.add_argument('--bounce',
            dest='bounce',
            default=5,
            help='Percentage of bounce email.')
        parser.add_argument('--complain',
            dest='complain',
            default=5,
            help='Percentage of complain email.')
        parser.add_argument('--reject',
            dest='reject',
            default=5,
            help='Percentage of reject email.')
        parser.add_argument('--sendTest',
            dest='sendTest',
            default=90,
            metavar='SENT_TEST',
            help='Percentage of test email.')
        parser.add_argument('--dumpdata',
            dest='dataFileName',
            default='test.json',
            metavar='DATA_FILE_NAME',
            help='Dump data as json format.')

    def handle(self, *args, **options):
        from django.db import connection
        from django.core.management import call_command
        
        # Create a test database.
        db_name = connection.creation.create_test_db(
            verbosity=True, autoclobber=True, serialize=False)

        # Import the fixture data into the test database.
        # from django.core.management import call_command        
        # call_command('loaddata', *fixture_labels, **{'verbosity': True})
        
        _contactNum = int(options['contactNum'])
        _testContactNum = int(options['testContactNum'])
        _testDays = int(options['testDays'])
        
        _sent = int(options['sent'])
        _error = int(options['error'])
        _invalid = int(options['invalid'])
        _opened = int(options['opened'])
        _openedOnSite = int(options['openedOnSite'])
        _linkOpened = int(options['linkOpened'])
        _unsubscription = int(options['unsubscription'])
        _bounce = int(options['bounce'])
        _complain = int(options['complain'])
        _reject = int(options['reject'])
        _sendTest = int(options['sendTest'])
        
        if options['randomData'] == True:
            _sent = random.randint(80, 100)
            _error = 100 - _sent
            _invalid = random.randint(0, int(_sent/4))
            __valid = _sent - _invalid
            _opened = random.randint(int(__valid/2), __valid)
            _openedOnSite = random.randint(int(_opened/2), _opened)
            _linkOpened = random.randint(int(_opened/2), _opened)
            _unsubscription = random.randint(0, int(__valid/2))
            _bounce = random.randint(0, int(__valid/2))
            _complain = random.randint(0, int(__valid/2))
            _reject = random.randint(0, int(__valid/2))
            _sendTest = random.randint(0, 100)
            
            output = ("RANDOM TEST DATA: \n"
                     "SENT: {sent}\n"
                     "ERROR: {error}\n"
                     "INVALID: {invalid}\n"
                     "OPENED: {opened}\n"
                     "OPENED_ON_SITE: {openedOnSite}\n"
                     "LINK_OPENED: {linkOpened}\n"
                     "UNSUBSCRIPTION: {unsubscription}\n"
                     "BOUNCE: {bounce}\n"
                     "COMPLAIN: {complain}\n"
                     "REJECT: {reject}\n"
                     "SEND_TEST: {sendTest}")
            output = output.format(sent=_sent,
                                   error=_error,
                                   invalid=_invalid,
                                   opened=_opened,
                                   openedOnSite=_openedOnSite,
                                   linkOpened=_linkOpened,
                                   unsubscription=_unsubscription,
                                   bounce=_bounce,
                                   complain=_complain,
                                   reject=_reject,
                                   sendTest=_sendTest)
            self.stdout.write(output)
        
        createStatTestData(contactNum=_contactNum, 
                           testContactNum=_testContactNum,
                           testDays=_testDays,
                           sent=_sent, 
                           error=_error, 
                           invalid=_invalid, 
                           opened=_opened, 
                           openedOnSite=_openedOnSite,
                           linkOpened=_linkOpened, 
                           unsubscription=_unsubscription,
                           bounce=_bounce, 
                           complain=_complain,
                           reject=_reject, 
                           sendTest=_sendTest)
        
        call_command(
            'dumpdata', 'newsletter', output=options['dataFileName'])
        self.stdout.write('Successfully create fixture.')
        
        self.stdout.write('Successfully create test data.')
