#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: views.py 10806 2018-03-02 04:29:05Z David $
#
# Copyright (c) 2013 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: David $
# $Date: 2013-01-30 12:13:45 +0800$
# $Revision: 10806 $

from django.shortcuts import render_to_response, get_object_or_404
from django.utils.translation import ugettext_lazy as _
from django.views.decorators.csrf import csrf_exempt
from django.template.loader import render_to_string
from django.core.urlresolvers import reverse
from django.shortcuts import redirect
from django.conf import settings
from django.contrib.admin.views.decorators import staff_member_required
from django.template import RequestContext
from django.contrib.auth.decorators import login_required

from Iuppiter.dispatch import ThreadPool
from Iuppiter.extension.views import jsonpCallback
from Iuppiter.Logging import createLogger

from emencia.django.newsletter.models import Newsletter, MailingList, \
                                             ContactMailingStatus, Contact

from Theophrastus.cloud.Mailer import CloudMailer
from Theophrastus.newsletter_extension.models import Header, Footer, ErrorMessage
from Theophrastus.cloud.service import CloudService
from Theophrastus.Utility import _slugify

@staff_member_required
@jsonpCallback(noPadding=True)
def getDefaultHeaderText(request):

    c = render_to_string('NewsletterDefaultHeader.html')
    return {'content': unicode(c)}

@staff_member_required
@jsonpCallback(noPadding=True)
def getDefaultFooterText(request):

    c = render_to_string('NewsletterDefaultFooter.html')
    return {'content': unicode(c)}

@csrf_exempt
@jsonpCallback(noPadding=True)
def getNewsletterInfo(request):
    try:
        newsletterId = request.GET.get('id', None)
        if newsletterId is None:
            raise RuntimeError('Invalid id.')
        else:
            newsletterId = int(newsletterId)
            newsletter = Newsletter.objects.get(id=newsletterId)
            formatStr = '%Y-%m-%d %H:%M:%S'
            return {
                'id': newsletter.id,
                'status': newsletter.status,
                'sendingDate': newsletter.sending_date.strftime(formatStr),
                'slug': newsletter.slug
            }
    except Exception as e:
        return {'error': str(e)}

def resendErrorContact(request, newsletterId):
    """
    Resend newsletter to the contacts which has error when
    sending mail previously.

    @param request Request instance.
    """
    SENDBOX_THREAD_POOL_COUNT = settings.SENDBOX_THREAD_POOL_COUNT
    ENABLE_FAKE_MAIL = settings.ENABLE_FAKE_MAIL
    ENABLE_FAKE_SMTP = settings.ENABLE_FAKE_SMTP
    LOG_LEVEL = settings.LOG_LEVEL
    MAILER_LOGFILE = settings.MAILER_LOGFILE

    sendboxPool = ThreadPool(SENDBOX_THREAD_POOL_COUNT)

    mailerLogger = createLogger(
                'theophrastusMailer', MAILER_LOGFILE, loggingLevel=LOG_LEVEL)

    newsletter = get_object_or_404(Newsletter, id=newsletterId)
    opts = Newsletter._meta

    context = {'title': _('Historic of %s') % newsletter.__unicode__(),
               'original': newsletter,
               'opts': opts,
               'object_id': newsletter.pk,
               'app_label': opts.app_label}

    # TODO: set --> distinct()
    sentContactIdList = set(
        ContactMailingStatus.objects.filter(
            newsletter_id=newsletter.id, status=ContactMailingStatus.SENT
        ).values_list('contact_id').distinct())

    contactIdSet = set(
        ContactMailingStatus.objects.filter(
            newsletter_id=newsletter.id, 
            status=ContactMailingStatus.ERROR
        ).exclude(
                contact_id__in=sentContactIdList).values_list('contact_id'))

    if contactIdSet:
        contactIdList = [ele[0] for ele in list(contactIdSet)]
        contactList = Contact.objects.filter(id__in=contactIdList)

        mailer = CloudMailer(newsletter, pool=sendboxPool,
                                        fakeMail=ENABLE_FAKE_MAIL,
                                        fakeSMTP=ENABLE_FAKE_SMTP,
                                        logger=mailerLogger)

        mailer.useCloud = True if newsletter.server.id == 1 else False
        mailer.smtp_connect()
        mailer.attachments = mailer.getAttachments()

        for contact in contactList:
            mailer.send(contact)

        context['sendConfirm'] = _('Resend error emails finish')
    else:
        context['information'] = _('All errors have been resent')

    return render_to_response('NewsletterHistoric.html',
                              context, context_instance=RequestContext(request))


def viewErrorDetails(request, slug, contactMailingStatusId):
    """
    Display the error message of a newsletter contact.

    @param request Request instance.
    @param slug Newsletter's slug.
    @param cloudmailinformationId Id of CloudMailInformation.
    """

    contactMailingStatus = ContactMailingStatus.objects.filter(
                                id=contactMailingStatusId)[0]
    opts = Newsletter._meta
    cloudSrv = CloudService.getInstance()
    newsletter = get_object_or_404(Newsletter, slug=slug)

    if contactMailingStatus.status == ContactMailingStatus.ERROR:
        content = ErrorMessage.objects.filter(
            newsletter=contactMailingStatus.newsletter,
            contact=contactMailingStatus.contact
        )[0].message
    else:
        content = cloudSrv.getCloudMailInformation(
            newsletter_id=contactMailingStatus.newsletter.id,
            contact_id=contactMailingStatus.contact.id,
            errorOnNotFound=False
        ).message
    
    context = {
        'title': _('Error of contact: %s') % contactMailingStatus.contact.email,
        'original': newsletter,
        'opts': opts,
        'object_id': newsletter.pk,
        'app_label': opts.app_label,
        'content': content
    }
    return render_to_response(
        'NewsletterHistoricErrorDetail.html',
        context,
        context_instance=RequestContext(request)
    )

@login_required
def viewNewsletterHistoric(request, slug):
    """Display the historic of a newsletter"""
    opts = Newsletter._meta
    newsletter = get_object_or_404(Newsletter, slug=slug)

    context = {'title': _('Historic of %s') % newsletter.__unicode__(),
               'original': newsletter,
               'opts': opts,
               'object_id': newsletter.pk,
               'app_label': opts.app_label}
    return render_to_response('NewsletterHistoric.html',
                              context, context_instance=RequestContext(request))

# Theophrastus/static/js/Newsletter.js
# Theophrastus/templates/admin/newsletter/newsletter/change_form.html
@jsonpCallback(noPadding=True)
def getSlugName(request):
    """
    Get slug name.

    @param request Request instance.
    """
    title = request.GET.get('title', '')
    if title:
        return {'slug': _slugify(title)}
    else:
        return {}

# Theophrastus/static/js/Newsletter.js
# Theophrastus/templates/admin/newsletter/newsletter/change_form.html
@jsonpCallback()
def getMailingListInfo(request):
    """
    Get MailingList information.

    @param request Request instance.
    """
    mailingListId = request.GET.get('mailingList', '')
    newsletterIds = request.GET.get('newsletters', '')
    newsletterIds = newsletterIds.split('_')
    try:
        if mailingListId:
            mailingList = MailingList.objects.get(id=mailingListId)
            return {
                'subscribersCount': mailingList.subscribers_count(),
                'name': mailingList.name
            }
        elif newsletterIds:
            info = []
            for n in Newsletter.objects.filter(id__in=newsletterIds):
                mailingList = n.mailing_list
                info.append({
                    'subscribersCount': mailingList.subscribers_count(),
                    'name': n.title
                })

            return {'info': info}
    except Exception as e:
        pass
    return {'error': str(e)}