#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: views.py 10694 2017-12-29 09:22:09Z David $
#
# Copyright (c) 2013 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: David $
# $Date: 2017-12-29 17:22:09 +0800 (Fri, 29 Dec 2017) $
# $Revision: 10694 $

import base64
from datetime import datetime
from urlparse import unquote
from cgi import parse_qsl

from django.http import Http404
from django.views.decorators.csrf import csrf_exempt
from django.contrib.auth import authenticate
from django.conf import settings
from django.contrib.admin.views.decorators import staff_member_required

from Iuppiter.Encryption import Encryptor
from Iuppiter.extension.views import jsonpCallback, view
from Iuppiter.Encoding import _unicode
from Iuppiter.Logging import createDebugLogger

from Theophrastus.integration.service import IntegrationService
from Theophrastus.statistics.StatisticsHandler import getStatisticsTotalData
from Theophrastus.statistics.StatisticsHandler import getStatisticsTotalRateData
from Theophrastus.statistics.StatisticsHandler import getSubscriberHistory
from Theophrastus.models import ContactMailingStatus
from Theophrastus.cloud.views import filterEmail, doVerifyEmailAddress
from Theophrastus.cloud.views import doSetNotificationTopic, checkSenderByCache
from Theophrastus.cloud.views import checkNotificationTopic
from Theophrastus.cloud.Mailer import CloudMailer
from Theophrastus.Utility import getComment

integrationSrv = IntegrationService.getInstance()

ENABLE_FAKE_MAIL = settings.ENABLE_FAKE_MAIL
ENABLE_FAKE_SMTP = settings.ENABLE_FAKE_SMTP

SECRET_KEY = settings.SECRET_KEY

logger = createDebugLogger('TheophrastusIntegrationApi', settings.API_LOG)

class TokenGenerator(object):
    def __init__(self, secretKey):
        self.encryptor = Encryptor(secretKey)

    def getToken(self, content):
        return base64.urlsafe_b64encode(self.encryptor.encrypt(content))

    def getContent(self, token):
        return self.encryptor.decrypt(base64.urlsafe_b64decode(token))

class LoginException(Exception):
    def __init__(self, msg):
        self.msg = msg

    def getWithDict(self):
        return {'error': 'LoginException', 'message': self.msg}

    def __str__(self):
        return self.msg

class CreateMailException(Exception):
    def __init__(self, msg):
        self.msg = msg

    def getWithDict(self):
        return {'error': 'CreateMailException', 'message': self.msg}

    def __str__(self):
        return self.msg

#@csrf_exempt
#@jsonpCallback(noPadding=True)
#def checkUserExist(request):
#    """
#    Check user exist.
#
#    test:
#    from urllib import urlencode, urlopen
#    data = urlencode({'user': 'admin', 'pwd': 'admin'})
#    print urlopen('host:port/newsletters/api/user/exist/', data=data).read()
#
#    @param request Request instance.
#    @return JSON type, example: {"token": "XNO9rHkIzWnyaPZ5a1Y_MM3KI623rtk="}
#    """
#    try:
#        if not request.method == 'POST':
#            raise RuntimeError('You have to use POST method.')
#
#        username = request.POST.get('user', '').strip()
#        password = request.POST.get('pwd', '').strip()
#
#        if username and password:
#            user = authenticate(username=username, password=password)
#            if user and user.is_active and user.is_staff:
#                return {'exist': True}
#        else:
#            raise LoginException("The user or password were incorrect.")
#    except LoginException as e:
#        return e.getWithDict()
#    except Exception as e:
#        _e = LoginException(str(e))
#        return _e.getWithDict()

@csrf_exempt
@jsonpCallback(noPadding=True)
def login(request):
    """
    Login app.

    test:
    import urllib
    data = urllib.urlencode({'user': 'admin', 'pwd': 'admin', 'app': 'hgfdd'})
    print urllib.urlopen('localhost:8000/newsletters/
                          api/app/login/', data=data).read()

    @param request Request instance.
    @return JSON type, example: {"token": "XNO9rHkIzWnyaPZ5a1Y|MM3KI623rtk="}
    """
    try:
        if not request.method == 'POST':
            raise RuntimeError('You have to use POST method.')

        username = request.POST.get('user', '').strip()
        password = request.POST.get('pwd', '').strip()
        appName = request.POST.get('app', '').strip()

        if username and password and appName:
            user = authenticate(username=username, password=password)
            if user is not None:
                if user.is_active and user.is_staff:
                    app = integrationSrv.getApp(name=appName, user=user,
                                                errorOnNotFound=False)
                    if not app:
                        # Create app by gave app name.
                        app = integrationSrv.createApp(name=appName, user=user)

                    # 1.
                    generator = TokenGenerator(username)
                    t = datetime.now().strftime('%Y%m%d%H%M%S')
                    token = generator.getToken('%s|%d' % (t, app.id))
                    token = '%s|%s' % (username, token)
                    # 2.
                    generator = TokenGenerator(SECRET_KEY)
                    token = generator.getToken(token)
                    return {'token': token}
                else:
                    raise LoginException("The password is valid, but the "
                                         "account has been disabled!")
            else:
                raise LoginException("The user or password were incorrect.")
        else:
            raise LoginException("Invalid user, password or app.")
    except LoginException as e:
        return e.getWithDict()
    except Exception as e:
        _e = LoginException(str(e))
        return _e.getWithDict()

def getAppId(token):
    """
    Get app id from token

    @param token Token.
    """
    try:
        # 1.
        generator = TokenGenerator(SECRET_KEY)
        token = generator.getContent(str(token))
        # 2.
        _token = token.split('|')
        username = _token.pop(0)
        generator = TokenGenerator(username)
        content = generator.getContent('|'.join(_token))

        _content = content.split('|')
        appId = _content[-1]
        return int(appId)
    except Exception as e:
        raise CreateMailException(str(e))

def sendMail(newsletter):
    mailer = CloudMailer(
        newsletter, fakeMail=ENABLE_FAKE_MAIL, fakeSMTP=ENABLE_FAKE_SMTP)
    result = mailer.run()
    return {'id': newsletter.id, 'allResults': result}

@csrf_exempt
@jsonpCallback(noPadding=True)
def mail(request):
    """
    Send mail.

    @param request Request instance.
    """
    try:
        logger.debug('[createMail]_START:')
        if not request.method == 'POST':
            raise RuntimeError('You have to use POST method.')

        requestData = request.POST

        token = requestData.get('token', '').strip()
        groupName = requestData.get('group', '').strip()
        emails = requestData.get('emails', '')
        title = requestData.get('title', '').strip()
        mailContent = requestData.get('text', '')

        if not token:
            raise CreateMailException("Invalid token.")

        if not groupName:
            raise CreateMailException("Need a group.")

        emails = [email.strip() for email in emails.split(',') if email.strip()]
        if not emails:
            raise CreateMailException("Need email address.")

        if not title:
            raise CreateMailException("Mail need a title.")

        headerSender = requestData.get('sender', '').strip()
        useCache = True
        # Check header.
        if headerSender:
            _email = filterEmail(headerSender)
            if useCache and not checkSenderByCache(_email):
                _result = doVerifyEmailAddress(_email)
                _verification = _result['verification']
                if _verification:
                    # Check notification topics.
                    checkResult = checkNotificationTopic(_email,
                                                         useCache=useCache)
                    if not checkResult['result']:
                        # Set notification topics.
                        _setResult = doSetNotificationTopic(_email,
                                                            useCache=useCache)
                        if not _setResult['result']:
                            raise RuntimeError(_setResult['message'])
                else:
                    raise RuntimeError(_result['message'])

        headerReply = requestData.get('reply', '').strip()

        if not headerReply:
            headerReply = headerSender

        # Get app.
        appId = getAppId(token)
        app = integrationSrv.getApp(id=appId)
        group = integrationSrv.getGroupByMappingTable(app, groupName)
        app.groups.add(group)

        sendMode = requestData.get('mode', 'sync')
        if sendMode == 'sync':
            async = False
        else:
            async = True

        # Attachments.
        try:
            attachmentIds = []
            for _key in requestData.keys():
                if _key.startswith('attachmentName'):
                    aId = _key.replace('attachmentName', '')
                    if aId not in attachmentIds:
                        attachmentIds.append(aId)

            attachments = []
            attachDict = {}

            _postData = parse_qsl(request.body, True)
            for (_k, _d) in _postData:
                _k = unquote(_k)
                if (_k.startswith('attachmentContent')) or (
                    _k.startswith('attachmentName')) or (
                    _k.startswith('attachmentType')):
                    _d = unquote(_d)
                    attachDict[_k] = _d

            for aId in attachmentIds:
                nameKey = 'attachmentName%s' % aId
                contentKey = 'attachmentContent%s' % aId
                mimeTypeKey = 'attachmentType%s' % aId

                name = attachDict.get(nameKey, None)
                if name:
                    name = _unicode(name, strict=True)
                else:
                    name = 'noname'

                content = attachDict.get(contentKey, '')
                mimeType = attachDict.get(mimeTypeKey, '')

                attachments.append({
                    'name': name,
                    'content': content,
                    'mimeType': mimeType,
                })

        except Exception as e:
            raise e

        newsletter = integrationSrv.createMail(
            group=group, title=title, emails=emails, text=mailContent,
            headerSender=headerSender, headerReply=headerReply,
            attachments=attachments, async=async)
        group.newsletters.add(newsletter)

        logger.debug('[createMail]_SUSSESS: newsletter<%d>' % newsletter.id)

        if async:
            return {'id': newsletter.id, 'status': 'Waiting for send.'}
        else:
            return sendMail(newsletter)
    except CreateMailException as e:
        logger.debug('[createMail]_FAILED: %s' % getComment(e))
        return e.getWithDict()
    except Exception as e:
        logger.debug('[createMail]_FAILED: %s' % getComment(e))
        _e = CreateMailException(_unicode(e))
        return _e.getWithDict()
    finally:
        logger.debug('[createMail]_END:')

@staff_member_required
@view('GroupStatistics.html')
def displayStatistics(request, groupId):
    """
    Show statistics information.

    @param request Request instance.
    @param groupId Group id.
    """
    try:
        group = integrationSrv.getGroup(id=groupId)
        return {'group': group}
    except:
        raise Http404()

@jsonpCallback()
def getTotalData(request, groupId):
    """
    Get total data of statistics.

    @param request Request instance.
    @param groupId Group id.
    """
    group = integrationSrv.getGroup(id=groupId)

    subscriberCount = 0
    testSubscriberCount = 0
    newsletterIds = []
    for newsletter in group.newsletters.all():
        newsletterIds.append(newsletter.id)
        subscriberCount += newsletter.mailing_list.subscribers_count()

        testSubscriberCount += newsletter.mailing_list.subscribers.filter(
                               tester=True).count()

    conditions = {'newsletter__id__in': newsletterIds}
    response = getStatisticsTotalData(
        request.GET, subscriberCount, 
        testSubscriberCount, conditions)

    return response

@jsonpCallback()
def getTotalRateData(request, groupId):
    """
    Get total rate data of statistics.

    @param request Request instance.
    @param groupId Group id.
    """
    group = integrationSrv.getGroup(id=groupId)

    subscriberCount = 0
    testSubscriberCount = 0
    newsletterIds = []
    for newsletter in group.newsletters.all():
        newsletterIds.append(newsletter.id)
        subscriberCount += newsletter.mailing_list.subscribers_count()
        testSubscriberCount += newsletter.mailing_list.subscribers.filter(
            tester=True).count()
    conditions = {'newsletter__id__in': newsletterIds}
    response = getStatisticsTotalRateData(
        request.GET, subscriberCount, testSubscriberCount, conditions)

    return response

@staff_member_required
@view('GroupSubscribers.html')
def displaySubscribers(request, groupId):
    """
    Get statistics of subscriber's every rates.

    @param request Request instance.
    @param groupId Group id.
    """
    group = integrationSrv.getGroup(id=groupId)
    _status = dict(ContactMailingStatus.STATUS_CHOICES)
    return {
        'group': group,
        'statusNames': _status.values(),
        'statusCode': _status.keys(),
    }

@jsonpCallback()
def getHistory(request, groupId):
    """
    Get history of newsletter.

    @param request Request instance.
    @param groupId Group id.
    """
    group = integrationSrv.getGroup(id=groupId)
    newsletterIds = group.newsletters.values('id')
    newsletterIds = [i['id'] for i in newsletterIds]
    conditions = {'newsletter__id__in': newsletterIds}
    status = request.GET.get('status', '')
    if status:
        conditions.update({'status': int(status)})

    keyword = request.GET.get('keyword', '')
    if keyword:
        conditions.update({'contact__email__icontains': keyword})

    response = getSubscriberHistory(request.GET, conditions)
    return response

@jsonpCallback(noPadding=True)
def getGroups(request):
    app = request.GET.get('app', None)
    if app:
        app = integrationSrv.getApp(id=app)
        groups = app.groups.all()
        groups = [{'id': group.id, 'name': group.name} for group in groups]
        return {'groups': groups}
    return {'error': 'Need a app.'}
