#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Mailer.py 11004 2018-05-01 01:15:42Z Kevin $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Kevin $ (Last)
# $Date: 2013-01-30 12:13:45 +0800$
# $Revision: 11004 $

import os
import re
import random
import mimetypes
import time
import threading

from StringIO import StringIO
from datetime import datetime
from smtplib import SMTPRecipientsRefused

from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText

from django.core.mail import EmailMessage, EmailMultiAlternatives
from django.utils.encoding import smart_str, smart_unicode
from django.contrib.sites.models import Site
from django.template import Context
from django.template.loader import render_to_string
from django.conf import settings
from django.core.urlresolvers import reverse

from emencia.django.newsletter.mailer import Mailer
from emencia.django.newsletter.utils.tokens import tokenize
from emencia.django.newsletter.utils.newsletter import body_insertion
from emencia.django.newsletter.settings import TRACKING_LINKS
from emencia.django.newsletter.settings import TRACKING_IMAGE
from emencia.django.newsletter.settings import INCLUDE_UNSUBSCRIPTION

from Iuppiter.dispatch import makeRequests, NoResultsPending, NoWorkersAvailable
from Iuppiter.Logging import createLogger
from Iuppiter.Encoding import utf8

from Iuno.cloud.mail import EmailBackend
from Iuno.cloud.mail import SendMessageException, RetryAndSendException

from emencia.django.newsletter.models import ContactMailingStatus, Newsletter
from html2text import html2text as html2text_orig

from Theophrastus.cloud.models import ErrorMessage
from Theophrastus.cloud.service import CloudService
from Theophrastus.Utility import getComment, trackLinks
from Theophrastus.newsletter_extension.views import getHeaderContent, \
                                                    getFooterContent

cloudSrv = CloudService.getInstance()

LINK_RE = re.compile(r"https?://([^ \n]+\n)+[^ \n]+", re.MULTILINE)

def html2text(html):
    """Use html2text but repair newlines cutting urls.
    Need to use this hack until
    https://github.com/aaronsw/html2text/issues/#issue/7 is not fixed"""
    txt = html2text_orig(html)
    links = list(LINK_RE.finditer(txt))
    out = StringIO()
    pos = 0
    for l in links:
        out.write(txt[pos:l.start()])
        out.write(l.group().replace('\n', ''))
        pos = l.end()
    out.write(txt[pos:])
    return out.getvalue()

class CloudSMTP(object):
    def __init__(self, newsletter, test=False, fakeTest=False):
        """
        Constructor.

        @param newsletter Newsletter instance.
        """
        self.newsletter = newsletter
        self.server = self.newsletter.server
        self.test = test
        self.fakeTest = fakeTest

    def sendMail(self, contact, subject, fromEmail, replyTo, content,
                 attachments=None):
        """
        Send mail.
        @param contact Contact instance.
        @param subject Subject.
        @param fromEmail From email.
        @param replyTo Reply to someone.
        @param content Content.
        @param attachments Attachments.
        """
        mail = None
        try:
            backend = EmailBackend(fallback=False)
            
            contentHtml = content
            contentText = html2text(contentHtml)

            mail = EmailMultiAlternatives(
                subject, contentText, from_email=fromEmail, to=[contact.email],
                connection=backend, headers={'Reply-To': replyTo})
            mail.attach_alternative(contentHtml, "text/html")

            if attachments:
                for (title, c, maintype) in attachments:
                    mail.attach(utf8(title), c, maintype)

            mail.content_subtype = "mixed"
            if self.fakeTest:
                time.sleep(3)
            else:
                mail.send(fail_silently=False)

                extra = mail.extra_headers
                messageId = extra.get('message_id', '')
                requestId = extra.get('request_id', '')
                message = extra.get('message', '')

                cloudSrv.recordSendInformation(
                    self.newsletter, contact, messageId, message)

            if not self.test:
                self.newsletter.status = Newsletter.SENT
                self.newsletter.save()

            return True
        except Exception as e:
            raise e

    def quit(self):
        return

class CloudMailer(Mailer):
    def __init__(self, newsletter, pool=None, test=False, fakeMail=False,
                 fakeSMTP=False, logger=None):
        """
        Constructor.
        @param newsletter Newsletter instance.
        @param test Is test?
        """
        super(CloudMailer, self).__init__(newsletter, test=test)

        # If newsletter's SMTP server was cloud(default: id=1)
        self.useCloud = True if self.newsletter.server.id == 1 else False

        self.pool = pool

        # Create logger.
        if not logger:
            self.logger = createLogger('theophrastusMailer', 'mailer.log')
        else:
            self.logger = logger

        self.fakeMail = fakeMail
        if self.fakeMail:
            self.logger.info(
                '[%d]__init__: Open fake send mode.' % newsletter.id)

        self.fakeSMTP = fakeSMTP
        if self.fakeSMTP:
            self.logger.info(
                '[%d]__init__: Open fake SMTP mode.' % newsletter.id)

        self.smtpLock = threading.Lock()

    def _getFileContent(self, path):
        """
        Get file content from path.

        @param path File's path.
        """
        with open(path, 'rb') as f:
            return f.read()

    def getAttachments(self):
        """
        Get attachments.
        """
        attachments = []
        for attachment in self.newsletter.attachment_set.all():
            filePath = attachment.file_attachment.path
            attachContent = self._getFileContent(filePath)

            if attachment.contentType:
                ctype = attachment.contentType
            else:
                ctype, encoding = mimetypes.guess_type(filePath)
                if ctype is None or encoding is not None:
                    ctype = 'application/octet-stream'

            root, ext = os.path.splitext(filePath)
            attachments.append(
                ('%s%s' % (attachment.title, ext), attachContent, ctype))

        return attachments

    @property
    def can_send(self):
        """
        Check if the newsletter can be sent
        """
        if self.newsletter.server.credits() <= 0:
            self.logger.debug(
                '[%d]_[can_send]_CANNOT: Server credits <= 0.' %
                self.newsletter.id)
            return False

        if self.test:
            return True

        if self.newsletter.sending_date <= datetime.now() and \
               (self.newsletter.status == Newsletter.WAITING):
            return True

        self.logger.info(
            ('[%d]_[can_send]_CANNOT: Newsletter(status: %d) can not send:'
             'sending_date: %s, datetime.now: %s (TZ: %s).') %
            (self.newsletter.id, self.newsletter.status,
             self.newsletter.sending_date, datetime.now(),
             os.environ.get('TZ', 'None')))

        return False

    def testSend(self, contact, wait=5):
        """
        Test send.
        We can't sent any mail only wait few seconds.

        @param contact Contact instance.
        @param wait Sleeping time, default is five seconds.
        """
        newsletterId = self.newsletter.id
        contactId = contact.id
        _s = 'Send a test mail to contact<%d>' % contactId
        try:
            if random.randint(1, 100) > 15:
                time.sleep(wait)
                # Change newsletter's status.
                self.newsletter.status = Newsletter.SENT
                self.newsletter.save()
                s = '%s success.' % _s
                self.logger.info('[%d]_TEST_SUCCESS: %s.' % (newsletterId, s))
                return True
            else:
                raise RuntimeError('Failed...')
        except Exception as e:
            s = '%s failed.' % _s
            self.logger.error('[%d]_TEST_FAILED: %s.' % (newsletterId, s))
            raise e
        finally:
            s = '%s end.' % _s
            self.logger.info('[%d]_TEST_END: %s' % (newsletterId, s))

    def buildEmailContent(self, contact):
        """
        Generate the mail content for a contact.

        @param contact Contact instance.
        """
        uidb36, token = tokenize(contact)
        context = Context({'contact': contact,
                           'domain': Site.objects.get_current().domain,
                           'newsletter': self.newsletter,
                           'uidb36': uidb36, 'token': token})

        content = self.newsletter_template.render(context)
        if TRACKING_LINKS:
            content = trackLinks(content, context)

        headerContent = getHeaderContent(self.newsletter)
        if headerContent is None:
            linkSite = render_to_string(
                'newsletter/newsletter_link_site.html', context)
        else:
            linkSite = headerContent.render(context)
        content = body_insertion(content, linkSite)

        if INCLUDE_UNSUBSCRIPTION:
            footerContent = getFooterContent(self.newsletter)
            if footerContent is None:
                unsubscription = render_to_string(
                    'newsletter/newsletter_link_unsubscribe.html', context)
            else:
                unsubscription = footerContent.render(context)
            content = body_insertion(content, unsubscription, end=True)

        if TRACKING_IMAGE:
            image_tracking = render_to_string(
                'newsletter/newsletter_image_tracking.html', context)
            content = body_insertion(content, image_tracking, end=True)
        
        return smart_unicode(content)

    def send(self, contact):
        """
        Send mail to contact.

        @param contact Contact instance.
        """
        result = False
        resultMessage = ''
        newsletterId = self.newsletter.id
        contactId = contact.id

        try:
            self.logger.info(
                '[%d]_START: Send mail to contact<%d>.' %
                (newsletterId, contactId))
            if self.fakeMail:
                result = self.testSend(contact)
                resultMessage = 'Send test mail success.'
            elif self.useCloud:
                subject = self.build_title_content(contact)
                replyTo = smart_str(self.newsletter.header_reply)

                contentHtml = self.buildEmailContent(contact)

                result = self.smtp.sendMail(
                    contact,
                    subject,
                    smart_str(self.newsletter.header_sender),
                    replyTo,
                    contentHtml,
                    attachments=self.attachments)
                resultMessage = 'Send mail success.'
                
            else:
                message = self.build_message(contact)
                
                with self.smtpLock:    
                    self.smtp.sendmail(
                        smart_str(self.newsletter.header_sender),
                        contact.email,
                        message.as_string()
                    )

                result = True
                resultMessage = 'Send mail success.'

            status = self.test and ContactMailingStatus.SENT_TEST \
                     or ContactMailingStatus.SENT
            self.logger.info(
                '[%d]_SUCCESS: Send mail to contact<%d> success.' %
                (newsletterId, contactId))

        except SMTPRecipientsRefused as e:
            status = ContactMailingStatus.INVALID
            contact.valid = False
            contact.save()
            _eMsg = str(e)
            resultMessage = 'Send mail failed: %s.' % _eMsg
            self.logger.error(
                '[%d]_FAILED: Send mail to contact<%d> failed, %s.' %
                (newsletterId, contactId, _eMsg))

            # Record error message  to db.
            ErrorMessage.objects.create(
                message=(resultMessage + _eMsg), 
                contact=contact,
                newsletter=self.newsletter
            )

        except Exception as e:
            _eMsg = getComment(e)
            resultMessage = 'Send mail failed: %s.' % _eMsg
            errorMessage = (
                '[%d]_FAILED: Send mail to contact<%d> failed,\n %s.' %
                (newsletterId, contactId, _eMsg)
            )
            self.logger.error(errorMessage)

            if (isinstance(e, RetryAndSendException) or
                isinstance(e, SendMessageException)):
                try:
                    errorCode = e.exception.error_code
                    errorMessage = e.exception.error_message
                    requestId = e.exception.request_id

                    cloudSrv.recordSendInformation(
                        self.newsletter, contact, requestId, errorMessage)

                    if 'rejected' in errorCode.lower():
                        status = ContactMailingStatus.REJECTED
                    else:
                        status = ContactMailingStatus.ERROR
                except Exception as e:
                    self.logger.error(
                        '[%d]_GET_ERROR_INFO_FAILED: contact<%d>.' %
                        (newsletterId, contactId)
                    )
                    status = ContactMailingStatus.ERROR

            else:
                status = ContactMailingStatus.ERROR

            # Record error message  to db.
            ErrorMessage.objects.create(
                message=(resultMessage + errorMessage), 
                contact=contact,
                newsletter=self.newsletter
            )

        finally:
            self.logger.info(
                '[%d]_END: Send mail to contact<%d> end.' %
                (newsletterId, contactId))

        # FIXME # If no wait, it might be no record contactmailingstatus.
        # time.sleep(3)
        ContactMailingStatus.objects.create(
            newsletter=self.newsletter, contact=contact, status=status
        )

        return {'result': result , 'message': resultMessage, 'id': contactId}

    def run(self):
        """
        Override.
        Send the mails

        @param pool ThreadPool instance.
        """
        self.logger.info('doRun: %d' % self.newsletter.id)

        newsletterId = self.newsletter.id

        if not self.can_send:
            self.logger.info(
            '[%d][run]_PASS: Newsletter cannot send.' % newsletterId)
            return

        # If you want pass multiple arguments, you need give a list, this list
        # should include a tuple, like: (args, kwargs)
        # argsList = []
        # for _c in self.expedition_list:
        #     argsList.append(([_c, self.newsletter], {}))
        try:
            self.update_newsletter_status()

            if not self.smtp:
                self.smtp_connect()

            if self.useCloud:
                self.attachments = self.getAttachments()
            else:
                self.attachments = self.build_attachments()

            if self.pool:
                self.logger.info(
                    '[%d][run]_START: Preparing to put worker to pool.' %
                    newsletterId)
                # Put worker to pool.
                for req in makeRequests(self.send, self.expedition_list):
                    self.pool.putRequest(req)

                # FIXME: We should override method "poll", set timeout value for
                # get value from pool's queue.
                self.pool.poll()
                self.logger.info(
                    '[%d][run]_SUCCESS: Put worker to pool success.' %
                    newsletterId)
            else:
                self.logger.info(
                    '[%d][run]_START: Preparing to send mails.' % newsletterId)

                resposne = []
                for contact in self.expedition_list:
                    resposne.append(self.send(contact))

                self.logger.info(
                    '[%d][run]_SUCCESS: Send mails success.' % newsletterId)
                return resposne

        except Exception as e:
            if self.pool:
                _s = 'Put worker to pool failed'
            else:
                _s = 'Send mail failed'

            s = ('[%d][run]_FAILED: %s,' % (newsletterId, _s))
            if isinstance(e, NoResultsPending):
                self.logger.error('%s reason: NoResultsPending, %s.' % 
                                 (s, str(e)))
            elif isinstance(e, NoWorkersAvailable):
                self.logger.error('%s reason: NoWorkersAvailable, %s.' % 
                                 (s, str(e)))
            else:
                self.logger.error('%s reason: %s.' % (s, getComment(e)))
        finally:
                       
            if self.smtp:
                if self.pool:
                    self.pool.wait()
                self.smtp.quit()
                
            self.update_newsletter_status()                

            if self.pool:
                _s = 'Put worker to pool over.'
            else:
                _s = 'Send mail over.'
            self.logger.info(
                '[%d][run]_END: %s' % (newsletterId, _s))

    def smtp_connect(self):
        """
        Override.
        Make a connection to the SMTP
        """
        if self.useCloud:
            self.smtp = CloudSMTP(self.newsletter, test=self.test,
                                  fakeTest=self.fakeSMTP)
        else:
            self.smtp = self.newsletter.server.connect()
