#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: models.py 10694 2017-12-29 09:22:09Z David $
#
# Copyright (c) 2013 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: David $
# $Date: 2017-12-29 17:22:09 +0800 (Fri, 29 Dec 2017) $
# $Revision: 10694 $

from django.db import models
from django.contrib.auth.models import User
from django.utils.translation import ugettext as _

from Iuppiter.Model import BaseModelMixin

from Theophrastus.models import Newsletter


class Group(models.Model, BaseModelMixin):
    name = models.CharField(_('name'), max_length=255)

    newsletters = models.ManyToManyField(
        Newsletter, verbose_name=_('Newsletters'), blank=True, null=True)

    createdTime = models.DateTimeField(
        auto_now_add=True, verbose_name=_('Created time'))

    @property
    def canCountNewsletter(self):
        for newsletter in self.newsletters.all():
            if (newsletter.status == Newsletter.SENDING or
                newsletter.status == Newsletter.SENT):
                return True
        return False

    @models.permalink
    def statisticsUrl(self):
        return ('Theophrastus.integration.views.displayStatistics', (self.id,))

    @models.permalink
    def subscriberUrl(self):
        return ('Theophrastus.integration.views.displaySubscribers', (self.id,))

    def getStatisticsLink(self):
        """
        Display link for statistics(override).
        """
        """Display link for statistics"""
        if self.canCountNewsletter:
            return '<a href="%s">%s</a>' % (self.statisticsUrl(),
                                            _('View statistics'))
        else:
            return _('Not available')

    def getSubscriberLink(self):
        """
        Display link for subscribers.
        """
        if self.canCountNewsletter:
            return '<a href="%s">%s</a>' % (self.subscriberUrl(),
                                            _('View subscribers'))
        else:
            return _('Not available')

    def __unicode__(self):
        return self.name

    class Meta:
        verbose_name = _('group')
        verbose_name_plural = _('groups')

class App(models.Model, BaseModelMixin):
    name = models.CharField(max_length=255, verbose_name=_('Name'))
    user = models.ForeignKey(User, verbose_name=_('User'))
    groups = models.ManyToManyField(Group, related_name='groups', null=True)
    createdTime = models.DateTimeField(auto_now_add=True,
                                       verbose_name=_('Created time'))

    def __unicode__(self):
        return self.name

    class Meta:
        verbose_name = _('app')
        verbose_name_plural = _('apps')

class GroupMappingTable(models.Model, BaseModelMixin):

    KEYWORD = 0
    REGULAR_EXPRESSION = 1
    CONDITION_TYPES_CHOICES = (
        (KEYWORD, _('Keyword')),
        (REGULAR_EXPRESSION, _('Regular expression')),
    )

    app = models.ForeignKey(
        App, verbose_name=_('App'), help_text=_('Please choice a app.'))

    conditionType = models.IntegerField(
        choices=CONDITION_TYPES_CHOICES, default=0,
        verbose_name=_('Condition type'),
        help_text=_('Please choice a condtion type, default is keyword.'))

    condition = models.TextField(max_length=1024,
                                 verbose_name=_('Regular expression'))

    group = models.CharField(
        max_length=255, verbose_name=_('Group name'), blank=True, null=True)

    current = models.ForeignKey(Group, verbose_name=_('Current Group'),
                                null=True)

    def __unicode__(self):
        _to = self.current if self.current else self.group
        return '%s: %s %s' % (self.app.name , _('mapping'), _to)

    class Meta:
        verbose_name = _('group mapping table')
        verbose_name_plural = _('group mapping tables')
