#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: setup.py 10199 2017-08-20 02:22:54Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License, 
# you may not use this file except in compliance with the License. 
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and 
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2017-08-20 10:22:54 +0800 (週日, 20 八月 2017) $
# $Revision: 10199 $ 

import os
import sys
import logging
import shutil
import argparse

import django

from django.db import transaction

SETUP_DIR = os.path.dirname(os.path.abspath(__file__))

logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)
handler = logging.StreamHandler()
handler.setFormatter(
    logging.Formatter("%(asctime)s - %(levelname)s - %(message)s"))
logger.addHandler(handler)

def copyToPath(baseDir, templateDir, staticDir, mode):
    logger.info("-----Start copy templates-----")
    for root, dirs, files in os.walk(os.path.join(SETUP_DIR, "static")):
        for d in dirs:
            index = os.path.join(root, d).index("static") + len("static/")
            path = os.path.join(
                staticDir, os.path.join(os.path.join(root, d))[index:])
            if not os.path.exists(path):
                os.mkdir(path)
                
    if not os.path.exists(os.path.join(baseDir, "fixtures")):
        os.mkdir(os.path.join(baseDir, "fixtures"))
    
    for root, dirs, files in os.walk(SETUP_DIR):
        for f in files:
            if f == 'setup.py':
                continue
            origin = os.path.join(root, f)
            
            if "static" in root:
                index = len(os.path.join(SETUP_DIR, "static")) + 1
                dst = os.path.join(staticDir, origin[index:])
            elif "templates" in root:
                result = os.path.splitext(origin)[0]
                if mode == 'programmer':
                    if result.endswith("_user"):
                        continue
                    dst = os.path.join(templateDir, f)
                else:
                    if not result.endswith("_user"):
                        continue
                    dst = os.path.join(
                        templateDir, f.replace('_user.html', '.html'))
            elif "fixtures" in root:
                index = len(os.path.join(SETUP_DIR, "fixtures")) + 1
                dst = os.path.join(baseDir, "fixtures", origin[index:])
            else:
                dst = os.path.join(baseDir, f)
                
            
            if os.path.isfile(dst):
                logger.error("%s is exist. failed to copy file." % dst)
            else:
                shutil.copyfile(origin, dst)
                logger.info("Succeed to copy file to %s." % dst)
    

def findSettings(path):
    for root, dirs, files in os.walk(path):
        for f in files:
            if f == 'settings.py':
                return os.path.join(root, f)
                
def writeSettings(settingsPath, templateDir):
    logger.info("-----Start write and backup settings.py-----")
    def getCMSTemplates():
        templates = []
        
        path = os.path.join(SETUP_DIR, 'templates')
        index = None
                
        for template in os.listdir(path):
            if "_user" in template:
                continue
            if template.lower() == 'index.html':
                index = template
                continue
            templates.append((template, template))
        
        if index:
            return [(index, index),] + templates
        else:
            return templates
    
    settingsDir = os.path.dirname(settingsPath)
    bak = os.path.join(settingsDir, 'settings.py.bak')
    if os.path.isfile(bak):
        logger.error("%s is exist. failed to backup file." % bak)
        return 
    shutil.copyfile(settingsPath, bak)
    logger.info("Succeed to backup settings.py to %s." % bak)
                
                
    with open(settingsPath, 'a+') as settings:
        content = \
'''
CMS_TEMPLATES = list(CMS_TEMPLATES)
CMS_TEMPLATES += %s
''' % str(getCMSTemplates())
        settings.write(content)
    logger.info("Succeed to write settings.py")
              
@transaction.atomic    
def loadData(language):
    logger.info("-----Start load data-----")
    data = os.path.join(".", 'fixtures', 'initial_data.json')
    os.system("%s manage.py loaddata %s" % (sys.executable, data))
    
    from cms.models.pagemodel import Page
    from cms.models.titlemodels import Title
    from cms.api import copy_plugins_to_language
    
    for page in Page.objects.all():
        copy_plugins_to_language(page, "zh-hant", language)
    for title in Title.objects.all():
        title.language = language
        title.save()
    
        

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Path to Project')
    parser.add_argument('path', metavar='path', type=str, help='path')
    parser.add_argument(
        '-m', '--mode', 
        default='programmer',
        choices=['programmer', 'user',],
        help='programmer mode or user mode.')
    
    args = parser.parse_args()
    
    if args.mode == 'programmer':
        mode = 'programmer'
    else:
        mode = 'user'
    
    settingsPath = findSettings(args.path)
    
    settingModule = os.path.splitext(settingsPath[len(args.path) + 1:])[0]
    settingModule = settingModule.replace("/", ".")
    settingModule = settingModule.replace("\\", ".")
    
    sys.path.append(args.path)
    os.environ['DJANGO_SETTINGS_MODULE'] = settingModule
    django.setup()
    
    from django.conf import settings
    os.chdir(args.path)
    
    templateDir = settings.TEMPLATES[0]['DIRS'][0]
    staticDir = settings.STATIC_ROOT
    
    if hasattr(settings, "STATICFILES_DIRS"):
        staticDir = settings.STATICFILES_DIRS[0]
    
    copyToPath(args.path, templateDir, staticDir, mode)
    
    writeSettings(settingsPath, templateDir)
    loadData(settings.LANGUAGE_CODE)
    
    