#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Service.py 11262 2018-11-16 06:29:44Z Kevin $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Kevin $ (last)
# $Date: 2018-11-16 14:29:44 +0800 (Fri, 16 Nov 2018) $
# $Revision: 11262 $

import os
import httplib2

from apiclient import discovery
from oauth2client import client, tools
from oauth2client.file import Storage

from googleapiclient.errors import HttpError

SCRIPT_DIR = os.path.dirname(os.path.realpath(__file__))
CLOUD_STORAGE_GOOGLE_SECRET = os.path.join(SCRIPT_DIR, "client_secret.json")
CLOUD_STORAGE_GOOGLE_SCOPE = "https://www.googleapis.com/auth/drive",
CLOUD_STORAGE_GOOGLE_APP_NAME = "Drive API Python Quickstart"

class ServiceSingleton:
    __single = None

    def __init__(self):
        if ServiceSingleton.__single:
            raise ServiceSingleton.__single
        else:
            credentials = self.getCredentials(
                CLOUD_STORAGE_GOOGLE_SECRET,
                CLOUD_STORAGE_GOOGLE_SCOPE,
                CLOUD_STORAGE_GOOGLE_APP_NAME
            )
            self.service = discovery.build(
                "drive", "v3", http=credentials.authorize(httplib2.Http()))
        ServiceSingleton.__single = self

    @staticmethod
    def getSingleton():
        if not ServiceSingleton.__single:
            ServiceSingleton.__single = ServiceSingleton()
        return ServiceSingleton.__single

    def getCredentials(self, clientSecretFile=None, scope=None,
                       appName=None, flags=None):
        # get or create .credentials in app dir
        credentialDir = os.path.join(SCRIPT_DIR, ".credentials")
        os.makedirs(credentialDir, exist_ok=True)

        # get credential
        credentialPath = os.path.join(credentialDir,
                                      "drive-python-quickstart.json")
        store = Storage(credentialPath)
        credentials = store.get()

        # create credential if not exist
        if not credentials or credentials.invalid:
            if all(param is not None for param in (
                    clientSecretFile, scope, appName)):
                flow = client.flow_from_clientsecrets(
                    clientSecretFile, scope)
                flow.user_agent = appName
                if flags:
                    credentials = tools.run_flow(flow, store, flags)
                else:
                    credentials = tools.run_flow(flow, store)
            else:
                raise ValueError("'clientSecretFile', 'scope', 'appName'"
                                 " are required for creating credentials.")

        return credentials

service = ServiceSingleton.getSingleton().service


def handleHttpError(error):
    if error.resp.status == 404:
        raise FileNotFoundError(str(e))
    else:
        raise error


def _getFileByID(fileId, fields=()):
    """
    Get file properties by file id in google drive api.

    @param fileId File's id.
    @param fields File's properties list.
    @return A dictionary contains specified properties.
    """
    try:
        # find the file with this id
        return service.files().get(
            fileId=fileId, fields=", ".join(fields)).execute()
    except HttpError as e:
        handleHttpError(e)

def _getPath(fileId):
    """
    Get file path by file id in google drive api.

    @param fileId The file's id in google drive.
    @return The file's path in google drive.
    """
    pathNames = []
    while True:
        file = _getFileByID(fileId, fields=["name", "parents"])

        if "parents" in file:
            fileId = file["parents"][0]
            pathNames.insert(0, file["name"])
        else:
            break

    return "/".join(pathNames)


def _getFileByPath(path, fields=()):
    """
    Get file properties by file path in google drive api.

    @param path The file's path.
    @param fields The file's properties list.
    @return A dictionary contains specified properties.
    """
    path = path.replace("\\", "/")
    filename = path.split("/")

    # find all the files with this name
    files = service.files().list(
        q="name='{}'".format(filename[-1]),
        fields="files({})".format(", ".join(fields))).execute()["files"]

    # check the path (avoiding the condition that
    # two different files have the same name)
    for file in files:
        if path == _getPath(file["id"]):
            return file
    else:
        raise FileNotFoundError()


BY_ID = 1
BY_PATH = 2


def getFile(key, type=BY_ID, fields=None):
    """
    Get file properties by file id or file path in google drive api.

    @param key The query key.
    @param type The query condition is by id or by path.
    @param fields The file's properties list. Available properties see
                  https://developers.google.com/drive/api/v3/reference/files
    @return A dictionary contains file's id, name and specified properties.
    """

    # init fields
    _fields = ["id", "name"]
    if fields is not None:
        _fields.extend(fields)

    if type == BY_ID:
        return _getFileByID(fileId=key, fields=_fields)
    elif type == BY_PATH:
        return _getFileByPath(path=key, fields=_fields)
    else:
        raise AttributeError()


def getFiles(key, type=BY_ID, fields=None):
    """
    Get files properties by folder id or folder path in google drive api.

    @param key The query key.
    @param type The query condition is by id or by path.
    @param fields The files' properties list. Available properties see
                  https://developers.google.com/drive/api/v3/reference/files
    @return A list contains the dictionaries
            with file's id, name and specified properties.
    """

    # init fileId
    if type == BY_ID:
        fileId = key
    elif type == BY_PATH:
        fileId = getFile(key, type=BY_PATH).get("id")
    else:
        raise AttributeError()

    # init fields
    _fields = ["id", "name"]
    if fields is not None:
        _fields.extend(fields)

    # Query all the files
    files = []
    page_token = None
    while True:
        response = service.files().list(
            q="'{}' in parents".format(fileId),
            fields="nextPageToken, files({})".format(", ".join(_fields)),
            pageToken=page_token
        ).execute()

        # save files
        files.extend(response.get("files", []))

        # refresh token
        page_token = response.get("nextPageToken", None)
        if page_token is None:
            break

    return files
