#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 9485 2015-11-11 14:24:13Z Bear $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Bear $ (Last)
# $Date: 2015-11-11 22:24:13 +0800 (週三, 11 十一月 2015) $
# $Revision: 9485 $

import os

from Iuppiter.Util import cd, run

TREEIO_REPOSITORY_URL = "https://github.com/nuwainfo/treeio"
TREEIO_BRANCH = "portal"

def setup(settingsDir):
    """
    Setup treeio to project directory and make Yggdrasil workable.
    
    @param settingsDir Project settings directory.
    """
    import warnings
    import shutil
    
    projectDir = os.path.abspath(os.path.join(settingsDir, '..'))
    treeioDir = os.path.join(projectDir, 'treeio')
    settingsModule = os.path.join(settingsDir, 'settings.py')
    treeioSettings = os.path.join(treeioDir, 'settings.py')
    treeioSettingsBak = os.path.join(treeioDir, '_settings.py')
    
    import site
    import sys
    
    site.addsitedir(treeioDir)

    if treeioDir not in sys.path:
        sys.path.insert(0, treeioDir)    
        
    if not os.path.exists(settingsModule):
        raise RuntimeError('There is no settings.py inside project directory.')
        
    if 'VIRTUAL_ENV' not in os.environ:
        raise RuntimeError('You must run inside virtualenv.')        

    def update():
        with cd(treeioDir):
        
            # Initialize settings.            
            shutil.move(treeioSettings, treeioSettingsBak)        
            shutil.copy2(settingsModule, treeioSettings)              
        
            run('pip install --upgrade -r requirements.pip', 
                shell=True, printCmd=True, printOut=True)
            
            djangoDir = run('python -c "import django; import os; '
                            'print os.path.dirname(django.__file__)"', 
                            shell=True, printCmd=True, printOut=True)
            djangoDir = djangoDir.strip()
            
            run('python related_fields_patch.py "%s"' % djangoDir, 
                shell=True, printCmd=True, printOut=True)
                
            # See issue #121. (https://github.com/treeio/treeio/issues/121)
            run('python manage.py collectstatic --noinput', 
                shell=True, printCmd=True, printOut=True)            
        
        
    if os.path.exists(treeioSettings):
        # The project is initialized, update its code.
        with cd(treeioDir):
            # We must reset settings.py.
            os.remove(treeioSettings)
        
            run("git checkout -- .", shell=True, printCmd=True, printOut=True)
            run("git pull origin %s" % TREEIO_BRANCH, 
                shell=True, printCmd=True, printOut=True)
                
        update()
        
        return
            
    with cd(projectDir):
        clone = "git clone"
        if TREEIO_BRANCH:
            clone = "%s -b %s" % (clone, TREEIO_BRANCH)
        clone = "%s %s" % (clone, TREEIO_REPOSITORY_URL)
        
        run(clone, shell=True, printCmd=True, printOut=True)

        staticDir = os.path.join(treeioDir, 'static')   
         
        if sys.platform.startswith('win'):
            run('mklink /D static "%s"' % staticDir, 
                shell=True, printCmd=True, printOut=True)
        else:
            run('ln -s "%s" static' % staticDir, 
                shell=True, printCmd=True, printOut=True)            
                    
    with cd(treeioDir):
        
        update()
                
        # python manage.py installdb
        run('python manage.py syncdb --all --noinput', 
            shell=True, printCmd=True, printOut=True)
        run('python manage.py migrate --all --fake --noinput --no-initial-data',
            shell=True, printCmd=True, printOut=True)
                
        run('python manage.py loaddata data.json', 
            shell=True, printCmd=True, printOut=True)                
        
def attachSettings(settingsDict):
    """
    Attach default settings to settings.py.
    It must be called in settings by passing locals().

    @param settingsDict locals() in settings.py.
    """
    from Iuno import require, warnWithout
       
    import imp
    
    # Load treeio's default settings module.
    # Propagate DEBUG to environment variable to make treeio's settings work.
    if 'DEBUG' not in os.environ:
        os.environ['DEBUG'] = str(settingsDict['DEBUG'])
        
    projectDir = os.path.abspath(os.path.dirname(settingsDict["__file__"]))
    f, filename, description = imp.find_module('_settings', [projectDir])
    _settings = imp.load_module('_settings', f, filename, description)
    
    _settingsDict = dict([(k, v) for k, v in _settings.__dict__.items() 
                         if not (k.startswith('__') and k.endswith('__'))])
    
    # We will attach our own database settings latter.
    _settingsDict.pop('DATABASES', None) 
    # This value should follow the SERVER_MODE.
    _settingsDict.pop('DEBUG', None) 
    _settingsDict.pop('TEMPLATE_DEBUG', None) 
    
    # Admins.
    _settingsDict.pop('ADMINS', None)
    _settingsDict.pop('MANAGERS', None)
    _settingsDict.pop('ALLOWED_HOSTS', None)
    
    # i18n.
    _settingsDict.pop('TIME_ZONE', None)
    _settingsDict.pop('LANGUAGE_CODE', None)
    
    # Secret key.
    _settingsDict.pop('SECRET_KEY', None)
    
    # Email settings should be cleared.
    require('EMAIL_SERVER', settingsDict)
    require('EMAIL_USERNAME', settingsDict)
    require('EMAIL_PASSWORD', settingsDict)
    require('EMAIL_FROM', settingsDict)
    warnWithout('DEFAULT_SIGNATURE', settingsDict, _settingsDict)
    
    _settingsDict.pop('EMAIL_SERVER', None)
    _settingsDict.pop('IMAP_SERVER', None)
    _settingsDict.pop('EMAIL_USERNAME', None)
    _settingsDict.pop('EMAIL_PASSWORD', None)
    _settingsDict.pop('EMAIL_FROM', None)        
    _settingsDict.pop('DEFAULT_SIGNATURE', None)       
        
    settingsDict.update(_settingsDict)
