from django.shortcuts import redirect
from django.contrib import messages
from django.contrib.auth import login as auth_login
from django.contrib.auth.models import Permission
from oscar.core.loading import get_classes, get_model
from oscar.apps.customer import signals


(_AccountRegistrationView,
 _AccountAuthView,
) = get_classes('customer.views',(
        'AccountRegistrationView',
        'AccountAuthView',
    )
)

Partner = get_model('partner', 'Partner')

def linkUser(user, partner):
    if not partner.users.filter(pk=user.pk).exists():
        partner.users.add(user)
        
    if not user.is_staff:
        perm = Permission.objects.get(
            codename='dashboard_access',
            content_type__app_label='partner')
        user.user_permissions.add(perm)
   

def createPartner(user):
    # create partner
    partner, created = Partner.objects.get_or_create(
        name=user.email
    )
    # add permission
    linkUser(user, partner)

class AccountRegistrationView(_AccountRegistrationView):
    
    def form_valid(self, form):
        user = self.register_user(form)
        createPartner(user)
        return redirect(form.cleaned_data['redirect_url'])
        
class AccountAuthView(_AccountAuthView):

    def validate_login_form(self):
        form = self.get_login_form(bind_data=True)
        if form.is_valid():
            user = form.get_user()
            createPartner(user)
            # Grab a reference to the session ID before logging in
            old_session_key = self.request.session.session_key

            auth_login(self.request, form.get_user())

            # Raise signal robustly (we don't want exceptions to crash the
            # request handling). We use a custom signal as we want to track the
            # session key before calling login (which cycles the session ID).
            signals.user_logged_in.send_robust(
                sender=self, request=self.request, user=user,
                old_session_key=old_session_key)

            msg = self.get_login_success_message(form)
            if msg:
                messages.success(self.request, msg)

            return redirect(self.get_login_success_url(form))

        ctx = self.get_context_data(login_form=form)
        return self.render_to_response(ctx)
    
    def validate_registration_form(self):
        form = self.get_registration_form(bind_data=True)
        if form.is_valid():
            user = self.register_user(form)
            createPartner(user)
            msg = self.get_registration_success_message(form)
            messages.success(self.request, msg)

            return redirect(self.get_registration_success_url(form))

        ctx = self.get_context_data(registration_form=form)
        return self.render_to_response(ctx)