#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Configure.py 12425 2020-07-26 06:31:20Z Bear $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Bear $ (last)
# $Date: 2020-07-26 14:31:20 +0800 (日, 26  7 2020) $
# $Revision: 12425 $
"""
Django related functions to configure django project.
"""

import os
import importlib

def importSettings():
    """
    Import settings module.

    @return Project's settings or default settings.
    """
    
    if 'DJANGO_SETTINGS_MODULE' in os.environ:
        return importlib.import_module(os.environ.get('DJANGO_SETTINGS_MODULE'))
    else:
        try:
            import settings
        except ImportError:
            from Iuppiter import IuppiterSettings as settings
        return settings

settings = importSettings()

def reconfigureDjangoSettings(additional=None, tenderly=False,
                              _settings=settings):
    """
    Reconfigure django settings.

    @param additional Additional settings as a dictionary or module.
    @param tenderly True if you want just override django current settings
                    without using project's settings.
    @param _settings Settings module.
    """
    global settings

    def module2Dict(m, d):
        for i in dir(m):
            if not i.startswith('__'):
                v = getattr(m, i)
                # Avoid cycle reference.
                if not (v is _settings):
                    d[i] = v

    # Re-configure.
    from django.conf import settings as django_settings
    if not tenderly:
        django_settings._wrapped = None
        sets = {}
        module2Dict(_settings, sets)

        if additional != None:
            if type(additional) == type(settings):
                module2Dict(additional, sets)
            else:
                sets = dict(additional, **sets)

        django_settings.configure(**sets)
        
        # We must activate timezone by ourself because Django 1.2 only activates
        # it on which settings is loaded by DJANGO_SETTINGS_MODULE environment.
        import time
        if hasattr(time, 'tzset') and django_settings.TIME_ZONE:
            os.environ['TZ'] = django_settings.TIME_ZONE
            time.tzset()
    else:
        if type(additional) == type(_settings):
            d = {}
            module2Dict(additional, d)
        else:
            d = additional

        for key, value in list(d.items()):
            setattr(django_settings, key, value)

    settings = _settings

def getSettings(noEnv=True):
    """
    Get the settings of django project. Return string 'project.settings' 
    for django environment setting.
    
    @param noEnv True if you want to always search file systems.
    @return String 'project.settings'. 
            If settings.py not exist in the project, return None.
    """

    if not noEnv and 'DJANGO_SETTINGS_MODULE' in os.environ:
        return os.environ['DJANGO_SETTINGS_MODULE']

    curdir = os.getcwd()
    
    dir = curdir
    dirName = dir.split(os.sep)[-1]
    files = os.listdir(dir)

    # If __init__.py doesn't exist in the outer dir, add __init___.py to files 
    # list, then the loop will continue. Make sure the loop can run at least 
    # once.
    
    if '__init__.py' not in files:
        files.append('__init__.py')
    
    while True:
        if 'settings.py' in files:    
            return '{project}.settings'.format(project=dirName)
        else:  
            if '__init__.py' not in files:
                break
            # search for sub directory
            _dir = dir
            subDirs = []              
            for f in files:
                if os.path.isdir(f):
                    subDirs.append(f)
            
            for sub in subDirs:
                _dir = os.path.join(dir, sub)
                _dirName = sub.replace(os.sep, '.')
                if _dir == curdir:
                    continue
                
                files = os.listdir(_dir)
                if '__init__.py' in files:
                    if 'settings.py' in files:
                        return '{project}.settings'.format(project=_dirName)
                    else:
                        for f in files:
                            if not '.' in f:
                                subDirs.append(os.path.join(sub, f))

            # search for parent directory
            dir = os.path.abspath(os.path.join(dir, os.pardir))
            dirName = dir.split(os.sep)[-1]
            files = os.listdir(dir)
            
    print('settings.py does not exist in this project.')
    