#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Logging.py 9427 2015-09-15 08:01:29Z Judy $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Judy $ (last)
# $Date: 2015-09-15 16:01:29 +0800 (二, 15  9 2015) $
# $Revision: 9427 $

import os
import logging

def createLogger(loggerName='', loggingFile=None, loggingLevel=logging.INFO,
                 formatter=None, clear=False):
    """
    Create specified logger with our default configuration for package.

    @param loggerName Logger name.
    @param loggingFile Logging file, None if you want to output to stdout.
    @param loggingLevel Logging level.
    @param formatter Logging format, None if you want to use our default.
    @param clear Clear existed logging file if True.
    @return Logger.
    """
    # Set up a specific logger with our desired output level.
    logger = logging.getLogger(loggerName)
    logger.setLevel(loggingLevel)

    created = False
    if logger.handlers:
        if loggingFile:
            absoultPath = os.path.abspath(loggingFile)
            for _handler in logger.handlers:
                if (isinstance(_handler, logging.FileHandler)) and (
                    absoultPath == _handler.baseFilename):
                    created = True
                    break
        else:
            for _handler in logger.handlers:
                if isinstance(_handler, logging.StreamHandler):
                    created = True
                    break
    if not created: # Add the log message handler to the logger.
        if loggingFile:
            handler = logging.FileHandler(loggingFile)
            if clear and os.path.exists(loggingFile):
                try:
                    os.remove(loggingFile)
                except:
                    pass
        else:
            handler = logging.StreamHandler()

        if not formatter:
            formatter = (
                "%(asctime)s - %(levelname)s - "
                "Thread: %(threadName)s(%(thread)d) - %(name)s - %(message)s")
            formatter = logging.Formatter(formatter)

        handler.setFormatter(formatter)
        logger.addHandler(handler)

    return logger

def createDebugLogger(name, fileName, formatter=None, clear=False):
    """
    Shortcut to create logger for debugging.

    @param name Logger name.
    @param fileName Logging file, None if you want to output to stdout.
    @param formatter Logging format, None if you want to use our default.
    @param clear Clear existed logging file if True.
    @return Logger.
    """
    return createLogger(name, fileName, logging.DEBUG, formatter, clear)
