#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Widgets.py 12461 2020-08-17 09:04:46Z Lavender $
#
# Copyright (c) 2013 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2020-08-17 17:04:46 +0800 (一, 17  8 2020) $
# $Revision: 12461 $

from django import forms
from django.utils.safestring import mark_safe

# ==============================================================================
# Example:
# ==============================================================================
#
#from django import forms
#
#class AppAdminForm(forms.ModelForm):
#
#    groupTables = forms.CharField(label=_('Group list'))
#
#    def __init__(self, *args, **kwargs):
#        super(AppAdminForm, self).__init__(*args, **kwargs)
#        obj = kwargs.get('instance', None)
#        if obj:
#            # Widgets.
#            self.fields['user'].widget = ReadOnlyWidget(
#                originalValue=obj.user.id,
#                displayValue=obj.user.username,
#                link=reverse('admin:auth_user_change', args=[obj.user.id])
#            )
#
#            wObjs = []
#            for g in obj.groups.all():
#                link = reverse('admin:integration_group_change', args=[g.id])
#                createdTime = g.createdTime.strftime("%Y-%m-%d %H:%M:%S")
#                wObjs.append([
#                    TableWidgetStructure(g.name, link=link),
#                    TableWidgetStructure(g.newsletters.count()),
#                    TableWidgetStructure(createdTime),
#                    TableWidgetStructure('-'),
#                    TableWidgetStructure(g.getStatisticsLink()),
#                    TableWidgetStructure(g.getSubscriberLink())
#                ])
#
#            captions = [_('title'), _('Newsletter count'), _('Created time'),
#                        _('History'), _('Statistics'), _('Subscribers')]
#            self.fields['groupTables'].widget = TableWidget(
#                wObjs, captions=captions)
#
#    class Meta:
#        model = App

class FileDragDropWidget(forms.FileInput):

    template_name = 'extension/widgets/DragDropWidget.html'


class ReadOnlyWidget(forms.Widget):

    def __init__(self, originalValue='', displayValue=None,
                 link='', attach=None):
        """
        Constructor.

        @param originalValue The original value for edit and save action.
        @param displayValue The display value.
        @param link Hyper link value.
        @param attach Give widgets what you want added.
        """
        self.originalValue = originalValue
        self.displayValue = displayValue

        self.link = link

        self.attach = attach

        super(ReadOnlyWidget, self).__init__()

    def render(self, name, value, attrs=None, renderer=None):
        """
        Override render method.
        """
        if self.displayValue is not None:
            displayName = str(self.displayValue)
        else:
            displayName = str(self.originalValue)

        if self.link:
            displayName = ('<a href="%s" target="_blank">%s</a>' %
                           (self.link, displayName))

        content = []
        if self.attach:
            for attachWidget in self.attach:
                content.append(attachWidget.render())

        return mark_safe('%s %s' % (displayName, '<br/>'.join(content)))

    def value_from_datadict(self, data, files, name):
        """
        Override value_from_datadict method.
        """
        return self.originalValue

class TableWidget(forms.Widget):

    def __init__(self, originalValues, captions=None):
        """
        Constructor.

        @param originalValue The original value for edit and save action.
        @param captions The display value.
        """
        self.originalValues = originalValues

        self.captions = captions

        super(TableWidget, self).__init__()

    def render(self, name, value, attrs=None, renderer=None):
        """
        Override render method.
        """
        output = '<table>'
        if self.captions:
            output += '<tr>'
            for c in self.captions:
                output += '<th>%s</th>' % c
            output += '</tr>'

        for _obj in self.originalValues:
            output += '<tr>'
            for _o in _obj:
                _str = ''
                if _o.link:
                    _str = '<a target="_blank" href="%s">%s</a>' % (
                           _o.link, _o.getValue())
                else:
                    _str = '%s' % _o.getValue()
                output += '<td>%s</td>' % _str
            output += '</tr>'

        output += '</table>'

        return mark_safe(output)

    def value_from_datadict(self, data, files, name):
        """
        Override value_from_datadict method.
        """
        return self.originalValues

class TableWidgetStructure(object):
    def __init__(self, value, link=None):
        self.value = value
        self._link = link

    def getValue(self):
        return self.value

    def link(self):
        return self._link
