#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: views.py 11265 2018-11-23 05:42:44Z Kevin $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Kevin $ (last)
# $Date: 2018-11-23 13:42:44 +0800 (五, 23 11 2018) $
# $Revision: 11265 $
#
# Contributors:
#  Bear (original author)
#  Jumann
#  Jaco
"""
General view functions for all apps.
"""

from django.shortcuts import render
from django.template import Context, RequestContext

def view(template):
    """
    The decorator contain template argument to save data of context instance
    then render to response.

    @param template Specify template file name.
    @return Decorated function.
    """
    def decorator(func):
        def wrap(request, *args, **kws):
            d = func(request, *args, **kws)

            if isinstance(d, dict):
                if '__processed' not in d:
                    d['__processed'] = [template]

                    return render(request, template, context=d)
            elif isinstance(d, Context):
                if '__processed' not in d:
                    d['__processed'] = [template]

                    return render(request, template, context=d)
            else:
                return d
        return wrap
    return decorator

from django.template import TemplateDoesNotExist
from django.views.generic.base import TemplateView
from django.http import Http404

def getDoc(request, name, extraContext=None, mimetype=None,
           dir='', ext='.html', **kwargs):
    """
    Simple wrapper for TemplateView to make it more generic.

    Typical use case:
        urls.py:
            ('^docs/(\w+)?/?$', getDoc, { # All optional.
                'dir': 'docs',
                'ext': '.html',
            }),
            or
            ('^docs/$', getDoc, { # All optional.
                'dir': 'docs',
                'name': 'Index',
                'ext': '.html',
            }),
            ('^docs/(?P<name>\w+)/$', getDoc, { # All optional.
                'dir': 'docs',
                'ext': '.html',
            }),
            # Match all pattern. If your dictionary has child dictionary or
            # more dictionary.
            # If you use template tag 'url' in your tempalte files, the url
            # need argument.
            # Ex: {% url getDocs 'Name' %}
            ('^docs/(.*?)/?$', getDoc, { # All optional.
                'dir': 'docs',
            }),

    @param name Template name without directory and extension.
    @param extraContext extra_context parameter for TemplateView.
    @param mimetype mimetype parameter for TemplateView.
    @param dir Template directory.
    @param ext Template extension.
    @param **kwargs Extra keyword arguments for TemplateView.
    @return Rendered template.
    """
    try:
        class T(TemplateView):
            template_name="%s%s%s" % (
                '%s/' % dir if dir else '', name if name else 'Index', ext)
            
            def get(self, request, *args, **kwargs):
                context = self.get_context_data(**kwargs)
                if extraContext:
                    context.update(extraContext)
                return self.render_to_response(context)
                        
        return T.as_view(
            content_type=mimetype, 
            **kwargs)(request)
    except TemplateDoesNotExist:
        raise Http404()

import json

from django.http import HttpResponse

def jsonpCallback(noPadding=False):
    """
    The decorator will return a JSON HTTP response, the response's argument is
    a JSON format.
    Like: callbackName(JSON data);

    @return Decorated function.
    """
    def decorator(func):
        def wrap(request, *args, **kws):
            d = func(request, *args, **kws)

            if isinstance(d, dict):
                response = json.dumps(d)
                jsonpcallback = request.GET.get('jsonpcallback', '')

                #if noPadding == False:
                response = jsonpcallback + '(' + response + ');'
                return HttpResponse(
                    response, content_type='application/json; charset=utf8')
            else:
                return d
        return wrap
    return decorator
