#!/usr/bin/env python
#! /usr/bin/python
# https://github.com/simonpainter/pyflare/blob/master/pyflare.py
# -*- coding: utf-8 -*-
# $Id: Pyflare.py 1232 2021-09-06 03:01:02Z Casey $
#
# Copyright (c) 2021 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Casey $
# $Date: 2021-09-06 11:01:02 +0800 (週一, 06 九月 2021) $
# $Revision: 1232 $

import json, os
import requests

class Cloudflare:
    def __init__(self, email, key="", token=""):
        self.endpoint = "https://api.cloudflare.com/client/v4"
        if len(token)!=0:
            bearer = str("Bearer " + token)
            self.headers = {'Authorization': bearer, 'Content-Type': 'application/json'}
        elif len(key)!=0 :
            self.headers = {'X-Auth-Email': email, 'X-Auth-Key': key, 'Content-Type': 'application/json'}

    def user(self):
        r = requests.get(self.endpoint + "/user", headers=self.headers)
        return r.json()

    def zones(self, zone):
        payload = {'name': zone}
        r = requests.get(self.endpoint + "/zones", headers=self.headers, params=payload)
        return r.json()

    def dns_records(self, zone_id, record):
        payload = {'name': record}
        r = requests.get(self.endpoint + "/zones/" + zone_id + "/dns_records", headers=self.headers, params=payload)
        return r.json()
        
    def create_record(self, zone_id, record, ip):
        payload = {
            "type": "A",
            "name": record, 
            "content": ip, 
            "ttl": 1
        }
        r = requests.post(
            self.endpoint + "/zones/" + zone_id + "/dns_records", headers=self.headers, data=json.dumps(payload))
        return r.json()

    def delete_record(self, zone_id, record_id, record, ttl, ip_address, proxied):
        payload = {'type': 'A', 'name': record,'ttl': int(ttl), 'content': ip_address, 'proxied': bool(proxied)}
        r = requests.delete(
            self.endpoint + "/zones/" + zone_id + "/dns_records/" + record_id, 
            headers=self.headers, data=json.dumps(payload))
        return r.json()

    def update_record(self, zone_id, record_id, record, ttl, ip_address, proxied):
        payload = {'type': 'A', 'name': record,'ttl': int(ttl), 'content': ip_address, 'proxied': bool(proxied)}
        r = requests.put(
            self.endpoint + "/zones/" + zone_id + "/dns_records/" + record_id, 
            headers=self.headers, data=json.dumps(payload))
        return r.json()

    def __call__(self,zone,record,ttl,proxied):
        zone_id = cf.zones(zone)['result'][0]['id']
        record_id = cf.dns_records(zone_id, record)['result'][0]['id']
        ip_address = cf.getmyip()
        if ip_address != cf.dns_records(zone_id, record)['result'][0]['content']:
            return cf.update_record(zone_id, record_id, record, ttl, ip_address, proxied)
        else:
            return "Record is up-to-date"

if __name__ == '__main__':
    __location__ = os.path.realpath(os.path.join(os.getcwd(), os.path.dirname(__file__)))
    try:
        with open(os.path.join(__location__,'config.json')) as json_data_file:
            config = json.load(json_data_file)
            for item in config['items']:
                email = item['email']
                key = item['key']
                token = item['token']
                zone = item['zone']
                record = item['record']
                ttl = item['ttl']
                proxied = item['proxied']
                cf = Cloudflare(email, key=key,token=token)
                print(cf(zone,record,ttl, proxied))
    except IOError:
        print("Unable to find config file.")
