#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: cms_plugins.py 11590 2019-05-28 18:28:03Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2019-05-29 02:28:03 +0800 (週三, 29 五月 2019) $
# $Revision: 11590 $

import datetime

from collections import OrderedDict
from cms.plugin_base import CMSPluginBase
from cms.plugin_pool import plugin_pool
from django.utils.translation import ugettext as _
from django.template.loader import select_template
from django.db.models import Q
from aldryn_newsblog.models import Article, Category
from taggit.models import Tag

from Zephyrus.blog_extension import settings as _settings
from Zephyrus.blog_extension.models import FilterPlugin, DatePluginConfig

class FilterArticlePlugin(CMSPluginBase):
    model = FilterPlugin
    module = 'News & Blog'
    name = _("Filter Articles")
    render_template =_settings.ZEPHYRUS_BLOG_MAYLIKE_DEFAULT_TEMPLATE

    def get_fieldsets(self, request, obj=None):
        fieldsets = (
            (None, {
                'fields': (
                    'limit',
                    'filterMethod',
                    'renderTemplate',
                )
            }),
            (_('Categories'), {
                'classes': ('collapse', 'wide', 'extrapretty'),
                'fields': (
                    ('allCatagory', 'catagory'),
                )
            }),
            (_('Tag'), {
                'classes': ('collapse', 'wide', 'extrapretty'),
                'fields': (
                    ('allTag', 'tag'),
                )
            }),
        )
        return fieldsets

    def get_render_template(self, context, instance, placeholder):
        return select_template([
            instance.renderTemplate,
            _settings.ZEPHYRUS_BLOG_MAYLIKE_TEMPLATES,
        ])

    def render(self, context, instance, placeholder):
        if instance.allCatagory:
            categoryList = []
        else:
            categoryList = []      
            if instance.catagory:
                stack = [instance.catagory, ]
                while stack:
                    category = stack.pop()
                    if not category in categoryList:
                        categoryList.append(category)

                    for child in category.get_children():
                        stack.append(child)
            else:
                categoryList.append(None)

        if instance.allTag:
            tagList = []
        else:
            tagList = []
            if instance.tag.all():
                tagList = [tag.name for tag in instance.tag.all()]
            else:
                tagList.append(None)

        if instance.filterMethod == 'or':
            query = None
            for category in categoryList:
                if query:
                    query = query | Q(categories=category)
                else:
                    query = Q(categories=category)
            for tag in tagList:
                if query:
                    if tag:
                        query = query | Q(tags__name__in=[tag,])
                    else:
                        query = query | Q(tags=None)
                else:
                    if tag:
                        query = Q(tags__name__in=[tag,])
                    else:
                        query = Q(tags=None)
        else:
            query = None
            for category in categoryList:
                if query:
                    query = query & Q(categories=category)
                else:
                    query = Q(categories=category)
            for tag in tagList:
                if query:
                    if tag:
                        query = query & Q(tags__name__in=[tag,])
                    else:
                        query = query & Q(tags=None)
                else:
                    if tag:
                        query = Q(tags__name__in=[tag,])
                    else:
                        query = Q(tags=None)

        if query:
            queryset = Article.objects.filter(query).distinct()
        else:
            queryset = Article.objects.all()
            
        if queryset.count() >= instance.limit:
            articles = queryset[:instance.limit]
        else:
            articles = queryset

        context.update({
            'articles': articles,
        })
        return context

class DatePlugin(CMSPluginBase):
    model = DatePluginConfig
    module = 'News & Blog'
    name = _("Date List")
    render_template =_settings.ZEPHYRUS_BLOG_DATE_DEFAULT_TEMPLATE

    def get_render_template(self, context, instance, placeholder):
        return select_template([
            instance.renderTemplate,
            _settings.ZEPHYRUS_BLOG_DATE_TEMPLATES,
        ])

    def render(self, context, instance, placeholder):
        dateDict = OrderedDict()
        for article in Article.objects.order_by("-publishing_date"):
            year = article.publishing_date.year
            month = article.publishing_date.month
            
            # only use year month, default day = 1
            date = datetime.date(year=year, month=month, day=1) 

            if date in dateDict:
                dateDict[date] += 1
            else:
                dateDict[date] = 1

        context.update({
            'dateDict': dateDict,
        })
        return context

plugin_pool.register_plugin(FilterArticlePlugin)
plugin_pool.register_plugin(DatePlugin)