#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: views.py 11643 2019-06-28 14:14:33Z Andy $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Andy $
# $Date: 2019-06-28 22:14:33 +0800 (週五, 28 六月 2019) $
# $Revision: 11643 $

import datetime
import logging
import urllib
import six

from django.shortcuts import redirect, get_object_or_404
from django.views.generic import View
from django.core.mail import send_mail
from django.conf import settings
from django.template.loader import get_template
from django.template import Context
from django.template.base import Template
from django.contrib import auth
from django.core.validators import validate_email
from django.core.exceptions import ValidationError
from django.template.defaultfilters import slugify

from djangocms_forms.models import FormDefinition
from djangocms_forms.utils import hashid_to_int
from djangocms_forms.forms import FormBuilder

from Zephyrus.landing import ErrorCode
from Zephyrus.landing.models import Email

logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)
logger.addHandler(logging.StreamHandler())

def sendAndRedirect(request, email, fromUrl, redirectTo, formId=None):
    errorCode = ErrorCode.NORMAL
    
    # Deal with Form Plugin
    if 'form_id' in request.POST:
        formId = hashid_to_int(request.POST.get('form_id'))
        landingForm = get_object_or_404(FormDefinition, pk=formId)
        form = FormBuilder(landingForm, data=request.POST, files=request.FILES)
        if not form.is_valid():
            return
            
        order = landingForm.fields.all()
        excludedFieldTypes = ['MathCaptcha', 'ReCaptcha']
        order = [field for field in order 
                 if field.field_type not in excludedFieldTypes]
        
        dataDict = {}
        for field in order:
            key = slugify(field.label)
            value = form.cleaned_data.get(key, '(no input)')
            dataDict[key] = {'value': value, 'field': field,}
    elif formId:
        formId = hashid_to_int(formId)
        landingForm = get_object_or_404(FormDefinition, pk=formId)
        dataDict = {}
        form = True
    else:
        landingForm = None
        form = None
   
    try:
        validate_email(email)

        # Save data to db
        try:
            if not form:
                email = Email.objects.create(
                    email=email,
                    fromUrl=fromUrl,    
                )
                email.save()
            else:
                data = {}
                for key in dataDict:
                    value = dataDict[key]['value']
                    data[key] = value
                email = Email.objects.create(
                    email=email,
                    fromUrl=fromUrl,
                    data=data,
                )
                email.save()
        except Exception as e:
            errorCode = ErrorCode.SAVE_EMAIL
            logger.error('Not successfully save email: %s' % str(e))

        # Send email
        try:
            message = (
                "Customer id : %d\nCustomer email : %s\n" % 
                (email.id, email.email))
            message += "Date : %s\n" % datetime.datetime.now()
            message += "From : %s\n" % fromUrl
            
            if not form:
                title = (
                    '[Landing %s] - customer use landing page' % 
                    request.META['HTTP_HOST'])
            else:
                title = (
                    '[Landing %s] - %s' % (
                    request.META['HTTP_HOST'], landingForm.name))
            
            if not form:
                send_mail(
                    title,
                    message,
                    settings.EMAIL_HOST_USER,
                    settings.ZEPHYRUS_LANDING_SEND_TO,
                    fail_silently=False
                )
            else:
                message += "Other Data : \n"
                
                for key in dataDict:
                    value = dataDict[key]['value']
                    field = dataDict[key]['field']
                    message += "\t%s : %s\n" % (field.label, value)
                send_mail(
                    title,
                    message,
                    settings.EMAIL_HOST_USER,
                    landingForm.email_to.split(','),
                    fail_silently=False
                )
            
            replaceStr = {
                'email': email.email,
            }
            if not form:
                mail = get_template('landing/ThankEmail.txt')
                
                title = get_template('landing/ThankEmailSubject.txt')
                
                send_mail(
                    title.render(),
                    mail.render(replaceStr),
                    settings.EMAIL_HOST_USER,
                    [email.email,],
                    fail_silently=False
                )
            else:
                mail = Template(landingForm.post_submit_msg)
                title = landingForm.email_subject
                send_mail(
                    title,
                    mail.render(Context(replaceStr)),
                    settings.EMAIL_HOST_USER,
                    [email.email,],
                    fail_silently=False
                )
                
        except Exception as e:
            errorCode = ErrorCode.SEND_EMAIL
            logger.error('Not successfully send email: %s' % str(e))
    except ValidationError as e:
        errorCode = ErrorCode.EMAIL_FORMAT 
        logger.error('Email format error: %s' % str(e))

    # Reditrct
    if redirectTo:     
        if six.PY3:
            url = urllib.parse.quote(redirectTo)
        else:
            url = urllib.quote(redirectTo)
    else:
        if hasattr(settings, "ZEPHYRUS_LANDING_THANK_URL"):
            url = fromUrl + settings.ZEPHYRUS_LANDING_THANK_URL
        else:
            url = fromUrl + "thank"
        
    url = url + "?from_url=%s&error=%d" % (fromUrl, errorCode)
    
    if not form:
        return redirect(url)
    else:
        if landingForm.page_redirect:
            url = landingForm.page_redirect.get_public_url() + \
                "?from_url=%s&error=%d" % (fromUrl, errorCode)
        return redirect(url)
    
class EmailView(View):  
    def post(self, request, *args, **kwargs):
        email = request.POST.get('email')
        fromUrl = request.POST.get('fromUrl')
        redirectTo = request.POST.get('redirectTo')
        return sendAndRedirect(request, email, fromUrl, redirectTo)
            
class LandingSocialView(View):
    def get(self, request, *args, **kwargs):
        user = request.user
        email = user.email
        fromUrl = request.GET.get('fromUrl')
        redirectTo = request.GET.get('redirectTo')
        formId = request.GET.get('formId')
        auth.logout(request)
        return sendAndRedirect(
            request, email, fromUrl, redirectTo, formId=formId)
