#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 11055 2018-05-14 09:46:32Z Lavender $
#
# Copyright (c) 2018 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $ (last)
# $Date: 2018-05-14 17:46:32 +0800 (週一, 14 五月 2018) $
# $Revision: 11055 $

from django.conf import settings
from django.contrib.sitemaps import Sitemap
from django.core.urlresolvers import reverse
from django.utils.translation import get_language, activate
from django.utils import translation


sitemaps = {}

if 'cms' in settings.INSTALLED_APPS:
    from cms.sitemaps import CMSSitemap
    sitemaps['cmspages'] = CMSSitemap

if 'oscar' in settings.INSTALLED_APPS:
    from oscar.core.loading import get_model

    Product = get_model('catalogue', 'Product')
    Category = get_model('catalogue', 'Category')


    """
    A basic example what a sitemap could look like for a multi-language Oscar
    instance.
    Creates entries for the homepage, for each product and each category.
    Repeats those for each enabled language.
    """


    class I18nSitemap(Sitemap):
        changefreq = "monthly"
        priority = 0.5
        """
        A language-specific Sitemap class. Returns URLS for items for passed
        language.
        """
        def __init__(self, language):
            self.language = language
            self.originalLanguage = get_language()

        def get_obj_location(self, obj):
            return obj.get_absolute_url()

        def location(self, obj):
            activate(self.language)
            location = self.get_obj_location(obj)
            activate(self.originalLanguage)
            return location


    class StaticSitemap(I18nSitemap):

        def items(self):
            return ['promotions:home', ]

        def get_obj_location(self, obj):
            return reverse(obj)


    class ProductSitemap(I18nSitemap):

        def lastmod(self, obj):
            return obj.date_updated

        def items(self):
            return Product.browsable.all()


    class CategorySitemap(I18nSitemap):

        def items(self):
            return Category.objects.all()


    languageNeutralSitemaps = {
        'static': StaticSitemap,
        'products': ProductSitemap,
        'categories': CategorySitemap,
    }

    for language, __ in settings.LANGUAGES:
        for name, sitemapClass in languageNeutralSitemaps.items():
            sitemaps['{0}-{1}'.format(name, language)] = sitemapClass(language)
     
if 'aldryn_newsblog' in settings.INSTALLED_APPS:
    from aldryn_newsblog.models import Article
    
    class ArticleSitemap(Sitemap):
        changefreq = "monthly"
        priority = 0.5

        def items(self):
            return Article.objects.all()
            
        def lastmod(self, obj):
            return obj.publishing_date
            
        def location(self, obj):
            activate(obj.language_code)
            url = obj.get_absolute_url(language=obj.language_code)
            translation.deactivate()
            return url
            
    sitemaps['blog'] = ArticleSitemap
