#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: admin.py 10982 2018-04-16 08:14:06Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2018-04-16 16:14:06 +0800 (週一, 16 四月 2018) $
# $Revision: 10982 $

import os
import datetime
import hashlib

from functools import update_wrapper
from django.contrib import admin
from django import forms
from django.conf import settings
from django.utils.translation import ugettext_lazy as _
from django.contrib.admin.templatetags.admin_static import static
from django.forms import Media
from django.db import transaction
from django.conf.urls import url
from django.http import HttpResponseRedirect
from django.contrib import messages
from django.core.urlresolvers import reverse

# 介面拖拉上傳

# ------------------------------------------------------------------------------
from Zephyrus.template_editor.models import Static, Template, Version
from Zephyrus.template_editor import BACKUP_FILE_DIR

TEXT_SUB_FILE_NAME = [
    'txt', 'css', 'js', 'html', 'htm',
]

IMAGE_SUB_FILE_NAME = [
    'jpg', 'png', 'jpeg', 'bmp', 'gif', 'tif',
]


class FolderFrom(forms.ModelForm):
    class Meta(object):
        model = Static
        exclude = ('parent', 'type', 'name')

class TextFrom(forms.ModelForm):
    content = forms.CharField(
        label=_('Content'), widget=forms.Textarea, required=False)

    class Meta(object):
        model = Static
        exclude = ('parent', 'type', 'name')

class ImageFrom(forms.ModelForm):
    image = forms.ImageField(label=_('Image'), required=False)

    class Meta(object):
        model = Static
        exclude = ('parent', 'type', 'name')

class BaseFileAdmin(admin.ModelAdmin):
    change_list_template = 'template_editor/change_list.html'
    change_form_template = 'template_editor/change_form.html'

    def get_urls(self):
        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)
            return update_wrapper(wrapper, view)

        info = self.model._meta.app_label, self.model._meta.model_name
        
        urlpatterns = [
            url(r'^version/(?P<id>.+)/removeConflict/$', 
                wrap(self.removeConflictView), 
                name="%s_%s_removeConflict" % info),
        ] + super(BaseFileAdmin, self).get_urls()

        return urlpatterns

    def removeConflictView(self, request, extra_context=None, **kwargs):
        version = Version.objects.get(id=kwargs['id'])
        version.conflict = False
        version.save()

        info = self.model._meta.app_label, self.model._meta.model_name
        url = reverse('admin:%s_%s_changelist' % info)
        messages.success(request, 'Conflict solved.')
        return HttpResponseRedirect(url)

    def createVersion(self, node, conflict=False):
        fileName = node.name.split('.')[0]
        subFileName = node.name.split('.')[-1]

        fullPath = node.absolutePath
        path = node.path

        with open(fullPath, 'rb') as f:
            data = f.read()

        dirName = hashlib.md5(path).hexdigest()
        checksum = hashlib.md5(data).hexdigest()

        createFile = False

        if Version.objects.filter(dirName=dirName).exists():
            # conflict or not
            preVersion = node.version
            maxVersion = preVersion.number
            backupName = "%s_%d.%s" % (fileName, maxVersion + 1, subFileName)

            if not preVersion.checksum == checksum:
                version = Version.objects.create(
                    dirName=dirName,
                    checksum=checksum,
                    number=maxVersion + 1,
                    fileName=backupName,
                    conflict=conflict,
                    preVersion=preVersion,
                )
                preVersion.nextVersion = version
                preVersion.save()

                createFile = True 
            else:
                version = preVersion
        else:
            maxVersion = 0
            backupName = "%s_%d.%s" % (fileName, maxVersion, subFileName)

            version = Version.objects.create(
                dirName=dirName,
                checksum=checksum,
                number=maxVersion,
                fileName=backupName,
            )
            createFile = True 

        if createFile:
            dirPath = os.path.join(BACKUP_FILE_DIR, dirName)
            if not os.path.isdir(dirPath):
                os.mkdir(dirPath)

            backupPath = os.path.join(dirPath, backupName)

            with open(backupPath, 'wb') as f:
                f.write(data)

        return version

    @transaction.atomic
    def checkVersion(self):
        conflictNodeList = []
        for node in self.model.objects.filter(type='file'):
            version = self.createVersion(node, conflict=True)

            path = node.path
            dirName = hashlib.md5(path).hexdigest()

            for version in Version.objects.filter(
                dirName=dirName, conflict=True):
                conflictNodeList.append((node, version))

        return conflictNodeList


    def get_form(self, request, obj=None, **kwargs):
        if not obj:
            # add Form 新增能夠上傳檔案來新增 static 檔案
            return super(BaseFileAdmin, self).get_form(request, obj, **kwargs)

        # is folder
        if obj.type == 'folder':
            return FolderFrom

        # is file
        versionNum = request.GET.get('version', None)
        if versionNum:
            version = obj.getVersion(versionNumber=versionNum)
        else:
            version = obj.version

        fullPath = version.absoluteBackupPath
        subFileName = obj.name.split('.')[-1].lower()

        if subFileName in TEXT_SUB_FILE_NAME:
            form = TextFrom
            with open(fullPath, 'rb') as f:
                data = f.read()
            form.base_fields['content'].initial = data
        elif subFileName in IMAGE_SUB_FILE_NAME:
            form = ImageFrom
        else:
            form = FolderFrom

        return form

    def change_view(self, request, object_id, form_url='', extra_context=None):
        extra_context = {}
        obj = self.model.objects.get(id=object_id)

        versionNum = request.GET.get('version', None)
        if versionNum:
            extra_context['version'] = obj.getVersion(versionNumber=versionNum)
        else:
            extra_context['version'] = obj.version

        subFileName = obj.name.split('.')[-1].lower()

        if subFileName in TEXT_SUB_FILE_NAME:
            extra_context['isImg'] = False
        elif subFileName in IMAGE_SUB_FILE_NAME:
            extra_context['isImg'] = True
        else:
            extra_context['isImg'] = False

        extra_context['moduleName'] = obj._meta.verbose_name_plural

        return super(BaseFileAdmin, self).change_view(
            request, object_id, form_url=form_url, extra_context=extra_context)

    def response_change(self, request, obj):
        versionNum = request.GET.get('version', None)
        if versionNum:
            version = obj.getVersion(versionNumber=versionNum)
        else:
            version = obj.version

        if obj.type == 'file':
            subFileName = obj.name.split('.')[-1].lower()

            if subFileName in TEXT_SUB_FILE_NAME:
                # 能支援上傳修改檔案
                content = request.POST.get('content')
                
                fullPath = obj.absolutePath
                with open(fullPath, 'wb') as f:
                    data = f.write(content.encode('utf-8'))
            elif subFileName in IMAGE_SUB_FILE_NAME:
                # 圖片轉檔
                image = request.FILES.get('image')
                fullPath = obj.absolutePath

                if image:
                    data = image.read()
                    image.close()

                    with open(fullPath, 'wb') as f:
                        data = f.write(data)
                elif not obj.version == version:
                    backupPath = version.absoluteBackupPath
                    with open(backupPath, 'rb') as f:
                        data = f.read()
                    fullPath = obj.absolutePath
                    with open(fullPath, 'wb') as f:
                        data = f.write(data)
                else:
                    pass
            
            version = self.createVersion(obj)

        return super(BaseFileAdmin, self).response_change(request, obj)

class StaticAdmin(BaseFileAdmin):

    def changelist_view(self, request, extra_context=None):
        conflictNodeList = self.checkVersion()

        roots = [node for node in Static.objects.all() if node.is_root_node()]
         
        extra_context = {
            'roots': roots,
            'conflictNodeList': conflictNodeList,
        }
        
        return super(StaticAdmin, self).changelist_view(
            request,  extra_context=extra_context)

class TemplateAdmin(BaseFileAdmin):

    def changelist_view(self, request, extra_context=None):
        conflictNodeList = self.checkVersion()

        roots = [node for node in Template.objects.all() if node.is_root_node()]

        extra_context = {
            'roots': roots,
            'conflictNodeList': conflictNodeList,
        }
        
        return super(TemplateAdmin, self).changelist_view(
            request,  extra_context=extra_context)

admin.site.register(Static, StaticAdmin)
admin.site.register(Template, TemplateAdmin)