#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Util.py 11220 2018-09-02 04:29:45Z Lavender $
#
# Copyright (c) 2018 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $ (last)
# $Date: 2018-09-02 13:29:45 +0900 (週日, 02 九月 2018) $
# $Revision: 11220 $

import sys
import hashlib

from collections import OrderedDict

if sys.version_info.major == 3:
    from urllib.parse import urlencode
else:
    from urllib import urlencode

# ​https://github.com/vinta/python-pay2go

class Pay2GoMPGAPI(object):

    def __init__(self, hashKey, hashIv, sandbox, *args, **kwargs):
        self.hashKey = hashKey
        self.hashIv = hashIv
        if sandbox:
            self.ApiUrl = 'https://ccore.spgateway.com/MPG/mpg_gateway'
        else:
            self.ApiUrl = 'https://core.spgateway.com/MPG/mpg_gateway'

        self.requiredFields = (
            'Amt',
            'MerchantID',
            'MerchantOrderNo',
            'TimeStamp',
            'Version',
        )

    def buildFormData(self, data):
        def generateCheckValue(data):
            orderedDict = OrderedDict()

            orderedDict['HashKey'] = self.hashKey
            for field in self.requiredFields:
                orderedDict[field] = data[field]
            orderedDict['HashIV'] = self.hashIv

            checkValue = hashlib.sha256(
                urlencode(orderedDict).encode('utf-8')).hexdigest().upper()

            return checkValue

        formData = dict(data)
        formData['CheckValue'] = generateCheckValue(data)

        return formData

    def createPayment(self, data):
        payment = Payment(
            formAction=self.ApiUrl,
            formData=self.buildFormData(data)
        )

        return payment

CATHAPBK_SERVICE_URL = \
    'https://sslpayment.uwccb.com.tw/EPOSService/Payment/OrderInitial.aspx'

from django.template.loader import get_template
from django.template import Context

class CathaybkAPI(object):

    def __init__(self, merchantId, cubkey):
        self.merchantId = merchantId
        self.cubkey = cubkey

    def generateCheckValue(self, data):
        checkValue = hashlib.md5(
            self.merchantId + data['ORDERNUMBER'] + 
            str(data['AMOUNT']) + self.cubkey).hexdigest()
        return checkValue

    def buildXml(self, data):
        temp = get_template('shop/payment/Cathaybk.xml')
        context = Context({
            'data': data,
            'checkValue': self.generateCheckValue(data),
            'merchantId': self.merchantId,
        })
        xml = temp.render(context)
        return xml

    def buildForm(self, data):
        xml = self.buildXml(data)
        temp = get_template('shop/payment/Cathaybk.html')
        context = Context({
            'xml': xml,
            'apiUrl': CATHAPBK_SERVICE_URL,
        })
        form = temp.render(context)
        return form
        


class Payment(object):

    def __init__(self, formAction, formData, *args, **kwargs):
        self.formAction = formAction
        self.formData = formData