#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: views.py 12011 2020-03-25 11:01:56Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2020-03-25 19:01:56 +0800 (週三, 25 三月 2020) $
# $Revision: 12011 $

import json

import requests

from django.shortcuts import redirect, render
from django.views.generic import View
from django.views.decorators.csrf import csrf_exempt
from django.utils.decorators import method_decorator
from django.http import Http404, HttpResponse
from django.conf import settings
from django.utils.translation import ugettext as _
from django.urls import reverse

from Zephyrus.facebook import MANAGE_PAGES_STATE
from Zephyrus.facebook import settings as FBSettings
from Zephyrus.facebook import getCommentURL, getAPIData
from Zephyrus.facebook.models import APIData, FacebookSettings

def createAuthorizationURL(request, scope=["manage_pages", "pages_show_list"], state=MANAGE_PAGES_STATE):
    scope = ",".join(scope)
    redirectUrl = "https://%s%s" % (request.META['HTTP_HOST'], reverse("facebookAuthorization"))
    url = ("https://www.facebook.com/v6.0/dialog/oauth?client_id="
           "%s&redirect_uri=%s&scope=%s") % (
            FBSettings.ZEPHYRUS_FACEBOOK_DEFAULT_APP_ID, redirectUrl, scope)
            
    if state:
        url = url + "&state=%s" % state
    
    return url
    
def createAccessTokenAPIURL(request, code):
    redirectUrl = "https://%s%s" % (request.META['HTTP_HOST'], reverse("facebookAuthorization"))
    url = ("https://graph.facebook.com/v6.0/oauth/access_token"
           "?client_id=%s&redirect_uri=%s&client_secret=%s&code=%s") % (
           FBSettings.ZEPHYRUS_FACEBOOK_DEFAULT_APP_ID, redirectUrl, 
           FBSettings.ZEPHYRUS_FACEBOOK_DEFAULT_APP_SECRET, code)
    return url

class AuthorizationRedirectView(View):

    def get(self, request, *args, **kwargs):
        return redirect(createAuthorizationURL(request))
    

class FacebookAuthorizationView(View):

    def get(self, request, *args, **kwargs):
        if MANAGE_PAGES_STATE == request.GET.get("state"):
            # dealwith auth as access token
            code = request.GET.get("code")
            url = createAccessTokenAPIURL(request, code)
            
            try:
                data = requests.get(url)
                data = json.loads(data.text)
                accessToken = data.get("access_token")
                
                if accessToken:
                    # save accessToken
                    config, created = FacebookSettings.objects.get_or_create()
                    config.accessToken = accessToken
                    config.save()
                    msg = _("Authorization successfully, this window will close automatically.")
                else:
                    msg = _("Authorization failed, please try again. This window will close automatically.")
            except Exception as e:
                msg = _("Authorization failed, please try again. This window will close automatically.")
            return render(request, "facebook/Authorization.html", {"msg": msg,})
        raise Http404()
   
if "Zephyrus.comment" in settings.INSTALLED_APPS:
    from Zephyrus.comment.models import Comment
    
    class FacebookWebHookView(View):
        def get(self, request, *args, **kwargs):
            verify = request.GET.get('hub.verify_token')
            if verify == FBSettings.ZEPHYRUS_FACEBOOK_WEBHOOK_VERIFY_TOKEN:
                return HttpResponse(request.GET.get('hub.challenge'))
            else:
                return HttpResponse('Error, invalid token')
            
        @method_decorator(csrf_exempt)
        def dispatch(self, request, *args, **kwargs):
            return super(
                FacebookWebHookView, self).dispatch(request, *args, **kwargs)

        def post(self, request, *args, **kwargs):
            msg = request.body.decode('utf-8')
            
            data = json.loads(msg)

            entry = data.get("entry")
            if entry:
                for e in entry:
                    changes = e.get("changes")
                    if changes:
                        for c in changes:
                            field = c.get("field")
                            message = c.get("value", {}).get("message")
                            mid = c.get("value", {}).get("id")
                            name = c.get("value", {}).get("from", {}).get("name")

                            if field == "plugin_comment":
                                url = getCommentURL(mid)
                                commentCount = getAPIData(url, "comment_count")
                               

                                # save API data
                                obj, created = APIData.objects.get_or_create(
                                    url=url,
                                )
                                obj.commentCount = commentCount
                                
                                obj.save()

                                # save facebook comment
                                comment = Comment.objects.create(
                                    url=url,
                                    message=message,
                                    facebook=True,
                                    facebookMsgId=str(mid),
                                    facebookUser=name,
                                )
                                
            return HttpResponse('Success')