#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: views.py 12251 2020-05-20 19:31:27Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2020-05-21 04:31:27 +0900 (週四, 21 五月 2020) $
# $Revision: 12251 $

import json
import urllib
import base64

import requests

from django.shortcuts import redirect, render
from django.views.generic import View
from django.views.decorators.csrf import csrf_exempt
from django.utils.decorators import method_decorator
from django.http import Http404, HttpResponse
from django.conf import settings
from django.utils.translation import ugettext as _
from django.urls import reverse

from Iuppiter.Logging import createLogger
from Iuppiter.Encryption import Encryptor
from Zephyrus.facebook import MANAGE_PAGES_STATE, SCOPE
from Zephyrus.facebook import settings as FBSettings
from Zephyrus.facebook import getCommentURL, getAPIData
from Zephyrus.facebook.models import APIData, FacebookSettings

logger = createLogger(__name__)

def createAuthorizationURL(request, scope=SCOPE, state={"code": MANAGE_PAGES_STATE,}, redirectToURL=None):
    scope = ",".join(scope)
    redirectURL = "https://%s%s" % (request.META['HTTP_HOST'], reverse("facebookAuthorization"))
    url = ("https://www.facebook.com/v6.0/dialog/oauth?client_id="
           "%s&redirect_uri=%s&scope=%s") % (
            FBSettings.ZEPHYRUS_FACEBOOK_DEFAULT_APP_ID, redirectURL, scope)
            
    if redirectToURL:
        state.update({
            "redirectToURL": redirectToURL,
        })
            
    if state:
        url = url + "&state=%s" % json.dumps(state)
    
    return url
    
def createAccessTokenAPIURL(request, code):
    redirectURL = "https://%s%s" % (request.META['HTTP_HOST'], reverse("facebookAuthorization"))
    url = ("https://graph.facebook.com/v6.0/oauth/access_token"
           "?client_id=%s&redirect_uri=%s&client_secret=%s&code=%s") % (
           FBSettings.ZEPHYRUS_FACEBOOK_DEFAULT_APP_ID, redirectURL, 
           FBSettings.ZEPHYRUS_FACEBOOK_DEFAULT_APP_SECRET, code)
    return url
    
def encrypt(data):
    data = json.dumps(data) 
    encryptor = Encryptor(FBSettings.ZEPHYRUS_FACEBOOK_DEFAULT_SECRET_KEY.encode())
    secret = encryptor.encrypt(data.encode())  
    secret = base64.b64encode(secret)
    secret = urllib.parse.quote(secret)
    return secret
    
def decrypt(secret):
    encryptor = Encryptor(FBSettings.ZEPHYRUS_FACEBOOK_DEFAULT_SECRET_KEY.encode())
    secret = urllib.parse.unquote(secret)
    secret = base64.b64decode(secret.encode())
    data = encryptor.decrypt(secret)
    data = json.loads(data)
    return data
    
class FacebookAccessTokenCreateView(View):

    def get(self, request, *args, **kwargs):
        referer = request.META.get('HTTP_REFERER')
        # 要使用這個需要同一個站內的連結
        if referer:
            redirectURL = "%s://%s%s" % (
                request.scheme, request.META['HTTP_HOST'], reverse('facebookAccessTokenReturn'))
            
            # 使用這個 view 需要以下兩種參數
            data = {
                "redirect": redirectURL,
                "password": FBSettings.ZEPHYRUS_FACEBOOK_DEFAULT_PASSWORD,
            }
            secret = encrypt(data)
            
            return redirect("%s%s?secret=%s" % (
                FBSettings.ZEPHYRUS_FACEBOOK_DEFAULT_GET_ACCESS_TOKEN_URL, 
                reverse('facebookAuthorizationRedirect'), secret))
        else:
            raise Http404()
        
class FacebookAccessTokenReturnView(View):

    def get(self, request, *args, **kwargs):
        secret = request.GET.get("secret")
        data = decrypt(secret)
        accessToken = data.get("token")
        msg = data.get("msg")
        
        if accessToken:
            # save accessToken
            config, created = FacebookSettings.objects.get_or_create()
            config.accessToken = accessToken
            config.save()
            
        return render(request, "facebook/Authorization.html", {"msg": msg,})
        
class AuthorizationRedirectView(View):

    def get(self, request, *args, **kwargs):
        secret = request.GET.get("secret")
        data = decrypt(secret)
        redirectToURL = data.get("redirect")
        password = data.get("password")
        
        xForwardedFor = request.META.get('HTTP_X_FORWARDED_FOR')
        if xForwardedFor:
            ip = xForwardedFor.split(',')[0]
        else:
            ip = request.META.get('REMOTE_ADDR')
        
        logger.info("[Iuno.facebook] Someone call FB authorization view: ip=%s data=%s" % (ip, data))
        
        if redirectToURL and password == FBSettings.ZEPHYRUS_FACEBOOK_DEFAULT_PASSWORD:
            return redirect(createAuthorizationURL(request, redirectToURL=redirectToURL))
        raise Http404()
    

class FacebookAuthorizationView(View):
    
    SUCCESS_MESSAGE = _("Authorization successfully, this window will close automatically.")
    ERROR_MESSAGE = _("Authorization failed, please try again. This window will close automatically.")

    def get(self, request, *args, **kwargs):
        state = json.loads(request.GET.get("state", "{}"))
        redirectToURL = state.get("redirectToURL")
        code = state.get("code")
        if MANAGE_PAGES_STATE == code and redirectToURL:
            # dealwith auth as access token
            code = request.GET.get("code")
            url = createAccessTokenAPIURL(request, code)
            
            try:
                data = requests.get(url)
                data = json.loads(data.text)
                accessToken = data.get("access_token")
                
                if accessToken:
                    logger.info("Success get access token from facebook API: %s" % data)
                    msg = self.SUCCESS_MESSAGE
                else:
                    logger.error("Can't get access token from facebook API: %s" % data)
                    msg = self.ERROR_MESSAGE
            except Exception as e:
                logger.error("Can't get access token from facebook API: %s" % e)
                msg = self.ERROR_MESSAGE
                
            data = {
                "msg": msg,
                "token": accessToken,
            }
            secret = encrypt(data)  
            redirectToURL = "%s?secret=%s" % (redirectToURL, secret)
            return redirect(redirectToURL)
        raise Http404()
   
if "Zephyrus.comment" in settings.INSTALLED_APPS:
    from Zephyrus.comment.models import Comment
    
    class FacebookWebHookView(View):
        def get(self, request, *args, **kwargs):
            verify = request.GET.get('hub.verify_token')
            if verify == FBSettings.ZEPHYRUS_FACEBOOK_WEBHOOK_VERIFY_TOKEN:
                return HttpResponse(request.GET.get('hub.challenge'))
            else:
                return HttpResponse('Error, invalid token')
            
        @method_decorator(csrf_exempt)
        def dispatch(self, request, *args, **kwargs):
            return super(
                FacebookWebHookView, self).dispatch(request, *args, **kwargs)

        def post(self, request, *args, **kwargs):
            msg = request.body.decode('utf-8')
            
            data = json.loads(msg)

            entry = data.get("entry")
            if entry:
                for e in entry:
                    changes = e.get("changes")
                    if changes:
                        for c in changes:
                            field = c.get("field")
                            message = c.get("value", {}).get("message")
                            mid = c.get("value", {}).get("id")
                            name = c.get("value", {}).get("from", {}).get("name")

                            if field == "plugin_comment":
                                url = getCommentURL(mid)
                                commentCount = getAPIData(url, "comment_count")
                               

                                # save API data
                                obj, created = APIData.objects.get_or_create(
                                    url=url,
                                )
                                obj.commentCount = commentCount
                                
                                obj.save()

                                # save facebook comment
                                comment = Comment.objects.create(
                                    url=url,
                                    message=message,
                                    facebook=True,
                                    facebookMsgId=str(mid),
                                    facebookUser=name,
                                )
                                
            return HttpResponse('Success')