from django.conf.urls import url
from django.utils.translation import gettext_lazy as _

from oscar.core.application import OscarDashboardConfig
from oscar.core.loading import get_class

class PromotionsDashboardConfig(OscarDashboardConfig):
    label = 'dashboard_promotions'
    name = 'Iuno.shop.dashboard_promotions'
    verbose_name = _('Promotions dashboard')
    default_permissions = ['is_staff', ]

    namespace = 'dashboard'
    
    def ready(self):
        from Iuno.shop.dashboard_promotions.views import (
            ListView, PageListView, PageDetailView, 
            CreateRedirectView, DeletePagePromotionView)
            
        self.list_view = ListView
        self.page_list = PageListView
        self.page_detail = PageDetailView
        self.create_redirect_view = CreateRedirectView
        self.delete_page_promotion_view = DeletePagePromotionView

    def get_urls(self):
        from Iuno.shop.promotions.conf import PROMOTION_CLASSES
        view_names = (
            ('create_%s_view', 'Create%sView'),
            ('update_%s_view', 'Update%sView'),
            ('delete_%s_view', 'Delete%sView')
        )
        
        viewList = {}
        for klass in PROMOTION_CLASSES:
            for attr_name, view_name in view_names:
                full_attr_name = attr_name % klass.classname()
                full_view_name = view_name % klass.__name__
                view = get_class('dashboard_promotions.views', full_view_name)
                viewList[full_attr_name] = view
                
        urls = [
            url(r'^$', self.list_view.as_view(), name='promotion-list'),
            url(r'^pages/$', self.page_list.as_view(),
                name='promotion-list-by-page'),
            url(r'^page/(?P<path>/([\w-]+(/[\w-]+)*/)?)$',
                self.page_detail.as_view(), name='promotion-list-by-url'),
            url(r'^create/$',
                self.create_redirect_view.as_view(),
                name='promotion-create-redirect'),
            url(r'^page-promotion/(?P<pk>\d+)/$',
                self.delete_page_promotion_view.as_view(),
                name='pagepromotion-delete')]

        for klass in PROMOTION_CLASSES:
            code = klass.classname()
            urls += [
                url(r'create/%s/' % code,
                    viewList['create_%s_view' % code].as_view(),
                    name='promotion-create-%s' % code),
                url(r'^update/(?P<ptype>%s)/(?P<pk>\d+)/$' % code,
                    viewList['update_%s_view' % code].as_view(),
                    name='promotion-update'),
                url(r'^delete/(?P<ptype>%s)/(?P<pk>\d+)/$' % code,
                    viewList['delete_%s_view' % code].as_view(),
                    name='promotion-delete')]

        return self.post_process_urls(urls)

