#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Amazon.py 8963 2013-03-25 08:56:48Z Yachu $
#
# Copyright (c) 2013 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Yachu $
# $Date: 2013-03-25 16:56:48 +0800 (週一, 25 三月 2013) $
# $Revision: 8963 $

import boto

class SES(object):
    def __init__(self, AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY):

        self.connection = boto.connect_ses(
            AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY)

    def getIdentityNotificationAttributes(self, email):

        values = self.connection._make_request(
            "GetIdentityNotificationAttributes", {"Identities.member.1": email})
        values = values['GetIdentityNotificationAttributesResponse']
        values = values['GetIdentityNotificationAttributesResult']
        values = values['NotificationAttributes']['entry']['value']
        return values

    def setIdentityNotificationTopic(self, notificationType, identity, topic):

        self.connection._make_request(
            "SetIdentityNotificationTopic", {
                'NotificationType': notificationType,
                'Identity': identity,
                'SnsTopic': topic
            }
        )

    def getAllVerifiedSenders(self):

        response = self.connection.list_verified_email_addresses()
        response = response.get('ListVerifiedEmailAddressesResponse', {})
        result = response.get('ListVerifiedEmailAddressesResult', {})
        verifiedList = result.get('VerifiedEmailAddresses', [])
        return verifiedList

    def verifyEmailAddress(self, email):

        self.connection.verify_email_address(email)

    def deleteVerifiedEmailAddress(self, email):

        response = self.connection.delete_verified_email_address(email)
        response = response.get('DeleteVerifiedEmailAddressResponse', {})
        return response

class SNS(object):
    def __init__(self, AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY):

        self.connection = boto.connect_sns(
            AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY)

    def getAllTopicArns(self):
        allTopics = self.connection.get_all_topics()
        allTopics = allTopics['ListTopicsResponse']['ListTopicsResult']
        allTopics = allTopics['Topics']
        allTopics = [d['TopicArn'] for d in allTopics]
        return allTopics

    def createTopic(self, name):
        topic = self.connection.create_topic(name)
        topic = topic['CreateTopicResponse']['CreateTopicResult']
        arn = topic['TopicArn']
        return arn

    def getAllSubscriptionsByTopic(self, topic):
        response = self.connection.get_all_subscriptions_by_topic(topic)
        response = response['ListSubscriptionsByTopicResponse']
        response = response['ListSubscriptionsByTopicResult']
        response = response['Subscriptions']
        return response

    def checkSubscriptionExist(self, topic, arnName):
        def getNameFromArn(arn):
            s = arn.split(':')
            return s[-1]

        for r in self.getAllSubscriptionsByTopic(topic):
            if r['Protocol'] == 'http':
                arn = getNameFromArn(r['TopicArn'])
                if arn == arnName:
                    return True
        return False

    def subscribe(self, topic, subscriptionType, link):
        self.connection.subscribe(topic, 'http', link)

def verifyEmailAddress(request):
    """
    Verify email address.

    Reference:
        http://docs.amazonwebservices.com/AWSSDKforPHP/latest/index.html#
        m=AmazonSES/verify_email_address

        and

        http://docs.amazonwebservices.com/AWSSDKforPHP/latest/
        index.html#m=AmazonSES/list_verified_email_addresses
    @param email Email address.
    """
    try:
        email = filterEmail(request.GET.get('email', ''))

        ses = SES(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY)
        verifiedList = ses.getAllVerifiedSenders()

        if email in verifiedList:
            response = {
                'verification': True,
                'continue': False,
                'message': _("This email address already verified.\n")
            }

            checkBounce = None
            checkComplaint = None

            notifyAttrValues = ses.getIdentityNotificationAttributes(email)
            bounceTopic = notifyAttrValues.get('BounceTopic', '')
            if bounceTopic:
                checkBounce = False
                if getNameFromArn(bounceTopic) == BOUNCES_TOPIC_NAME:
                    checkBounce = True

            complaintTopic = notifyAttrValues.get('ComplaintTopic', '')
            if complaintTopic:
                checkComplaint = False
                if getNameFromArn(complaintTopic) == COMPLAINTS_TOPIC_NAME:
                    checkComplaint = True

            if (checkBounce is False) or (checkComplaint is False):
                response['continue'] = True
                response['message'] += _(
                    "Your email has been set for other system.\n"
                    "Are you sure you want to continue?")
            elif (checkBounce is None) or (checkComplaint is None):
                response['message'] += _("You not yet set mail notifications.")
            return response
        else:
            try:
                ses.verifyEmailAddress(email)
                return {
                    'verification': False,
                    'message': _(
                        "You not yet verify your email address.\n"
                        "Your verify request was sent, please to receive "
                        "verify email on your sender email box.\n")
                }

            except Exception as e:
                return {
                    'verification': False,
                    'message': _("Send verify request failed, please check "
                                 "your e-mail address.\nreason: %s" % str(e))
                }
    except EmailAddressException as e:
        return {
            'verification': False,
            'message': _unicode(e)
        }
    except Exception as e:
        return {
            'verification': False,
            'message': _("Unknown error: %s") % str(e)
        }
