#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Backend.py 9260 2013-07-05 03:27:04Z Yachu $
#
# Copyright (c) 2013 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Yachu $
# $Date: 2013-07-05 11:27:04 +0800 (週五, 05 七月 2013) $
# $Revision: 9260 $

from urllib import urlopen, urlencode, quote
from django.core.mail.backends.base import BaseEmailBackend

from Iuppiter.Encoding import utf8, _unicode
from Iuppiter.Json import serializer

class EmailBackend(BaseEmailBackend):

    def __init__(self, host=None, port=None, username=None, password=None,
                 use_tls=None, fail_silently=False, **kwargs):
        """
        Constructor.
        """
        super(EmailBackend, self).__init__(fail_silently=fail_silently)

        self.username = username
        self.password = password

        if host is None:
            host = 'localhost'

        _port = ':%s' % str(port) if port else ''
        _domain = '%s%s/newsletters/api/' % (host, _port)

        if use_tls is True:
            _scheme = 'https'
        else:
            _scheme = 'http'

        self.domain = '%s://%s' % (_scheme, _domain)

        self.kwargs = kwargs

    def parseResult(self, content):
        """
        Parse content.

        @param content Content.
        """
        try:
            content = serializer.deserialize(content)
            if 'error' in content:
                raise RuntimeError(
                    '[parseResult]:\n'
                    '%s: %s' % (content['error'], content['message']))
            else:
                return content
        except Exception as e:
            raise e

    def _getToken(self, app):
        """
        Get token.

        @param app App name.
        """
        try:
            data = urlencode({
                'user': self.username,
                'pwd': self.password,
                'app': utf8(app)
            })
            link = '%sapp/login/' % self.domain
            result = urlopen(link, data=data).read()
            result = self.parseResult(result)
            return result['token']
        except Exception as e:
            raise e

    def send_messages(self, email_messages):
        """
        Send messages.

        @param email_messages Email messages.
        """
        if not email_messages:
            return

        for email in email_messages:
            app = email.extra_headers.get('app', '')
            token = self._getToken(app)
            title = utf8(email.subject)
            data = {
                'token': token,
                'title': title,
                'emails': ','.join(email.to),
                'text': utf8(email.body),
                'sender': email.from_email,
            }

            replyTo = email.extra_headers.get('Reply-To', '')
            if replyTo:
                data['reply'] = replyTo

            senderEmail = email.extra_headers.get('Sender', '')
            if senderEmail:
                data['sender'] = senderEmail

            group = email.extra_headers.get('group', '')
            if group:
                data['group'] = utf8(group)
            else:
                data['group'] = title

            data = urlencode(data)

            attachData = {}
            for (n, attach) in enumerate(email.attachments):
                nameKey = 'attachmentName%d' % (n + 1)
                contentKey = 'attachmentContent%d' % (n + 1)
                typeKey = 'attachmentType%d' % (n + 1)

                name, content, cType = attach

                if name:
                    attachData.update({
                        nameKey: quote(name),
                    })

                if cType:
                    attachData.update({
                        typeKey: quote(cType),
                    })

                attachData.update({
                    contentKey: quote(content),
                })
            data = '%s&%s' % (data, urlencode(attachData))

            try:
                link = '%sapp/mail/' % self.domain
                result = urlopen(link, data=data).read()
                result = self.parseResult(result)
                return result
            except Exception as e:
                raise e
