/**
$Id: Iuppiter.js 3723 2010-11-25 09:57:09Z Yachu $

Copyright (c) 2010 Nuwa Information Co., Ltd, and individual contributors.
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice,
     this list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.

  3. Neither the name of Nuwa Information nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

$Author: Yachu $
$Date: 2010-11-25 17:57:09 +0800 (週四, 25 十一月 2010) $
$Revision: 3723 $

Contributors:
 Jumann (original author)
 Bear
 Yachu
*/

if (typeof Iuppiter === 'undefined')
    Iuppiter = {
        version: '$Revision: 3723 $'.substring(11).replace(" $", ""),
        debug: false
    };

/**
 * Define all of namespaces.
 */
Iuppiter.Base = {};
Iuppiter.Util = {};
Iuppiter.DD = {};

/**
 * Copy all the properties of config to object.
 *
 * @param {Object} obj The receiver of the properties
 * @param {Object} config The source of the properties
 * @param {Object} defaults An object that will also be applied as default
 *                          value.
 * @return {Object} Proccessed object.
 */
Iuppiter.expand = function(obj, config, defaults) {
    if (defaults){
        // no "this" reference for friendly out of scope calls
        Iuppiter.expand(obj, defaults);
    }
    if (obj && config && typeof config == 'object') {
        for (var p in config) {
            obj[p] = config[p];
        }
    }
    return obj;
};

/**
 * Append functions to Function object.
 *
 * These functions are available on every Function object (any JavaScript
 * function).
 */
Iuppiter.expand(Function.prototype, {

    /**
     * The method will delay to do after the setting milliseconds.
     *
     * @param {Integer} millis Defer milliseconds.
     * @param {Object} obj The method is belonged from the object.
     * @param {Array} args Arguments of the callback method.
     * @return {Integer} The timeout id.
     */
    setIuppiterDelay: function(millis, obj, args) {
        var callback = this.createIuppiterDelegate(obj, args);
        if (millis)
            return setTimeout(callback, millis);

        callback();
        return 0;
    },

    /**
     * Create a delegate method to wrap the method that sets arguments to the
     * object.
     *
     * @param {Object} obj The method is belonged from the object.
     * @param {Array} args Arguments of the callback method.
     * @param A wrapped delegate method.
     */
    createIuppiterDelegate: function(obj, args) {
        var methodInfo = this;

        return function() {
            var callArgs = args || arguments;

            return methodInfo.apply(obj || window, callArgs);
        };
    },

    /**
     * Create a callback function to wrap the function that sets arguments.
     *
     * @param {Array} args Arguments of the callback method.
     * @param A wrapped function.
     */
    createIuppiterCallback : function(/* *args */){
        var args = arguments;
        var functionInfo = this;

        return function() {
            return functionInfo.apply(window, args);
        };
    }
});

Iuppiter.expand(Iuppiter, {

    /**
     * Get Iuppiter.Element object by the HTML DOM element or the element's id
     * string.
     *
     * @param {HTMLElement|String} element The HTML element object or the id of
     *                                     the HTML element.
     * @return {Element} The Mercuriu.Element object.
     */
    get: function(element){
        var dom = typeof element == "string" ?
                  document.getElementById(element) : element;

        if (!dom) { // invalid id/element.
            return null;
        }

        return new Iuppiter.Element(dom);
    },

    /**
     * Get the client browser information.
     *
     * @return {Object} The browser information data.
     */
    getBrowser: function() {
         var userAgent = navigator.userAgent.toLowerCase();
         var isStrict = document.compatMode == "CSS1Compat",
             // Is less equal IE 6.
             isOpera = userAgent.indexOf("opera") > -1,
             isChrome = userAgent.indexOf('chrome') > -1,
             isIE = !isOpera && userAgent.indexOf("msie") > -1,
             isLEIE6 = !isOpera && (!window.XMLHttpRequest ||
                navigator.userAgent.toLowerCase().indexOf("msie 6.") > -1),
             isIE7 = !isOpera && userAgent.indexOf("msie 7") > -1,
             isIE8 = !isOpera && userAgent.indexOf("msie 8") > -1,
             isIE9 = !isOpera && userAgent.indexOf("msie 9") > -1,
             isBorderBox = isIE && !isStrict;
         return {
             version: (userAgent.match(/.+(?:rv|it|ra|ie)[\/: ]([\d.]+)/) ||
                       [])[1],
             isStrict: isStrict,
             isIE: isIE,
             isLEIE6: isLEIE6,
             isIE7: isIE7,
             isIE8: isIE8,
             isIE9: isIE9,
             isBorderBox: isBorderBox,
             safari: /webkit/.test(userAgent),
             chrome: /chrome/.test(userAgent),
             opera: /opera/.test(userAgent),
             msie: /msie/.test(userAgent) && !/opera/.test(userAgent),
             mozilla: /mozilla/.test(userAgent) &&
                !/(compatible|webkit)/.test(userAgent),
             isIphone: /iphone/.test(userAgent),
             isIpod: /ipod/.test(userAgent),
             isAndroid: /android/.test(userAgent)
         }
    },

    /**
     * Iterate an array calling the passed function with each item, stopping if
     * your function returns false. If the passed array is not really an array,
     * your function is called once with it. The supplied function is called
     * with (Object item, Number index, Array allItems).
     * Reference: jQuery Library.
     *
     * @param {Object} object Object item, Number index, Array allItems.
     * @param {Function} callback The passed function with each item.
     * @param {Array} args Arguments is for internal usage only.
     */
    each: function(object, callback, args) {
        var name, i = 0, length = object.length;

        if (args) {
            if (length === undefined) {
                for (name in object)
                    if (callback.apply(object[name], args) === false)
                        break;
            }
            else {
                for (; i < length;)
                    if (callback.apply(object[i++], args) === false)
                        break;
            }
        }
        // A special, fast, case for the most common use of each.
        else {
            if (length === undefined) {
                for (name in object)
                    if (callback.call(object[name], name, object[name]) ===
                        false)
                        break;
            }
            else {
                var value;
                for (value = object[0];
                     i < length && callback.call(value, i, value) !== false;
                     value = object[++i]) {}
            }
        }

        return object;
    },

    /**
     * Convert resembled array's object to array.
     * Reference: jQuery Library.
     *
     * @param {Object} array The window, strings (and functions) also have
     *                       'length'.
     */
    toArray: function(array) {
        var ret = [];

        if(array != null){
            var i = array.length;
            // The window, strings (and functions) also have 'length'
            /*if (i == null || typeof array === "string" ||
                (typeof array).toLowerCase() === "function" ||
                array.setInterval)
                ret[0] = array;
            else*/
                while(i)
                    ret[--i] = array[i];
        }

        return ret;
    },

    /**
     * Merge two arrays together.
     * Reference: jQuery Library.
     *
     * @param {Array} first The first array to merge, the elements of second
     *                      added.
     * @param {Array} second The second array to merge into the first,
     *                       unaltered.
     * @return {Array} The result is the altered first argument with the
     *                 elements from the second array added.
     *                 To remove duplicate elements from the resulting array,
     *                 use Iuppiter.unique().
     */
    merge: function(first, second) {
        // We have to loop this way because IE & Opera overwrite the length
        // expando of getElementsByTagName
        var i = 0, elem, pos = first.length;

        while ((elem = second[i++]) != null)
            first[ pos++ ] = elem;

        return first;
    },

    /**
     * Remove all duplicate elements from an array of elements. Note that this
     * only works on arrays of DOM elements, not strings or numbers.
     * Reference: jQuery Library.
     *
     * @param {Array} array The array to remove all duplicate elements.
     * @return {Array} The array contains unique elements.
     *
    unique: function(array) {
        var ret = [], done = {}, i, id;

        try {
            for (i = 0, length = array.length; i < length; i++) {
                id = Iuppiter.data(array[i]);

                if (!done[id]) {
                    done[id] = true;
                    ret.push(array[i]);
                }
            }

        } catch(e) {
            ret = array;
        }

        return ret;
    },

    cache: {},

    /**
     *
     * Reference: jQuery Library.
     *
     * @param {Object} elem
     * @param {Object} name
     * @param {Object} data
     *
    data: function( elem, name, data ) {
        elem = elem == window ?
            windowData :
            elem;

        var id = elem[expando];

        // Compute a unique ID for the element
        if (!id)
            id = elem[expando] = ++uuid;

        // Only generate the data cache if we're
        // trying to access or manipulate it
        if (name && !Iuppiter.cache[id])
            Iuppiter.cache[id] = {};

        // Prevent overriding the named cache with undefined values
        if (data !== undefined)
            Iuppiter.cache[id][name] = data;

        // Return the named cache data, or the ID for the element
        return name ? Iuppiter.cache[id][name] : id;
    },
    */

    /**
     * Decode (parse) a JSON string to an object. If the JSON is invalid,
     * this function throws a SyntaxError.
     *
     * @param {String} json The JSON string.
     * @return {Object} The result object.
     */
    decodeJson: function(json) {
        return eval("(" + json + ')');
    },

    /**
     * Load script dynamically.
     *
     * @param {Object} win Window object.
     * @param {Array} scripts A config array object for the details of scripts
     *                        you want to load. Config:
     *                           url: Script URL.
     *                           checkNamespace: The namespace must exist to
     *                                           determine it loaded or not.
     *                           id: Script tag's id.
     * @param {Function} onReady The callback function will be called after all
     *                           scripts loaded.
     * @return {Object} The return value of onReady function.
     */
    loadScript: function(win, scripts, onReady) {
        function doLoad(scripts, next){
            var idx = next - 1;
            var url = scripts[idx].url;
            var checkNamespace = scripts[idx].checkNamespace;
            var id = scripts[idx].id;

            var s = win.document.createElement('script');
            s.type = 'text/javascript';
            s.src = url;
            if (id)
                s.id = id;
            win.document.body.appendChild(s);

            function doNext(next){
                if (scripts.length <= next) {
                    if (typeof(onReady) != 'undefined')
                        return onReady();
                    else
                        return;
                }

                return doLoad(scripts, next + 1);
            }

            if (typeof(checkNamespace) != 'undefined') {
                function check(checkNamespace, next){
                    var exp = 'typeof(' + checkNamespace + ')';
                    /*
                     * IE missing the eval method on the iframe.
                     * Resource: http://www.thismuchiknow.co.uk/?p=25
                     */
                    var evalResult;
                    if (!win.eval && win.execScript) {
                        evalResult = win.execScript(exp);
                    }
                    else {
                        evalResult = win.eval(exp);
                    }
                    if (evalResult == 'undefined') {
                        setTimeout(function(){
                            check(checkNamespace, next);
                        }, 50);
                    }
                    else {
                        return doNext(next);
                    }
                }
                return check(checkNamespace, next);
            }
            else {
                return doNext(next);
            }
        }

        doLoad(scripts, 1);
    },

    /**
     * Load CSS dynamically.
     *
     * @param {Document} doc Document object.
     * @param {String} url CSS URL.
     */
    loadCss: function(doc, url) {
        var s = doc.createElement('link');
        s.type = 'text/css';
        s.rel = 'stylesheet';
        s.href = url;
        try {
            doc.getElementsByTagName("head")[0].appendChild(s);
        }
        catch (e) {
            doc.body.appendChild(s);
        }
    },

    /**
     * The function checks the value that it whether is array or not.
     *
     * @param {Object} value Value you want to check.
     * @return {Boolean} True if value is array.
     */
    isArray: function(value) {
        return value && typeof(value.push) == 'function';
    },

    /**
    * Get the field data of the form convert to JSON.
    *
    * @param {HTMLElement} form The form of the HTML DOM.
    * @return {Object} Converted object.
    */
    formToJson: function(form) {
        var selectFilter = function(el){
            var values = [];
            for (var j = 0; j < element.options.length; j++) {
                if (el.options[j].selected) {
                    if (browser.msie) {
                        values.push(
                            el.options[j].attributes['value'].specified ?
                            el.options[j].value : el.options[j].text);
                    }
                    else {
                        values.push(el.options[j].hasAttribute('value') ?
                                    el.options[j].value : el.options[j].text);
                    }
                }
            }
            if (values.length > 0)
                return values;
        }

        var groupFilter = function(el){
            if (el.checked)
                return el.value;
        };

        var notProccessedFilter = function(el){
        };

        this.hasSubmit = false;
        this.formFieldsMap = {
            'select-one': selectFilter,
            'select-multiple': selectFilter,
            'radio': groupFilter,
            'checkbox': groupFilter,
            'file': notProccessedFilter,
            undefined: notProccessedFilter,
            'reset': notProccessedFilter,
            'button': notProccessedFilter,
            'submit': function(el){
                if (this.hasSubmit == false) {
                    this.hasSubmit = true;
                    return el.value;
                }
            }
        };

        if (typeof form == 'string')
            form = (document.getElementById(form) || document.forms[form]);

        var el, jsonData = {};
        // this.hasSubmit = false;

        for (var i = 0; i < form.elements.length; i++) {
            el = form.elements[i];
            if (!form.elements[i].disabled && el.name) {
                var filter = this.formFieldsMap[el.type];

                if (typeof(filter) != "undefined") {
                    var result = filter(el);
                    if (result !== undefined || result !== null)
                        jsonData[el.name] = result;
                }
                else {
                    jsonData[el.name] = el.value;
                }
            }
        }
        return jsonData;
    },

    /**
     * Take an object and convert it to an encoded query string.
     * e.g. Iuppiter.jsonToQueryString({foo: 1, bar: 2});
     * would return "foo=1&bar=2".
     * Optionally, property values can be arrays, instead of keys and the
     * result string returned will contain a name/value pair for each array
     * value.
     *
     * @param {Object} o Object.
     * @param {String} ignoreAttrs Wants to ignore encoding URI attributes.
     * @return {String} Processed query string.
     */
    jsonToQueryString: function(o, ignoreAttrs) {
        if (!o)
            return "";

        var ignore, defaultFilter = function(key, value){
            if (Iuppiter.isArray(value)) {
                var buf = [], i, j;
                if (value.length) {
                    for (i = 0, len = value.length; i < len; i++) {
                        ignore = false;
                        if (ignoreAttrs) {
                            for (j = 0; j < ignoreAttrs.length; j++) {
                                if (key == ignoreAttrs[j]) {
                                    ignore = true;
                                    break;
                                }
                            }
                        }
                        if (ignore)
                            buf.push(key, "=",
                                value[i] === undefined ? '' : value[i]), "&";
                        else
                            buf.push(key, "=",
                                     encodeURIComponent(value[i] === undefined ?
                                         '' : value[i]), "&");
                    }
                }
                else {
                    buf.push(key, "=&");
                }

                if (buf.length > 0)
                    return buf;
            }
        }

        var typeMap = {
            'undefined': function(key, value) {
                var buf = [];
                buf.push(key, "=&");
                return buf;
            },
            'function': defaultFilter,
            'object': defaultFilter
        };

        var buf = [];
        for (var key in o) {
            var v = o[key], k = encodeURIComponent(key);

            var filter = typeMap[typeof(v)];

            if (typeof(filter) != "undefined") {
                var result = filter(k, v);
                if (result !== undefined || result !== null) {
                    if (Iuppiter.isArray(result))
                        buf = buf.concat(result);
                    else
                        buf.push(result);
                }
            }
            else {
                ignore = false;
                if (ignoreAttrs) {
                    for (j = 0; j < ignoreAttrs.length; j++) {
                        if (key == ignoreAttrs[j]) {
                            ignore = true;
                            break;
                        }
                    }
                }
                if (ignore)
                    buf.push(k, "=", v, "&");
                else
                    buf.push(k, "=", encodeURIComponent(v), "&");
            }
        }
        buf.pop();
        return buf.join("");
    },

    /**
     * Convert the JavaScript syntax string to Python string.
     *
     * @param {String} str The JavaScript syntax string.
     * @return {String} Converted Python string.
     */
    toPython: function(str) {
        // Test case:
        // '{"a": true, "b": "fadfa", "c": [true, true, false], ' +
        // '"d": {"a": true, "b": "i have true love", ' +
        // '"c": "string=falseandstring=true"}}'

        // Test case 2:
        //'[{"url": "http://shopping.pchome.com.tw/?mod=store&func=style_show&SR_NO=BEAB07", "path": ' +
        // '"/0/0/0/4/0/0/2/4/0/0/0,/0/0/0/4/0/0/2/4/0/0/2,/0/0/0/4/0/0/2/4/0/0/4,' +
        // '/0/0/0/4/0/0/2/4/0/0/6", "lastModified": "06/23/2010 17:20:25", "frames": [false, false, false, ' +
        // 'false]}]'

        key = {
            'true': 'True',
            'false': 'False',
            'null': 'None'
        }

        replace = function(m, p1, p2, p3) {
            return p1 + key[p2] + p3;
        };

        return str.replace(/([:\[] ?)(true|false|null)()/g, replace)
                  .replace(/(,? )(true|false|null)(,)/g, replace)
                  .replace(/(\]|, )(true|false|null)(\])/g, replace);
    },

    /**
     * Set digital format string.
     *
     * @param {Integer} digit The digital integer.
     * @param {Integer} fixed It will be set digital how many digital integer.
     * @return {String} The digital format string.
     */
    digitalFormat: function (digit, fixed) {
        if (!fixed)
            fixed = 4;
        if (digit == 0) {
            var r = '';
            for (; i < fixed - 1; i++)
                r += '0'
            return r;
        }

        var i = 0, dividend = '1';
        for (; i < fixed - 1; i++)
            dividend += '0';

        if (digit / dividend >= 1.0)
            return digit;

        return '0' + digitalFormat(digit, fixed - 1);
    }

});

/**
 * The browser is shortcut from Iuppiter.getBrowser.
 */
Iuppiter.browser = Iuppiter.getBrowser();

/**
 * The document src is used for temperature from the HTML of the current page.
 */
Iuppiter.documentSrc = null;

/**
 * Script loader for loading cross domain service by script tag.
 *
 * @param {Object} config Sameple: {
 *                            jWindow: window,
 *                            transes: [{
 *                                *url: 'xxxxx',
 *                                id: 'xxx',
 *                                checkNamespace: 'xxx',
 *                                clearPrevious: true,
 *                                params: {aa: 'xx', bb: 'xx'},
 *                                form: formDom,
 *                                nocache: true,
 *                                hasCallback: true,
 *                                removable: true
 *                            }]}
 *                            "*" (Required)
 * @note Please note that 'hasCallback' means this callback function is 
 *       triggered by server while script is loading. If 'hasCallback' = false,
 *       the callback function you give still called, but it is called after
 *       script is on ready (checkNamespace is found).
 */
Iuppiter.ScriptLoader = function(config) {
    this.timeout = 30000;
    this.callbackParam = "callback";
    this.autoAbort = false;
    this.autoRemoveLoadingMask = true;

    Iuppiter.expand(this, config);

    if (typeof(this.jWindow) == "undefined")
        this.jWindow = window;

    this.jDocument = this.jWindow.document;
};

Iuppiter.ScriptLoader.prototype = {

    /**
     * Set transes config.
     *
     * @param {Object} transes The transes config.
     */
    setTranses: function(transes) {
        this.transes = transes;
    },

    /**
     * Bind customize event.
     *
     * @param {String} eventName Event name string.
     * @param {Function} callback The callback function.
     */
    bind: function(eventName, callback) {
        //jQuery(this.jWindow).bind(eventName, callback);
    },

    /**
     * Private method that creates the loading mask.
     *
     * @param {Object} ops The message settings will display on the loading
     *                     mask. Example: {
     *                         id: 'xxxx',
     *                         icon: 'http://.....',
     *                         modal: false,
     *                         outer: true,
     *                         message: 'Loading...',
     *                         top: '0px',
     *                         left: '0px',
     *                         color: 'black',
     *                         backgroundColor: 'none'
     *                     }
     * @param {Function} doLoading It will be executed after the loading mask
     *                             created. Why we do this? Because rendering
     *                             created loading mask element causes race
     *                             condition by following code in FireFox.
     */
    createLoadingMask: function(ops, doLoading) {
        if (typeof ops === 'string')
            ops = {message: ops};

        var isFixed = !(Iuppiter.browser.isBorderBox ||
                        Iuppiter.browser.isLEIE6),
            jDocument = this.jDocument,
            mercuriusEl = jDocument.getElementById('mercurius'), loading,
            overlay, defaultOps = {
                id: 'loadingMask',
                icon: null,
                modal: false,
                outer: false,
                message: 'Loading...',
                top: (this.jWindow.mercuriusToolbar) ? "29px" : "0",
                left: '0px'
            };

        Iuppiter.expand(defaultOps, ops);

        if (defaultOps.modal) {
            overlay = jDocument.createElement('DIV');
            overlay.id = 'overlay';
            overlay.align = 'left';
            if (isFixed) {
                overlay.style.position = 'fixed';
                overlay.style.width = '100%';
                overlay.style.height = '100%';
            }
            else {
                overlay.style.position = 'absolute';

                var de = jDocument.documentElement,
                    ow = !Iuppiter.browser.isLEIE6 ? 21 : 0,
                    oh = !Iuppiter.browser.isLEIE6 ? 3.5 : 0,
                    body = jDocument.body,
                    loff = (de.scrollLeft || body.scrollLeft) -
                           (de.clientLeft || 0),
                    toff = (de.scrollTop || body.scrollTop) -
                           (de.clientTop || 0),
                    box = jDocument.documentElement.getBoundingClientRect(),
                    l = box.left + loff, t = box.top + toff,
                    r = l + Math.max(jDocument.body.clientWidth,
                                     jDocument.body.scrollWidth,
                                     jDocument.documentElement.scrollWidth),
                    b = t + Math.max(jDocument.body.clientHeight,
                                     jDocument.body.scrollHeight,
                                     jDocument.documentElement.scrollHeight),
                    width = r - l - ow, height = b - t - oh;

                // Negative values are not allowed in IE.
                if (width < 0) width = 0;
                if (height < 0) height = 0;
                overlay.style.width = width + 'px';
                overlay.style.height = height + 'px';
            }
            overlay.style.left = '0px';
            overlay.style.top = '0px';
            overlay.style.zIndex = '2147483647';
            overlay.style.padding = '0px';
            overlay.style.margin = '0px';
            overlay.style.textAlign = 'left';
            overlay.style.display = 'inline';
            overlay.style.verticalAlign = 'middle';
            overlay.style.backgroundColor = '#AAAAAA';
            if (Iuppiter.browser.msie) {
                overlay.style.filter = 'alpha(opacity: 30)';
            }
            else {
                overlay.style.opacity = '0.3';
                overlay.style.Mozopacity = '0.3';
            }
        }

        this.loadingMask = jDocument.getElementById(defaultOps.id) ||
                           jDocument.createElement("div");
        this.loadingMask.id = defaultOps.id;
        this.loadingMask.style.position = 'static';

        loading = jDocument.createElement('div');
        loading.id = 'nuwa_loading';
        this.loadingMask.appendChild(loading);
        // this.loading.position = 'static';
        if (defaultOps.icon) {
            var img = jDocument.createElement('img');
            img.src = defaultOps.icon;
            img.alt = defaultOps.message;
            img.style.display = 'inline';
            img.style.verticalAlign = 'middle';
            img.style.paddingRight = '10px';
            loading.appendChild(img);
        }

        if ((!defaultOps.icon && defaultOps.message) ||
            (defaultOps.outer && defaultOps.message))
            loading.appendChild(jDocument.createTextNode(defaultOps.message));

        loading.style.cssText =
            "font:normal 14px arial; font-weight:bold;" +
            "padding:2px; padding-left:6px;" +
            "z-index:3000009; display:inline; position:" +
            (!isFixed ? "absolute;" : "fixed;");

        loading.style.top = defaultOps.top;
        loading.style.left = defaultOps.left;

        loading.style.color = defaultOps.color || 'black';//loading.style.color;
        loading.style.backgroundColor = defaultOps.backgroundColor ||
                                        loading.style.backgroundColor;
                                        // #007F0E;
        loading.style.verticalAlign = 'middle';

        if (!mercuriusEl) {
            mercuriusEl = jDocument.createElement('div');
            mercuriusEl.id = 'mercurius';
            jDocument.body.appendChild(mercuriusEl);
        }

        if (overlay)
            this.loadingMask.appendChild(overlay);
        mercuriusEl.appendChild(this.loadingMask);

        if (doLoading) {
            setTimeout(doLoading, 0);
        }
    },

    /**
     * Remove loading mask.
     */
    removeLoadingMask: function() {
        if (this.loadingMask && this.loadingMask.parentNode) {
            this.loadingMask.parentNode.removeChild(this.loadingMask);
            try {
                delete this.loadingMask;
            }
            catch (e) {
                if (Iuppiter.debug)
                    alert(e.message);
            }
        }
    },

    /**
     * Load data from the configured URL.
     *
     * @param {Function} It will do callback with some script loaded or it will
     *                   do callback until scripts loaded.
     * @param {Object} message The message settings will display on the loading
     *                 mask.
     * @return {ScriptLoader} The self of the ScriptLoader.
    */
    load: function(callback, message) {
        //jQuery(this.jWindow).trigger("beforeload", this);
        if (message)
            this.createLoadingMask(message);

        var transId = new Date().getTime();
        this.transes.cb = "callback" + transId;

        this.transes.callback = undefined;

        var conn = this;

        this.jWindow[this.transes.cb] = function(o){
            conn.handleResponse(o, conn.transes);
        };

        for (var i = 0; i < this.transes.length; i++) {
            var trans = this.transes[i];

            if (trans.params) {
                trans.url += (trans.url.indexOf("?") != -1 ? "&" : "?") +
                Iuppiter.jsonToQueryString(trans.params);
            }

            if (trans.form) {
                trans.url += (trans.url.indexOf("?") != -1 ? "&" : "?") +
                Iuppiter.jsonToQueryString(Iuppiter.formToJson(trans.form));
            }

            if (trans.nocache) {
                trans.url += trans.url.indexOf("?") != -1 ? "&" : "?";
                trans.url += "dummy=" + new Date().getTime();
            }

            if (trans.hasCallback) {
                this.transes.callback = callback;
                trans.url += trans.url.indexOf("?") != -1 ? "&" : "?";
                trans.url += this.callbackParam + '=' + this.transes.cb;
            }

            if (trans.id)
                trans.id += '_' + transId;
            /*else
                trans.id = '_' + transId;*/

            if (trans.checkNamespace && trans.clearPrevious) {
                eval('this.jWindow.' + trans.checkNamespace + ' = undefined;');
                try {
                    eval('delete this.jWindow.' + trans.checkNamespace + ';');
                }
                catch(e) {
                    if (Iuppiter.debug)
                        alert(e.message);
                }
            }

            if (this.autoAbort !== false) {
                this.abort();
            }
        }

        this.transes.timeoutId = this.handleFailure.setIuppiterDelay(
                                            this.timeout, this, [this.transes]);

        if (this.transes.callback) {
            Iuppiter.loadScript(this.jWindow, this.transes);
        }
        else {
            Iuppiter.loadScript(this.jWindow, this.transes, function(){
                if (callback)
                    callback();
                conn.handleResponse(null, conn.transes);
            });
        }

        return this;
    },

    /**
     * Private method that checks whether the script is loading or not.
     *
     * @return {Boolean} True if the script is still loading.
     */
    isLoading: function() {
        return this.transes ? true : false;
    },

    /**
     * Private method that aborts the current server request.
     */
    abort: function() {
        if (this.isLoading())
            this.destroyTrans(this.transes);
    },

    /**
     * Private method that destroys transactions.
     *
     * @param {Object} transes Transactions config.
     * @param {Boolean} isLoaded The script is loading or not.
     */
    destroyTrans: function(transes, isLoaded) {
        for (var i = 0; i < transes.length; i++) {
            if (transes[i].removable) {
                var scriptEl = this.jDocument.getElementById(transes[i].id);
                if (scriptEl && scriptEl.parentNode)
                    scriptEl.parentNode.removeChild(scriptEl);
            }
        }

        clearTimeout(transes.timeoutId);
        if (isLoaded) {
            this.jWindow[transes.cb] = undefined;
            try {
                delete this.jWindow[transes.cb];
            }
            catch(e) {
                if (Iuppiter.debug)
                    alert(e.message);
            }
        }
        else {
            // if hasn't been loaded, wait for load to remove it to prevent
            // script error.
            this.jWindow[transes.cb] = function() {
                this.jWindow[transes.cb] = undefined;
                try {
                    delete this.jWindow[transes.cb];
                }
                catch(e) {
                    if (Iuppiter.debug)
                        alert(e.message);
                }
            };
        }
    },

    /**
     * Private method that handles response processed.
     *
     * @param {Object} o The results of the server response.
     * @param {Object} transes Transactions config.
     */
    handleResponse: function(o, transes) {
        this.transes = false;
        try {
            if (transes.callback) {
                transes.callback(this, o, true);
                if (this.autoRemoveLoadingMask)
                    this.removeLoadingMask();
            }
            this.destroyTrans(transes, true);
        }
        catch(e) {
            //jQuery(this.jWindow).trigger("loadexception", this, o, e);
            this.destroyTrans(transes, false);
            return;
        }
        //jQuery(this.jWindow).trigger("load", this, o);
    },

    /**
     * Private method that handles failure processed.
     *
     * @param {Object} transes Transactions config.
     */
    handleFailure: function(transes) {
        if (!this.isLoading())
            return;
        this.transes = false;

        if (transes.callback)
            transes.callback(this, undefined, false);

        this.destroyTrans(transes, false);
        //jQuery(this.jWindow).trigger("loadexception", this, null);
    }
};

/**
 * Abstract base class that provides a common interface for publishing events.
 */
Iuppiter.Observable = function() {
    this.observers = {};
};

Iuppiter.Observable.prototype = {

    /**
     * Register a delegate function to observer list.
     *
     * @param {Function} delegate The delegate function.
     */
    register: function(delegate) {
        if (delegate.id && this.observers[delegate.id]) {
            alert("The delegate function has already registered, you fool!!");
            return;
        }

        delegate.id = '_' + new Date().getTime() +
                      Math.round(Math.random() * 100);
        for (key in this.observers) {
            if (key == delegate.id)
                delegate.id += Math.round(Math.random() * 100);
        }
        this.observers[delegate.id] = delegate;
    },

    /**
     * Remove the delegate function that has been registered.
     *
     * @param {Function} delegate The delegate function.
     */
    unregister: function(delegate) {
        if (!delegate.id) {
            alert("The delegate wasn't registered!!");
            return;
        }

        this.observers[delegate.id] = undefined;
        try {
            delete this.observers[delegate.id];
        }
        catch(e) {}
    },

    /**
     * Trigger all of registered observers.
     */
    triggerAll: function() {
        for (key in this.observers) {
            this.observers[key]();
        }
    }
};

/**
 * Build in functions for cross browser.
 *
 * @param {Window} oWindow The window object.
 */
Iuppiter.builtIn = function(oWindow) {

    /**
     * Check whether or not the current number is within a desired range.
     * If the number is already within the range it is returned, otherwise the
     * min or max value is returned depending on which side of the range is
     * exceeded. Note that this method returns the constrained value but does
     * not change the current number.
     *
     * @param {Number} min The minimum number in the range.
     * @param {Number} max The maximum number in the range.
     * @return {Number} The constrained value if outside the range, otherwise
     *                  the current value.
     */
    oWindow.Number.prototype.constrain = function(min, max) {
        return Math.min(Math.max(this, min), max);
    };

    // String does not exist if now is in IE's iframe window.
    if (typeof oWindow.String != 'undefined') {

        /**
         * Extends String's build-in function to strip blink spaces at the
         * start and the end.
         *
         * @return {String} Striped the string value.
         */
        oWindow.String.prototype.trim = function(){
            return this.replace(/^\s+|\s+$/g, "");
        };
    }

    if (!Iuppiter.browser.msie && typeof(oWindow.HTMLElement) != "undefined" &&
        !oWindow.opera) {

        oWindow.HTMLElement.prototype.__defineGetter__("outerHTML", function() {
            var a = this.attributes, str = "<" + this.tagName, i = 0;

            for (; i < a.length; i++)
                if (a[i].specified)
                    str += " " + a[i].name + '="' + a[i].value + '"';

            if (!this.canHaveChildren)
                return str + " />";

            return str + ">" + this.innerHTML + "</" + this.tagName + ">";
        });

        oWindow.HTMLElement.prototype.__defineSetter__("outerHTML", function(s){
            var r = this.ownerDocument.createRange();
            r.setStartBefore(this);

            var df = r.createContextualFragment(s);
            this.parentNode.replaceChild(df, this);

            return s;
        });

        oWindow.HTMLElement.prototype.__defineGetter__("canHaveChildren",
                                                       function() {
            return !/^(area|base|basefont|col|frame|hr|img|br|input|isindex|link|meta|param)$/
                    .test(this.tagName.toLowerCase());
        });
    }
};

Iuppiter.builtIn(window);

/**
 * Convert string value to a byte array.
 *
 * @param {String} input The input string value.
 * @return {Array} A byte array from string value.
 */
function byteArray(input) {
    var b = [], i, unicode;
    for(i = 0; i < input.length; i++) {
        unicode = input.charCodeAt(i);
        // 0x00000000 - 0x0000007f -> 0xxxxxxx
        if (unicode <= 0x7f) {
            b.push(unicode);
        // 0x00000080 - 0x000007ff -> 110xxxxx 10xxxxxx
        } else if (unicode <= 0x7ff) {
            b.push((unicode >> 6) | 0xc0);
            b.push((unicode & 0x3F) | 0x80);
        // 0x00000800 - 0x0000ffff -> 1110xxxx 10xxxxxx 10xxxxxx
        } else if (unicode <= 0xffff) {
            b.push((unicode >> 12) | 0xe0);
            b.push(((unicode >> 6) & 0x3f) | 0x80);
            b.push((unicode & 0x3f) | 0x80);
        // 0x00010000 - 0x001fffff -> 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
        } else {
            b.push((unicode >> 18) | 0xf0);
            b.push(((unicode >> 12) & 0x3f) | 0x80);
            b.push(((unicode >> 6) & 0x3f) | 0x80);
            b.push((unicode & 0x3f) | 0x80);
        }
    }

    return b;
}

/**
 * Base64 Class.
 * Reference: http://code.google.com/p/javascriptbase64/
 *            http://www.stringify.com/static/js/base64.js
 * They both under MIT License.
 */
Iuppiter.Base64 = {

    /// Encoding characters table.
    CA: "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",

    /// Encoding characters table for url safe encoding.
    CAS: "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_",

    /// Decoding reference table.
    IA: new Array(256),

    /// Decoding reference table for url safe encoded string.
    IAS: new Array(256),

    /**
     * Constructor.
     */
    init: function(){
        /// Initialize variables for Base64 namespace.
        var i;

        for (i = 0; i < 256; i++) {
            Iuppiter.Base64.IA[i] = -1;
            Iuppiter.Base64.IAS[i] = -1;
        }

        for (i = 0, iS = Iuppiter.Base64.CA.length; i < iS; i++) {
            Iuppiter.Base64.IA[Iuppiter.Base64.CA.charCodeAt(i)] = i;
            Iuppiter.Base64.IAS[Iuppiter.Base64.CAS.charCodeAt(i)] = i;
        }

        Iuppiter.Base64.IA['='] = Iuppiter.Base64.IAS['='] = 0;
    },

    /**
     * Encode base64.
     *
     * @param {Array|String} input A byte array or a string.
     * @param {Boolean} urlsafe True if you want to make encoded string is url
     *                          safe.
     * @return {String} Encoded base64 string.
     */
    encode: function(input, urlsafe, assignWindow) {

        var ca, dArr, sArr, sLen,
            eLen, dLen, s, d, left,
            i;

        if(urlsafe)
            ca = Iuppiter.Base64.CAS;
        else
            ca = Iuppiter.Base64.CA;

        if(input.constructor == Array)
            sArr = input;
        else if (typeof assignWindow != 'undefined' && input.constructor == assignWindow.Array)
            sArr = input;
        else
            sArr = byteArray(input);

        sLen = sArr.length;

        eLen = (sLen / 3) * 3;              // Length of even 24-bits.
        dLen = ((sLen - 1) / 3 + 1) << 2;   // Length of returned array
        dArr = new Array(dLen);

        // Encode even 24-bits
        for (s = 0, d = 0; s < eLen;) {
            // Copy next three bytes into lower 24 bits of int, paying attension to sign.
            i = (sArr[s++] & 0xff) << 16 | (sArr[s++] & 0xff) << 8 |
                (sArr[s++] & 0xff);

            // Encode the int into four chars
            dArr[d++] = ca.charAt((i >> 18) & 0x3f);
            dArr[d++] = ca.charAt((i >> 12) & 0x3f);
            dArr[d++] = ca.charAt((i >> 6) & 0x3f);
            dArr[d++] = ca.charAt(i & 0x3f);
        }

        // Pad and encode last bits if source isn't even 24 bits.
        left = sLen - eLen; // 0 - 2.
        if (left > 0) {
            // Prepare the int
            i = ((sArr[eLen] & 0xff) << 10) |
                 (left == 2 ? ((sArr[sLen - 1] & 0xff) << 2) : 0);

            // Set last four chars
            dArr[dLen - 4] = ca.charAt(i >> 12);
            dArr[dLen - 3] = ca.charAt((i >> 6) & 0x3f);
            dArr[dLen - 2] = left == 2 ? ca.charAt(i & 0x3f) : '=';
            dArr[dLen - 1] = '=';
        }
        return dArr.join("");
    },

    /**
     * Decode base64 encoded string or byte array.
     *
     * @param {Array|String} input A byte array or encoded string.
     * @param {Object} urlsafe True if the encoded string is encoded by urlsafe.
     * @return {Array|String} A decoded byte array or string depends on input
     *                        argument's type.
     */
    decode: function(input, urlsafe) {
        var ia, dArr, sArr, sLen, bytes,
            sIx, eIx, pad, cCnt, sepCnt, len,
            d, cc, left,
            i, j, r;

        if(urlsafe)
            ia = Iuppiter.Base64.IAS;
        else
            ia = Iuppiter.Base64.IA;

        if(input.constructor == Array) {
            sArr = input;
            bytes = true;
        }
        else {
            sArr = byteArray(input);
            bytes = false;
        }

        sLen = sArr.length;

        sIx = 0;
        eIx = sLen - 1;    // Start and end index after trimming.

        // Trim illegal chars from start
        while (sIx < eIx && ia[sArr[sIx]] < 0)
            sIx++;

        // Trim illegal chars from end
        while (eIx > 0 && ia[sArr[eIx]] < 0)
            eIx--;

        // get the padding count (=) (0, 1 or 2)
        // Count '=' at end.
        pad = sArr[eIx] == '=' ? (sArr[eIx - 1] == '=' ? 2 : 1) : 0;
        cCnt = eIx - sIx + 1;   // Content count including possible separators
        sepCnt = sLen > 76 ? (sArr[76] == '\r' ? cCnt / 78 : 0) << 1 : 0;

        // The number of decoded bytes
        len = ((cCnt - sepCnt) * 6 >> 3) - pad;
        dArr = new Array(len);       // Preallocate byte[] of exact length

        // Decode all but the last 0 - 2 bytes.
        d = 0;
        for (cc = 0, eLen = (len / 3) * 3; d < eLen;) {
            // Assemble three bytes into an int from four "valid" characters.
            i = ia[sArr[sIx++]] << 18 | ia[sArr[sIx++]] << 12 |
                ia[sArr[sIx++]] << 6 | ia[sArr[sIx++]];

            // Add the bytes
            dArr[d++] = (i >> 16) & 0xff;
            dArr[d++] = (i >> 8) & 0xff;
            dArr[d++] = i & 0xff;

            // If line separator, jump over it.
            if (sepCnt > 0 && ++cc == 19) {
                sIx += 2;
                cc = 0;
            }
        }

        if (d < len) {
            // Decode last 1-3 bytes (incl '=') into 1-3 bytes
            i = 0;
            for (j = 0; sIx <= eIx - pad; j++)
                i |= ia[sArr[sIx++]] << (18 - j * 6);

            for (r = 16; d < len; r -= 8)
                dArr[d++] = (i >> r) & 0xff;
        }

        if(bytes) {
            return dArr;
        }
        else {
            for(i = 0; i < dArr.length; i++)
                dArr[i] = String.fromCharCode(dArr[i]);

            return dArr.join('');
        }
    }
};

Iuppiter.Base64.init();

// regex cache
var patterns = {
    HYPHEN: /(-[a-z])/i, // to normalize get/setStyle
    ROOT_TAG: /^body|html$/i, // body for quirks mode, HTML for standards,
    OP_SCROLL: /^(?:inline|table-row)$/i
};

/**
 * Region class for getting the coordinates information.
 */
Iuppiter.Base.Region = function(top, right, bottom, left) {
    this.top = top;
    this[1] = this.top;
    this.right = right;
    this.bottom = bottom;
    this.left = left;
    this[0] = this.left;
};

Iuppiter.Base.Region.prototype = {

    /**
     * Check whether it contains other region or not.
     *
     * @param {Region} region The other region object.
     * @return {Boolean} True if it contains the region, otherwise else.
     */
    contains: function(region) {
        return region.left >= this.left && region.right <= this.right &&
               region.top >= this.top && region.bottom <= this.bottom;
    },

    /**
     * Check whether it was surrounded by the other region or not.
     *
     * @param {Region} region The other region object.
     * @return {Boolean} True if it was surrounded by the other region,
     *                   otherwise else.
     */
    beSurrounded: function(region) {
        return region.right >= this.right && region.left <= this.left &&
               region.bottom >= this.bottom && region.top <= this.top;
    },

    /**
     * Get the measure of the area.
     *
     * @return {Number} The measure of the area.
     */
    getArea: function() {
        return (this.bottom - this.top) * (this.right - this.left);
    },

    /**
     * Get the intersection region.
     *
     * @param {Region} region The other region object for calculating the
     *                        intersection region.
     * @return {Region} The intersection region if the two regions intersected,
     *                  else return null.
     */
    intersect: function(region){
        var t = Math.max(this.top, region.top),
            r = Math.min(this.right, region.right),
            b = Math.min(this.bottom, region.bottom),
            l = Math.max(this.left, region.left);

        if (b >= t && r >= l)
            return new Iuppiter.Base.Region(t, r, b, l);

        return null;
    },

    /**
     * Get the union region.
     *
     * @param {Region} region The other region object for calculating the union
     *                        region.
     * @return {Region} The union region.
     */
    union: function(region){
        var t = Math.min(this.top, region.top),
            r = Math.max(this.right, region.right),
            b = Math.max(this.bottom, region.bottom),
            l = Math.min(this.left, region.left);

        return new Iuppiter.Base.Region(t, r, b, l);
    },

    /**
     * Constrain the area in the comparative region.
     *
     * @param {Region} r The comparative region.
     * @return {Region} The constrained region.
     */
    constrainTo: function(r){
        this.top = this.top.constrain(r.top, r.bottom);
        this.bottom = this.bottom.constrain(r.top, r.bottom);
        this.left = this.left.constrain(r.left, r.right);
        this.right = this.right.constrain(r.left, r.right);
        return this;
    }
};

/**
 * Utility class for working with HTML DOM.
 */
Iuppiter.Base.Dom = {

    /**
     * Get HTML DOM elements by name.
     * fix IE problem.
     *
     * @param {HTMLElement} el The HTML DOM element.
     * @param {String} name The name of the HTML DOM element's attribute.
     * @param {String} tagName The tag name of the HTML DOM element(Optional).
     * @return {Array} Matched HTML DOM elements's array.
     */
    getElementsByName: function(el, name, tagName) {
        var oDocument = el.ownerDocument,
            returns = oDocument.getElementsByName(name), i;

        if (returns.length > 0 || !Iuppiter.browser.msie)
            return returns;

        returns = new Array();

        // #FIXME: 'div'? it shall be '*'.....?
        var e = oDocument.getElementsByTagName((tagName) ? tagName : 'div');

        for (i = 0; i < e.length; i++) {
            if (e[i].getAttribute("name") == name) {
                returns[returns.length] = e[i];
            }
        }

        return returns;
    },

    /**
     * Make contained line with the div element.
     *
     * @param {Document} oDocument The HTML DOM document object.
     * @param {Number} lTop The top pixels of the created div element.
     * @param {Number} lLeft The left pixels of the created div element.
     * @param {Number} lLength The width pixels of the created div element.
     * @param {Boolean} bVertical The created div element whether is Vertical or
     *                  not.
     * @param {String} sName The name of the created div element.
     */
    makeLine: function(oDocument, lTop, lLeft, lLength, bVertical, sName) {
        var oLineDiv = null;
        if (!sName) {
            sName = 'pageSelectorLineElement';
        }
        oLineDiv = oDocument.createElement('DIV');
        oLineDiv.setAttribute('name', sName);
        
        // We use 'getElementsByName' can not find any elements in IE.
        // So we assign the same value of 'id' attribute to fix this problem.
        oLineDiv.id = sName;
        
        oLineDiv.style.position = 'absolute';
        oLineDiv.style.border = '#0475b1 2px solid';
        oLineDiv.align = 'left';
        if (bVertical) {
            oLineDiv.style.height = lLength + 'px';
            oLineDiv.style.width = '0';
            oLineDiv.style.borderWidth = '0px 2px 0px 2px';
        }
        else {
            // The line height must be set the same with the height if client
            // used IE 6 or IE 7 in quirk mode.
            oLineDiv.style.lineHeight = oLineDiv.style.height = '0';
            oLineDiv.style.width = lLength + 'px';
            oLineDiv.style.borderWidth = '2px 0px 2px 0px';
        }
        oLineDiv.style.display = 'block';
        oLineDiv.style.zIndex = '99998';
        oLineDiv.style.padding = '0px';
        oLineDiv.style.margin = '0px';
        oLineDiv.style.left = (lLeft) + 'px';
        oLineDiv.style.top = (lTop) + 'px';
        // The font size in here must be set 0 if client used IE 6 or IE 7
        // in quirk mode.
        oLineDiv.style.fontSize = '0';
        oLineDiv.style.clear = 'both';

        oDocument.body.appendChild(oLineDiv);
    },

    /**
     * Get the window object's selection object.
     *
     * @param {Window} oWindow The window object.
     * @return The selection object's string format value.
     */
    getSelection: function(oWindow) {
        return (oWindow.getSelection) ?
            oWindow.getSelection().toString() :
                ((oWindow.document.getSelection)?
                 oWindow.document.getSelection().toString() :
                 oWindow.document.selection.createRange().text);
    },

    /**
     * Reference: YUI library.
     *
     * @param {String} property The property of the HTML DOM element.
     * @return {String}
     */
    toCamel: function(property){
        if (!patterns.HYPHEN.test(property)) {
            return property; // no hyphens
        }

        if (propertyCache[property]) { // already converted
            return propertyCache[property];
        }

        var converted = property;

        while (patterns.HYPHEN.exec(converted)) {
            converted = converted.replace(RegExp.$1,
                                            RegExp.$1.substr(1).toUpperCase());
        }

        propertyCache[property] = converted;
        return converted;
        // cant use function as 2nd arg yet due to safari bug
        // return property.replace(/-([a-z])/gi,
        //     function(m0, m1) {return m1.toUpperCase()})
    },

    /**
     * Normalize currentStyle and ComputedStyle.
     * Reference: YUI library.
     *
     * @param {HTMLElement} el An actual DOM reference.
     * @param {String} property The style property whose value is returned.
     * @return {String|Array} The current value of the style property for the
     *                        element(s).
     */
    getStyle: function(el, property) {
        // W3C DOM method
        if (document.defaultView && document.defaultView.getComputedStyle) {
            var value = null;

            // fix reserved word
            if (property == 'float') {
                property = 'cssFloat';
            }

            var computed = el.ownerDocument.defaultView
                           .getComputedStyle(el, '');
            // test computed before touching for safari
            if (computed)
                value = computed[this.toCamel(property)];

            return el.style[property] || value;
        }
        // IE method
        else if (document.documentElement.currentStyle &&
                 Iuppiter.browser.msie) {
            switch (this.toCamel(property)) {
                case 'opacity':// IE opacity uses filter
                    var val = 100;
                    try { // will error if no DXImageTransform
                        val = el.filters['DXImageTransform.Microsoft.Alpha']
                              .opacity;
                    }
                    catch (e) {
                        try { // make sure its in the document
                            val = el.filters('alpha').opacity;
                        }
                        catch (e) {
                        }
                    }
                    return val / 100;
                case 'float': // fix reserved word
                    property = 'styleFloat'; // fall through
                default:
                    // test currentStyle before touching
                    var value = el.currentStyle ?
                                el.currentStyle[property] : null;
                    return (el.style[property] || value);
            }
        }
        // default to inline only
        else {
            return el.style[property];
        }
    },

    /**
     * Wrapper for setting style properties of HTMLElements.
     * Normalize "opacity" across modern browsers.
     * Reference: YUI library.
     *
     * @param {HTMLElement} el Accepts an actual DOM reference.
     * @param {String} property The style property to be set.
     * @param {String} val The value to apply to the given property.
     */
    setStyle: function(el, property, val) {
        if (Iuppiter.browser.msie) {
            property = this.toCamel(property);

            switch (property) {
                case 'opacity':
                    // in case not appended
                    if (typeof el.style.filter === 'string') {
                        el.style.filter = 'alpha(opacity=' + val * 100 + ')';

                        if (!el.currentStyle || !el.currentStyle.hasLayout)
                            // when no layout or cant tell
                            el.style.zoom = 1;
                    }
                    break;
                case 'float':
                    property = 'styleFloat';
                default:
                    el.style[property] = val;
            }
        }
        else {
            property = this.toCamel(property);
            if (property == 'float')
                property = 'cssFloat';
            el.style[property] = val;
        }
    },

    /**
     * Get the current position of the element based on page coordinates.
     * Reference: YUI library.
     *
     * @param {HTMLElement} el The HTML DOM element.
     * @return {Array} The XY position of a dictionary.
     */
    getXY: function(el) {
        // has to be part of document to have pageXY
        if ((el.parentNode === null || el.offsetParent === null ||
             this.getStyle(el, 'display') == 'none') &&
            el != el.ownerDocument.body) {
            return {0: 0, 1: 0, left: 0, top: 0};
        }

        if (document.documentElement.getBoundingClientRect) {
            // IE
            //return function(el) {
            // http://svn.mochikit.com/mochikit/trunk/MochiKit/Style.js
            /*
              The IE shortcut can be off by two. We fix it. See:
              http://msdn.microsoft.com/workshop/author/dhtml/reference/methods/
              getboundingclientrect.asp
            */
            /*
            var d = MochiKit.DOM._document;
            var de = d.documentElement;
            var b = d.body;
            box = elem.getBoundingClientRect();

            c.x += box.left +
            (de.scrollLeft || b.scrollLeft) - (de.clientLeft || 0);
            c.y += box.top +
            (de.scrollTop || b.scrollTop) - (de.clientTop || 0);
            */
            var box = el.getBoundingClientRect(),
                rootNode = el.ownerDocument,
                l = box.left + Math.max(
                    rootNode.documentElement.scrollLeft,
                    rootNode.body.scrollLeft),
                r = box.top + Math.max(
                    rootNode.documentElement.scrollTop,
                    rootNode.body.scrollTop);
            return {
                0: l,
                1: r,
                left: l,
                top: r
            };
            //};
        }
        else {
            // manually calculate by crawling up offsetParents
            // return function(el) {
            var pos = {
                left: el.offsetLeft,
                top: el.offsetTop
            };
            var parentNode = el.offsetParent;

            // safari: subtract body offsets if el is abs (or any offsetParent),
            // unless body is offsetParent
            var accountForBody = (Iuppiter.browser.safari &&
                this.getStyle(el, 'position') == 'absolute' &&
                el.offsetParent == el.ownerDocument.body);

            if (parentNode != el) {
                while (parentNode) {
                    pos.left += parentNode.offsetLeft;
                    pos.top += parentNode.offsetTop;
                    if (!accountForBody &&
                        isSafari &&
                        this.getStyle(parentNode, 'position') == 'absolute') {
                        accountForBody = true;
                    }
                    parentNode = parentNode.offsetParent;
                }
            }

            if (accountForBody) { //safari doubles in this case
                pos.left -= el.ownerDocument.body.offsetLeft;
                pos.top -= el.ownerDocument.body.offsetTop;
            }
            parentNode = el.parentNode;

            // account for any scrolled ancestors
            while (parentNode.tagName &&
                   !patterns.ROOT_TAG.test(parentNode.tagName)) {
                if (parentNode.scrollTop || parentNode.scrollLeft) {
                    // work around opera inline/table scrollLeft/Top bug
                    // (false reports offset as scroll)
                    if (!patterns.OP_SCROLL.test(
                                        this.getStyle(parentNode, 'display'))) {
                        // opera inline-block misreports when visible
                        if (!Iuppiter.browser.opera ||
                            this.getStyle(parentNode, 'overflow') !== 'visible') {
                            pos.left -= parentNode.scrollLeft;
                            pos.top -= parentNode.scrollTop;
                        }
                    }
                }

                parentNode = parentNode.parentNode;
            }
            pos[0] = pos.left; pos[1] = pos.top;

            return pos;
        //};
        }

    },

    /**
     * Set the position of an HTML element in page coordinates, regardless of
     * how the element is positioned.
     * The element(s) must be part of the DOM tree to have page coordinates
     * (display:none or elements not appended return false).
     * Reference: YUI library.
     *
     * @param {HTMLElement} el Accepts an actual DOM reference.
     * @param {Array} pos Contains X & Y values for new position
     *                    (coordinates are page-based).
     */
    setXY: function(el, pos) {
        var xy = pos, pos = this.getStyle(el, 'position'),
            // assuming pixels; if not we will have to retry
            delta = [
                parseInt(this.getStyle(el, 'left'), 10),
                parseInt(this.getStyle(el, 'top'), 10)
            ], currentXY, newXY;

        // default to relative
        if (pos == 'static') {
            pos = 'relative';
            this.setStyle(el, 'position', pos);
        }

        currentXY = this.getXY(el);

        // has to be part of doc to have xy.
        if (!xy || currentXY === false)
            return false;

        // in case of 'auto'
        if (isNaN(delta[0]))
            delta[0] = (pos == 'relative') ? 0 : el.offsetLeft;
        // in case of 'auto'
        if (isNaN(delta[1]))
            delta[1] = (pos == 'relative') ? 0 : el.offsetTop;

        // from setX
        if (xy[0] !== null)
            this.setStyle(el, 'left', xy[0] - currentXY[0] + delta[0] + 'px');

        // from setY
        if (xy[1] !== null)
            this.setStyle(el, 'top', xy[1] - currentXY[1] + delta[1] + 'px');

        /*
        if (!noRetry) {
            newXY = this.getXY(el);

            // if retry is true, try one more time if we miss
            if ((xy[0] !== null && newXY[0] != xy[0]) ||
                (xy[1] !== null && newXY[1] != xy[1]))
                this.setXY(el, xy, true);
        }
        */
    },

    /**
     * Get the region of the HTML DOM element.
     * Reference: YUI library.
     *
     * @param {HTMLElement} el The HTML DOM element.
     * @return {Region} The region object of the HTML element on the document
     *                  object.
     */
    getRegion: function(el) {
        var p = this.getXY(el);

        return new Iuppiter.Base.Region(p.top, p.left + el.offsetWidth,
                                         p.top + el.offsetHeight, p.left);
    },

    /**
     * Check the overlapping region of the HTML element.
     *
     * @param {Region} region The region of the HTML element.
     * @param {HTMLElement} el The HTML element object.
     * @return {Region|Boolean} The inside region if the region of the inside
     *                          HTML DOM element, else false.
     */
    checkForOverlappingRegion: function(region, el) {
        var insideEl = el.previousSibling, insideRegion = null,
        oReturn = null;

        if (!insideEl)
            return false;

        if (insideEl.nodeType != 1)
            return this.checkForOverlappingRegion(region, insideEl);

        /*
        this.siblingsChecked++;
        if (this.siblingsChecked > 20)
            return false;
        */

        function isOverlap(region, insideRegion) {
            return ((insideRegion.top < region.bottom &&
                     insideRegion.bottom > region.top) &&
                    (insideRegion.left < region.right &&
                     insideRegion.right > region.left)) &&
                   !((insideRegion.top <= region.top &&
                      insideRegion.bottom >= region.bottom) &&
                     (insideRegion.left <= region.left &&
                      insideRegion.right >= region.right))
        }

        insideRegion = this.getRegion(insideEl);
        if (isOverlap(region, insideRegion))
            return insideRegion;

        oReturn = this.checkForOverlappingRegion(region, insideEl);
        if (oReturn)
            return oReturn;

        for (i = 0; i < insideEl.childNodes.length; i++) {
            if (insideEl.childNodes[i].nodeType == 1) {
                insideRegion = this.getRegion(insideEl.childNodes[i]);
                if (isOverlap(region, insideRegion))
                    return insideRegion;
            }
        }

        return false;
    },

    /**
     * Get the window object by given URL. It will recursively go through all
     * frames of current window to find the window whose URL equals parameter
     * URL.
     *
     * @param {Window} oWindow Current window object.
     * @param {String} url The URL you want to find.
     * @return {Window} Window object whose href equals url parameter.
     */
    getWindow: function(oWindow, url) {
        var i = 0, iframeWindow;
        try {
            // There is a possibility that oWindow is cross domain, that will
            // cause security exception.
            if (oWindow.location.href == url)
                return oWindow;
            do {
                iframeWindow = this.getWindow(oWindow.frames[i++], url);
                if (iframeWindow)
                    return iframeWindow;
            } while(i < oWindow.frames.length);
        }
        catch (e) {
            if (Iuppiter.debug)
                alert(e.message);
            return null;
        }
        return null;
    },

    /**
     * Get the root HTML DOM window object from the given HTML DOM element.
     *
     * @param {HTMLElement} el An actual DOM reference.
     * @return {Window} Root window.
     */
    getRootWindow: function(el) {
        var win = (el.ownerDocument.defaultView ||
                   el.ownerDocument.parentWindow);
        if (win === win.parent)
            return win;
        return this.getRootWindow(win.document.body)
    },

    /**
     * Clean HTML DOM elements fragments.
     * There has a bug of Safari: the object's type is funtion in Safari, but
     * the bug is resolve in jQuery 1.4.
     * We changed by jQuery's function -- html.
     * Reference: jQuery Library.
     *
     * @param {String|Array} elems The HTML DOM elements.
     * @param {Document} context The activity document object.
     * @param {Object} fragment The document fragment object.
     * @return {Array} Not processed script tag.
     */
    clean: function(elems, context, fragment) {
        context = context || document;

        // !context.createElement fails in IE with an error but returns typeof
        // 'object'
        if (typeof context.createElement === "undefined")
            context = context.ownerDocument ||
                      context[0] && context[0].ownerDocument || document;

        // If a single string is passed in and it's a single tag
        // just do a createElement and skip the rest
        if (!fragment && elems.length === 1 && typeof elems[0] === "string") {
            var match = /^<(\w+)\s*\/?>$/.exec(elems[0]);
            if (match)
                return [context.createElement(match[1])];
        }

        var ret = [], scripts = [], div = context.createElement("div"),
            tags, wrap, d;

        Iuppiter.each(elems, function(i, elem) {
            if (typeof elem === "number")
                elem += '';

            if (!elem)
                return;

            // Convert HTML string into DOM nodes
            if (typeof elem === "string") {
                // Fix "XHTML"-style tags in all browsers
                elem = elem.replace(/(<(\w+)[^>]*?)\/>/g,
                    function(all, front, tag) { return tag.match(
                    /^(abbr|br|col|img|input|link|meta|param|hr|area|embed)$/i)
                        ? all : front + "></" + tag + ">";
                });

                // If elem contains script tag with defer attribute, it will be
                // executed twice.
                // (first: it is inserted into div because of defer, second:
                //  it is inserted into head).
                // We avoid that, so strip script tags first from elem to
                // insert it into div element.
                elem = elem.replace(
                    /<script(?:[^>]|\s)*?(?:src=('|")([^>]*?)\1(?:[^>]|\s)*?)?>((?:.|\s)*?)<\/script>/gi,
                    function(all, prefix, src, text) {
                        scripts.push({
                            src: src,
                            text: text
                        });
                        return '';
                    }
                );
                tempDiv = document.createElement("div");
                tempDiv.innerHTML = '   <link/><table></table>' +
                    '<a href="/a" style="color:red;float:left;opacity:.5;">a' +
                    '</a><select><option>text</option></select><object>' +
                    '<param/></object>';
                // Trim whitespace, otherwise indexOf won't work as expected
                tags = elem.replace(/^\s+/, "").substring(0, 10)
                           .toLowerCase();

                wrap =
                    // option or optgroup
                    !tags.indexOf("<opt") &&
                    [ 1, "<select multiple='multiple'>", "</select>" ] ||

                    !tags.indexOf("<leg") &&
                    [ 1, "<fieldset>", "</fieldset>" ] ||

                    tags.match(/^<(thead|tbody|tfoot|colg|cap)/) &&
                    [ 1, "<table>", "</table>" ] ||

                    !tags.indexOf("<tr") &&
                    [ 2, "<table><tbody>", "</tbody></table>" ] ||

                    // <thead> matched above
                    (!tags.indexOf("<td") || !tags.indexOf("<th")) &&
                    [ 3, "<table><tbody><tr>", "</tr></tbody></table>" ] ||

                    !tags.indexOf("<col") &&
                    [ 2, "<table><tbody></tbody><colgroup>", "</colgroup></table>" ] ||

                    // Make sure that link elements get serialized correctly by
                    // innerHTML. This requires a wrapper element in IE.
                    // IE can't serialize <link> and <script> tags normally
                    !(!!tempDiv.getElementsByTagName("link").length) &&
                    [ 1, "div<div>", "</div>" ] ||

                    [ 0, "", "" ];

                // Go to HTML and back, then peel off extra wrappers
                div.innerHTML = wrap[1] + elem + wrap[2];

                // Move to the right depth
                while (wrap[0]--)
                    div = div.lastChild;

                // Make sure that tbody elements aren't automatically inserted.
                // IE will insert them into empty tables.
                // Remove IE's autoinserted <tbody> from table fragments.
                if (!(!tempDiv.getElementsByTagName("tbody").length)) {

                    // String was a <table>, *may* have spurious <tbody>
                    var hasBody = /<tbody/i.test(elem),
                        tbody = !tags.indexOf("<table") &&
                            !hasBody ?
                            div.firstChild && div.firstChild.childNodes :
                            // String was a bare <thead> or <tfoot>
                            wrap[1] == "<table>" &&
                                !hasBody ? div.childNodes : [];

                    for (var j = tbody.length - 1; j >= 0; --j)
                        if (tbody[j].nodeName.toLowerCase() === "tbody" &&
                            !tbody[j].childNodes.length)
                            tbody[j].parentNode.removeChild(tbody[j]);

                }

                // div.innerHTML = elem;

                // IE completely kills leading whitespace when innerHTML is used
                if (!(tempDiv.firstChild.nodeType == 3) && /^\s/.test(elem))
                    div.insertBefore(
                        context.createTextNode(elem.match(/^\s*/)[0]),
                                               div.firstChild);

                elem = Iuppiter.toArray(div.childNodes);
            }

            if (elem.nodeType)
                ret.push(elem);
            else
                ret = Iuppiter.merge(ret, elem);

        });

        if (fragment) {
            var i;
            for (i = 0; ret[i]; i++)
                fragment.appendChild(ret[i]);

            return scripts;
        }

        return ret;
    }
};

var IuppiterBaseDom = Iuppiter.Base.Dom;

/**
 * It represents an Element in the DOM.
 *
 * @param {HTMLElement|String} element The HTML element object or the id of
 *                                     the HTML element.
 */
Iuppiter.Element = function(element) {

    var dom = typeof element == "string" ?
            document.getElementById(element) : element;

    if(!dom) // invalid id/element
        return null;

    /**
     * The HTML DOM element id.
     * @type String.
     */
    this.id = dom.id;

    /**
     * The HTML DOM element object.
     * @type The HTML DOM element.
     */
    this.dom = dom;
};

Iuppiter.Element.prototype = {

    /**
     * Private method to add the CSS style's unit.
     *
     * @param {Object} v The raw value.
     * @param {Object} defaultUnit The default unit.
     */
    addUnits: function(v, defaultUnit) {
        if (v === "" || v == "auto")
            return v;

        if (v === undefined)
            return '';

        if (typeof v == "number" || !El.unitPattern.test(v))
            return v + (defaultUnit || 'px');

        return v;
    },

    /**
     * Return true if the HTML DOM element occupies block space.
     *
     * @param el {Object} The HTML DOM element (Optional).
     * @return {Boolean} True if the HTML DOM element occupies block space,
     *                   else false.
     */
    isBlock: function(el) {
        if (!el)
            el = this.dom;

        var oWindow = el.ownerDocument.defaultView ||
                      el.ownerDocument.parentWindow,
            displayStyle = (oWindow.getComputedStyle) ?
                oWindow.getComputedStyle(el, '').getPropertyValue('display'):
                el.currentStyle['display'],
            floatStytle = (oWindow.getComputedStyle) ?
                oWindow.getComputedStyle(el, '').getPropertyValue('float'):
                el.currentStyle['float'],
            bIsBlockElement = false,
            tagName = el.tagName, i = 0;

        bIsBlockElement = el.getAttribute('domselectorSpacer') == 'true';
        bIsBlockElement = (bIsBlockElement ||
            (displayStyle == 'block' || displayStyle == 'table' ||
             tagName == 'TBODY' || tagName == 'TR' || tagName == 'TH' ||
             tagName == 'TD' || tagName == 'BR'));

        var childNode;
        if (!bIsBlockElement && tagName != 'SELECT') {
            for (i = 0; i < el.childNodes.length; i++) {
                childNode = el.childNodes[i];
                if (childNode.nodeType == 1 && this.isBlock(childNode)) {
                    return true;
                }
            }
        }

        if (bIsBlockElement)
            bIsBlockElement = (floatStytle == 'none');

        return bIsBlockElement;
    },

    /**
     * Return true if this element is an ancestor of the passed element.
     *
     * @param {HTMLElement|String} el The element to check.
     * @return {Boolean} True if this element is an ancestor of this DOM
     *                   element, else false.
     */
    contains: function(element) {
        var contains = false;
        while (element) {
            if ((contains = this.dom == element)) {
                break;
            }
            element = element.parentNode;
        }
        return contains;
    },

    /**
     * Check whether it surrounded the HTML DOM element or not.
     *
     * @param {HTMLElement} el The HTML DOM element to check.
     * @return {Boolean} True if it surrounds the HTML DOM element, otherwise
     *                   else.
     */
    surround: function(el) {
        return this.getRegion().contains(Iuppiter.get(el).getRegion());
    },

    /**
     * Check whether it was surrounded by the HTML DOM element or not.
     *
     * @param {HTMLElement} el The HTML DOM element to check.
     * @return {Boolean} True if it was surrounded by the HTML DOM element,
     *                   otherwise else.
     */
    beSurrounded: function(el) {
        return this.getRegion().beSurrounded(Iuppiter.get(el).getRegion());
    },

    /**
     * Set the size of the element.
     *
     * @param {Number} width The new width.
     * @param {Number} height The new height.
     * @return {Element} This.
     */
    setSize : function(width, height) {
        if(typeof width == "object") {
            height = width.height;
            width = width.width;
        }

        /*
        width = this.adjustWidth(width);
        height = this.adjustHeight(height);
        */

        this.dom.style.width = this.addUnits(width);
        this.dom.style.height = this.addUnits(height);
        return this;
    },

    /**
     * Normalize currentStyle and ComputedStyle.
     * Reference: YUI library.
     *
     * @param {String} property The style property whose value is returned.
     * @return {String|Array} The current value of the style property for the
     *                        element(s).
     */
    getStyle: function(property) {
        return IuppiterBaseDom.getStyle(this.dom, property);
    },

    /**
     * Wrapper for setting style properties of HTML elements.
     * Normalize "opacity" across modern browsers.
     * Reference: YUI library.
     *
     * @param {String} property The style property to be set.
     * @param {String} val The value to apply to the given property.
     */
    setStyle: function(property, val) {
        IuppiterBaseDom.setStyle(this.dom, property, val);
    },

    /**
     * Get the current position of the element based on page coordinates.
     * Reference: YUI library.
     *
     * @param {HTMLElement} el The HTML DOM element.
     * @return {Array} The XY position of a dictionary.
     */
    getPosition: function() {
        return IuppiterBaseDom.getXY(this.dom);
    },

    /**
     * Set the position of the HTML element in page coordinates, regardless of
     * how the element is positioned.
     * The element(s) must be part of the DOM tree to have page coordinates
     * (display:none or elements not appended return false).
     * Reference: YUI library.
     *
     * @param {Array} pos Contains X & Y values for new position
     *                    (coordinates are page-based).
     */
    setPosition: function(pos) {
        IuppiterBaseDom.setXY(pos);
    },

    /**
     * Get the region of the HTML DOM element.
     *
     * @return {Region} The region of the HTML DOM element.
     */
    getRegion: function() {
        return IuppiterBaseDom.getRegion(this.dom);
    },

    /**
     * Set the element's position and size the the specified region.
     *
     * @param {Region} region The region to fill.
     * @param {Region} region The region to fill.
     * @return {Element} This.
     */
    setRegion : function(region) {
        this.setSize(region.right - region.left, region.bottom - region.top);
        IuppiterBaseDom.setXY(this.dom, [region.left, region.top]);
        return this;
    }
};
