#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: StatisticsHandler.py 9660 2016-05-04 10:35:44Z Judy $
#
# Copyright (c) 2012 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Judy $
# $Date: 2016-05-04 18:35:44 +0800 (週三, 04 五月 2016) $
# $Revision: 9660 $

from datetime import timedelta, datetime

from django.db.models import Max as modelsMax
from django.http import Http404
from django.utils.translation import ugettext_lazy as _

from Iuppiter.Encoding import _unicode

from Theophrastus.models import Newsletter
from Theophrastus.models import ContactMailingStatus

_STATUS = [
    'deliverie', 'error', 'invaild', 'open', 'openOnSite', 'click',
    'unsubscribe', 'bounce', 'complain', 'reject', 'sentTest'
]

STATUS_DICT = dict(ContactMailingStatus.STATUS_CHOICES)
STATUS_NAMES = STATUS_DICT.values()
STATUS_CODE = STATUS_DICT.keys()

def getDateRange(rangeName='week', defaultStart=None):
    """
    Get range of dates.

    week: from last week to today.
    month: from last months to today.
    3month: from last three months to today.

    @param rangeName The range of name.
    """
    if not rangeName.strip():
        rangeName = 'week'

    endDate = datetime.now()

    rangeMapping = {
        'week': 7,
        'month': int(1 * (365 / 12)),
        '3month': int(3 * (365 / 12)),
    }

    if rangeName in rangeMapping:
        deltaNumber = rangeMapping[rangeName]
        days = [endDate.strftime('%Y-%m-%d')]
        for i in range(1, deltaNumber):
            d = endDate - timedelta(days=i)
            if defaultStart and defaultStart > d:
                break

            d = d.strftime('%Y-%m-%d')
            days.append(d)

        days.reverse()

        class DateRange(object):
            def __init__(self, days):
                self._days = days
                self.startDate = '%s 00:00:00' % days[0]
                self.endDate = '%s 23:59:59' % days[-1]

        return DateRange(days)
    else:
        return None

#def countStatus(queryset):
#    """
#    Calculate the status number of occurrences.
#    ex: (c.newsletter.id, c.status, c.creation_date)
#
#    SENT_TEST = -1
#    SENT = 0
#    ERROR = 1
#    INVALID = 2
#    OPENED = 4
#    OPENED_ON_SITE = 5
#    LINK_OPENED = 6
#    UNSUBSCRIPTION = 7
#
#    BOUNCED = 8
#    COMPLAINT = 9
#    REJECTED = 10
#
#    @param queryset The queryset of ContactMailingStatus.
#    """
#    sentTestNumber = 0
#    sentNumber = 0
#    errorLogs = set()
#    invaildLogs = set()
#    openLogs = set()
#    openOnSiteLogs = set()
#    clickLogs = set()
#    unsubscriptionNumber = 0
#
#    bouncedLogs = set()
#    complainLogs = set()
#    rejectedLogs = set()
#
##    subscribesList = []
#
#    for (id, status, logTime) in queryset:
#        if status == ContactMailingStatus.SENT_TEST:
#            sentTestNumber += 1
#        elif status == ContactMailingStatus.SENT:
#            sentNumber += 1
#        elif status == ContactMailingStatus.ERROR:
#            errorLogs.add(id)
#        elif status == ContactMailingStatus.INVALID:
#            invaildLogs.add(id)
#        elif status == ContactMailingStatus.OPENED:
#            openLogs.add(id)
#        elif status == ContactMailingStatus.OPENED_ON_SITE:
#            openOnSiteLogs.add(id)
#        elif status == ContactMailingStatus.LINK_OPENED:
#            clickLogs.add(id)
#        elif status == ContactMailingStatus.UNSUBSCRIPTION:
#            unsubscriptionNumber += 1
#        ##
#        elif status == ContactMailingStatus.BOUNCED:
#            bouncedLogs.add(id)
#        elif status == ContactMailingStatus.COMPLAINT:
#            complainLogs.add(id)
#        elif status == ContactMailingStatus.REJECTED:
#            rejectedLogs.add(id)
##        elif status == ContactMailingStatus.STATUS_SUBSCRIBE:
##            subscribesList.append(id)
#
#    return {
#        'sentTest': sentTestNumber,
#        'deliverie': sentNumber,
#        'error': len(errorLogs),
#        'invaild': len(invaildLogs),
#        'open': len(openLogs),
#        'openOnSite': len(openOnSiteLogs),
#        'click': len(clickLogs),
#        'unsubscribe': unsubscriptionNumber,
#        ##
#        'bounce': len(bouncedLogs),
#        'complain': len(complainLogs),
#        'reject': len(rejectedLogs),
#        #'subscribe': len(subscribesList),
#    }

def getAccumulatedValues(listData, dataType):
    """
    'sentTest': sentTestNumber,
    'deliverie': sentNumber,
    'error': len(errorLogs),
    'invaild': len(invaildLogs),
    'open': len(openLogs),
    'openOnSite': len(openOnSiteLogs),
    'click': len(clickLogs),
    'unsubscribe': unsubscriptionNumber,
    ##
    'bounce': len(bouncedLogs),
    'complain': len(complainLogs),
    'reject': len(rejectedLogs),
    #'subscribe': len(subscribesList),
    """
    returnInfo = {}
    _returnInfo = {}

    tmpInfo = dict([(s, 0.0) for s in _STATUS])

    if dataType == 'number':
        for (day, d) in listData:
            for k in tmpInfo.keys():
                if k not in d:
                    continue
                v = tmpInfo[k]
                v += d[k]
                tmpInfo[k] = v

            returnInfo[day] = dict(tmpInfo)

        return returnInfo
    else:
        for (day, d) in listData:
            for k in tmpInfo.keys():
                if k not in d:
                    continue
                v = tmpInfo[k]
                v += d[k]
                tmpInfo[k] = v

            tt = {}
            for k in tmpInfo.keys():
                if k == 'subscribe':
                    continue
                if k == 'unsubscribe':
                    nv = percentage(tmpInfo[k], tmpInfo['deliverie'],
                                     showPercent=False)
                else:
                    nv = percentage(tmpInfo[k], tmpInfo['deliverie'],
                                     showPercent=False)

                tt[k] = nv

            returnInfo[day] = tt

        return returnInfo

def toChartStructure(dictionary):

    info = {}
    for day in dictionary:
        data = dictionary[day]
        for key in data:
            v = data.get(key, 0)
            ll = info.get(key, [])
            ll.append([day, v])
            info.update({key: ll})

    return info

def percentage(value, total, showPercent=True):
    """
    Compute percentage.

    @param value Value
    @param total Total value.
    @param showPercent Show percent?
    """
    if total <= 0:
        v = 0
    else:
        v = ((value * 1.0) / total)
        if v > 1:
            v = 1.0
        v = v * 100 if (value > 0 and total > 0) else 0

    if showPercent:
        v = ('%s' % str(round(v, 2))) + '%'
    else:
        v = round(v, 2)

    return v

def getEmptyDataStructure():
    return {
        'sentTest': 0,
        'deliverie': 0,
        'error': 0,
        'invaild': 0,
        'open': 0,
        'openOnSite': 0,
        'click': 0,
        'unsubscribe': 0,
        'bounce': 0,
        'complain': 0,
        'reject': 0,
    }

def countStatus(queryset):
    """
    Calculate the status number of occurrences.
    ex: (c.newsletter.id, c.status, c.creation_date)

    SENT_TEST = -1
    SENT = 0
    ERROR = 1
    INVALID = 2
    OPENED = 4
    OPENED_ON_SITE = 5
    LINK_OPENED = 6
    UNSUBSCRIPTION = 7

    BOUNCED = 8
    COMPLAINT = 9
    REJECTED = 10

    @param queryset The queryset of ContactMailingStatus.
    """
    sentTestNumber = set()
    sentNumber = set()
    errorLogs = set()
    invaildLogs = set()
    openLogs = set()
    openOnSiteLogs = set()
    clickLogs = set()
    unsubscriptionNumber = set()

    bouncedLogs = set()
    complainLogs = set()
    rejectedLogs = set()
    
    for (id, status) in [(i['contact'], i['status']) for i in queryset]:
        if status == ContactMailingStatus.SENT_TEST:
            sentTestNumber.add(id)
        elif status == ContactMailingStatus.SENT:
            sentNumber.add(id)
        elif status == ContactMailingStatus.ERROR:
            errorLogs.add(id)
        elif status == ContactMailingStatus.INVALID:
            invaildLogs.add(id)
        elif status == ContactMailingStatus.UNSUBSCRIPTION:
            unsubscriptionNumber.add(id)
        ##
        elif status == ContactMailingStatus.BOUNCE:
            bouncedLogs.add(id)
        elif status == ContactMailingStatus.COMPLAIN:
            complainLogs.add(id)
        elif status == ContactMailingStatus.REJECT:
            rejectedLogs.add(id)
    
    # formula reference: emencia.django.utils.statistics.py
    # openings
    openings = queryset.filter(Q(status=ContactMailingStatus.OPENED) | 
                               Q(status=ContactMailingStatus.OPENED_ON_SITE))
    openingsByLinks = len(
        set(queryset.filter(status=ContactMailingStatus.LINK_OPENED).exclude(
        contact__in=openings.values_list('contact', flat=True)
    ).values_list('contact', flat=True)))
    uniqueOpenings = len(set(openings.values_list('contact', flat=True))) + \
                     openingsByLinks
                          
    # on site openings
    onSiteOpenings = queryset.filter(
       status=ContactMailingStatus.OPENED_ON_SITE)
    uniqueOnSiteOpenings = len(set(onSiteOpenings.values_list(
                              'contact', flat=True)))
            
    # clicked
    clickedLinks = queryset.filter(status=ContactMailingStatus.LINK_OPENED)
    uniqueClicked = len(set(clickedLinks.values_list(
                              'contact', flat=True)))

    return {
        'sentTest': len(sentTestNumber),
        'deliverie': len(sentNumber),
        'error': len(errorLogs),
        'invaild': len(invaildLogs),
        'open': uniqueOpenings,
        'openOnSite': uniqueOnSiteOpenings,
        'click': uniqueClicked,
        'unsubscribe': len(unsubscriptionNumber),
        'bounce': len(bouncedLogs),
        'complain': len(complainLogs),
        'reject': len(rejectedLogs),
    }

from django.db.models import Q

from emencia.django.newsletter.models import ContactMailingStatus as Status

def getStatisticsTotalData(reqestGetData, subscriberCount, testSubscriberCount, 
                           conditions):
    """
    Get statistics of subscriber's every rates.
    Data for line chart. 

    @param slug Newsletter slug.
    @param request Request instance.
    """
    period = reqestGetData.get('period', 'week')
    dataType = reqestGetData.get('dataType', 'number')

    _firstCreationDate = ContactMailingStatus.objects.filter(**conditions)
    _s = {"d": "strftime('%%Y-%%m-%%d', creation_date)"}
    _firstCreationDate = _firstCreationDate.extra(_s)
    _firstCreationDate = _firstCreationDate.values('d').distinct().order_by('d')
    _t = datetime.strptime(_firstCreationDate[0]['d'], '%Y-%m-%d')
    dates = getDateRange(period, defaultStart=_t)

    def getContactStatus(conditions):
        contactStatus = ContactMailingStatus.objects.filter(**conditions)
        selectData = {"d": "strftime('%%Y-%%m-%%d', "
                            "newsletter_contactmailingstatus.creation_date)"}
        contactStatus = contactStatus.extra(select=selectData).values(
            'contact', 'status', 'd').distinct(
            ).order_by('contact', 'status', 'd')

        # [{'status': 0, 'contact': 1, 'd': u'2012-12-20'},
        #  {'status': 4, 'contact': 1, 'd': u'2012-12-20'}, ...]
        return contactStatus

    def sortByDatetime(contactStatus):
        completed = {}
        for r in contactStatus:
            date = r['d']
            year = date.split('-')[0]
            month = date.split('-')[1]
            day = date.split('-')[2]
            queryset = contactStatus.filter(creation_date__day=day,
                                            creation_date__month=month,
                                            creation_date__year=year)
            completed[date] = queryset
        return completed

    conditions.update(
        {'creation_date__range': [dates.startDate, dates.endDate]})
    
    if 'newsletter__id__in' in conditions:
        newsletterIds = conditions.pop('newsletter__id__in')

        dataTmp = {}
        for _id in newsletterIds:
            conditions.update({'newsletter__id': _id})
            _cs = getContactStatus(conditions)
            _statusCount = countStatus(_cs)

        # Merge.
        totalData = []
        for _dt in dataTmp:
            data = {}
            allDatas = dataTmp[_dt]
            for _t in allDatas:
                for _key in _t:
                    v = _t[_key]
                    if _key not in data:
                        data[_key] = 0
                    data[_key] += v
            totalData.append([_dt, data])
    else:
        contactStatus = getContactStatus(conditions)
        completed = sortByDatetime(contactStatus)
        totalData = []
        for c in completed:
            data = countStatus(completed[c])
            totalData.append([c, data])

    allDates = dict(totalData).keys()
    for day in dates._days:
        if day not in allDates:
            totalData.append([day, getEmptyDataStructure()])

    content = {}
    if dataType == 'number':
        #content[day] = [(status, value), ...]
        for (day, data) in totalData:
            content[day] = [(i, data[i]) for i in data]
    else:
        def _do(info, subscriberCount, testSubscriberCount):
            rates = []
            
            for i in info:
                if i in ('unsubscribe', 'deliverie', 'error'):
                    totalValue = subscriberCount
                elif i == 'sentTest':
                    totalValue = testSubscriberCount
                else:
                    totalValue = info['deliverie']        
                
                if i == 'open':            
                    value = percentage(info['open'] + info['openOnSite'], 
                                       totalValue, showPercent=False)
                else:
                    value = percentage(info[i], totalValue, showPercent=False)
                
                rates.append((i, value))
#                 if i == 'unsubscribe':
#                     totalValue = subscriberCount
#                 else:
#                     totalValue = info['deliverie']
#                 value = percentage(info[i], totalValue, showPercent=False)
#                 rates.append((i, value)) # (status, value)
            return rates

        for (day, data) in totalData:
            result = _do(data, subscriberCount, testSubscriberCount)
            content[day] = result # {'day': (status, value)}

    context = {}
    for day in content:
        for (statusName, value) in content[day]:
            dData = context.get(statusName, [])
            dData.append((day, value))
            context[statusName] = dData

    # Find the max value of Y-axis.
    tmp = []
    for k in context:
        for i in context[k]:
            tmp.append(i[-1])

    yaxisMaxValue = max(tmp)
    del tmp

    response = {
        'context': {
            'email': context,
        },
        'rateType': dataType,
        'dateRange': dates._days,
        'xaxis': {'minDate': dates._days[0], 'maxDate': dates._days[-1]},
        'yaxis': {'max': yaxisMaxValue},
        'status': _STATUS,
        'settings': {
            'labels': STATUS_NAMES,
        },
    }
    return response

def getStatisticsTotalRateData(reqestGetData, subscriberCount,
                               testSubscriberCount, conditions):
    """
    Get statistics of subscriber's every rates.
    Data for pie chart and statistics table.

    @param slug Newsletter slug.
    @param request Request instance.
    """

    # select distinct contact_id, status from newsletter_contactmailingstatus
    # where newsletter_id=3
    def getContactStatus(conditions):
        contactStatus = ContactMailingStatus.objects.filter(**conditions)
        # FIXME: Distinct multiple fields?
        #contactStatus = contactStatus.order_by().values(
        #    'contact', 'status').distinct(['contact', 'status'])
        contactStatus = contactStatus.order_by().values(
            'contact', 'status').distinct()        
        # [{'status': 0, 'contact': 1}, {'status': 4, 'contact': 1}, ...]
        return contactStatus   

    if 'newsletter__id__in' in conditions:
        newsletterIds = conditions.pop('newsletter__id__in')
        data = {}
        for _id in newsletterIds:
            conditions.update({'newsletter__id': _id})
            _cs = getContactStatus(conditions)
            
            _data = countStatus(_cs)
            for _d in _data:
                v = _data[_d]
                if _d not in data:
                    data[_d] = 0
                data[_d] += v
    else:
        contactStatus = getContactStatus(conditions)
        data = countStatus(contactStatus)

    rates = []
    
    totalValue = 0
    
    for d in data:
        totalValue += data[d]
    
    
    for d in data:
        if d in ('unsubscribe', 'deliverie', 'error'):
            totalValue = subscriberCount
        elif d == 'sentTest':
            totalValue = testSubscriberCount
        else:
            totalValue = data['deliverie']        

        value = percentage(data[d], totalValue)
        
        rates.append((d, value))

    response = {
        'context': dict(rates),
        'contextCount': data,
        'status': _STATUS,
        'statusNames': STATUS_NAMES,
    }
    return response

def getSubscriberHistory(requestGetData, conditions):
    """
    Get history data of subscribers.

    @param slug Slug.
    """
    from django.db import connection
    connection.queries = []

    statusCode = conditions.pop('status', None)

    contactStatus = ContactMailingStatus.objects.filter(**conditions)
    contactStatus = contactStatus.annotate(d=modelsMax("creation_date"),
                                           qid=modelsMax("id"))
    contactStatus = contactStatus.values('contact', 'qid')
    contactStatus.query.group_by = ['contact_id']
    qs = contactStatus.order_by('id')

    qids = [_i['qid'] for _i in qs]

    # Sort data.
    #sortedFieldName = requestGetData.get('sort', 'lastActivity')
    sortType = requestGetData.get('sortType', 'desc')
    if sortType == 'desc':
        sortType = True
    else:
        sortType = False

    #emailData = sortList(emailData, sortedFieldName, desc=sortType)
    info = []
    if statusCode is None:
        _query = ContactMailingStatus.objects.filter(
            id__in=qids).order_by('-creation_date')
    else:
        _query = ContactMailingStatus.objects.filter(
            id__in=qids, status=statusCode).order_by('-creation_date')

    total = _query.count()

    # Get query range.
    more = False
    start = int(requestGetData.get('start', 0))
    if (len(_query) - start) > 20:
        more = True
    _query = _query[start:start + 20]

    for q in _query:
        info.append({
            'contact': {
                'id': q.contact.id,
                'name': _unicode(q.contact),
                'email': q.contact.email,
            },
            'status': q.status,
            'creationDate': q.creation_date.strftime('%Y-%m-%d %I:%M%p'),
        })

    return {
        'content': info,
        'sortType': sortType,
        'more': more,
        'startNum': start,
        'totalNum': total,
        #'sortedFieldName': sortedFieldName,
    }
