#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Json.py 11483 2019-04-24 03:01:28Z Lavender $
#
# Copyright (c) 2012 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Lavender $
# $Date: 2019-04-24 11:01:28 +0800 (Wed, 24 Apr 2019) $
# $Revision: 11483 $
"""
JSON serializer.
"""

import sys
import six
import types
import datetime
import json

json.write = json.dumps
json.read = json.loads

datetimeAttrs = ('year', 'month', 'day', 'hour', 'minute', 'second',
                 'microsecond')

def datetimeToObjDict(obj):
    """
    Embedded JSON serailizer conversion function for datetime type.
    Reference: http://python.cx.hu/python-cjson/

    @param obj Datetime object.
    @return Object notation dictionary for datetime.
    """
    dic = {}
    g = lambda o, f=lambda a, v: v: [f(a, getattr(o, a)) for a in datetimeAttrs]
    dic['obj'] = "new Date(Date.UTC(%d,%d,%d,%d,%d,%d,%d))" % tuple(g(obj))
    g(obj, lambda a, v: dic.setdefault(a, v))

    return dic

def datetimeFromObjDict(dic):
    """
    Embedded JSON serailizer inversion function for datetime type.
    Reference: http://python.cx.hu/python-cjson/

    @param dic Object notation dictionary.
    @return Original datetime object.
    """
    return datetime.datetime(*[dic[attr] for attr in datetimeAttrs])

import re

def regexToObjDict(obj):
    """
    Embedded JSON serailizer conversion function for regular expression type.

    @param obj regular expression object.
    @return Object notation dictionary for regular expression.
    """
    dic = {'pattern': obj.pattern}
    return dic

def regexFromObjDict(dic):
    """
    Embedded JSON serializer inversion function for regular expression type.

    @param dic Object notation dictionary.
    @return Original regular expression object.
    """
    return re.compile(dic['pattern'])

from Iuppiter.Util import classForName
import inspect

if six.PY2:
    import cStringIO as io
else:
    import io

class Serializer(object):
    """
    A JSON serializer which tries to serialize all python objects type to
    be a valid JSON form and deserializes it back as normal python object.

    We append a special property '__pythonclass__' in serialized JSON to
    recognize the original python class type.

    That serialize and deserialize a JSON with '__pythonclass__' property is
    not easy, we will search the registered functions which describe how to
    do that or try to use 'brute-force' way if no registered functions for
    that class type found.

    Note: We only support single inheritance in this time.
    """

    ## The class key of object notation dictionary.
    CLASS_KEY = "__pythonclass__"

    ## These types must be handled specially.
    _SPECIAL_TYPES = [
        list,
        tuple,
        dict,
    ]
    if six.PY2:
        _SPECIAL_TYPES.append(types.InstanceType)

    ## The types we embedded the conversion function.
    if sys.version_info.major == 3 and sys.version_info.minor == 7:
        _SPECIAL_INSTANCE = (
            datetime.datetime,
            re.Pattern,
        )
    else:
        _SPECIAL_INSTANCE = (
            datetime.datetime,
            re._pattern_type,
        )

    ## The backend JSON implementation convertable types.
    _BACKEND_JSON_CONVERTABLE = [
        bool,
        dict,
        bytes,
        tuple,
        list,
        types.GeneratorType,
        float,
        type(None),
        six.text_type,
        six.binary_type
    ]

    for types in (six.string_types, six.integer_types):
        _BACKEND_JSON_CONVERTABLE.extend(types)

    @staticmethod
    def isSpecial(obj):
        """
        Is special types or special instance?

        @param obj Object
        @return True if this object's type is in SPECIAL_TYPES or
                SPECIAL_INSTANCE.
        """
        t = type(obj)

        if t in Serializer._SPECIAL_TYPES:
            return True
        elif isinstance(obj, Serializer._SPECIAL_INSTANCE):
            return True
        elif t in Serializer._BACKEND_JSON_CONVERTABLE:
            return False
        elif hasattr(obj, "__class__"):
            return True
        else:
            return False

    @classmethod
    def writePythonClassMark(cls, dic, clazz):
        """
        Write required __pythonclass__ key in object notation dictionary.

        @param dic Object notation dictionary.
        @param clazz Class type of this object.
        """
        dic[cls.CLASS_KEY] = "%s.%s" % (clazz.__module__, clazz.__name__)

    def __init__(self, backend=json):
        """
        Constructor.

        @param backend The backend JSON library to encode and decode. Default
                       backend is cjson for performance.
        """
        self.registered = {}
        self.backend = backend

        # Register embedded conversion functions.
        self.register(Serializer._SPECIAL_INSTANCE[0],
                      datetimeToObjDict, datetimeFromObjDict, True)
        self.register(Serializer._SPECIAL_INSTANCE[1],
                      regexToObjDict, regexFromObjDict, True)

    def register(self, clazz, toObjDictFunc, fromObjDictFunc,
                 forDerived=False):
        """
        Register conversion and inversion functions for specified class type.

        @param clazz Class object.
        @param toObjDictFunc A conversion function which convert object to a
                             proper object notation dictionary.
                             It can be None.
        @param fromObjDictFunc A inversion function which invert a object
                               notation dictionary to original object.
                               It can be None.
        @param forDerived True if you want that registered conversion and
                          inversion functions can be applied in derived class.
        """
        self.registered[clazz] = (toObjDictFunc, fromObjDictFunc, forDerived)

    def unregister(self, clazz):
        """
        Unregister conversion and inversion functions by given clazz argument.

        @param clazz Class object registered already.
        """
        del self.registered[clazz]

    def serialize(self, obj, stream=None):
        """
        Serialize a object to JSON.

        @param obj Object.
        @param stream Output stream, if you don't specified this argument,
                      this method will return a JSON string or it will write
                      the JSON string to stream.
        @note After serialization, list, dictionary and tuple type may be
              changed in conversion process. If you want to keep object
              unchanged, copy a cloned one first.
        """
        if stream == None:
            returnStr = True
            s = io.StringIO()
        else:
            returnStr = False
            s = stream

        s.write(self.backend.write(self.convert(obj)))

        if returnStr:
            r = s.getvalue()
            s.close()
            return r

    def _dispatch(self, obj, funcName):
        """
        Dispatch to private convert/revert method.

        @param obj Object.
        @param funcName convert/revert function name string.
        @return Result of dispatched function.
        """
        t = type(obj)

        funcName2 = t.__name__.capitalize()

        if t in Serializer._SPECIAL_TYPES:
            pass
        elif isinstance(obj, Serializer._SPECIAL_INSTANCE):
            return self._toObjDict(obj)
        else:
            funcName2 = "Instance"

        return getattr(self, '_%s%s' % (funcName, funcName2))(obj)

    def convert(self, obj):
        """
        Convert a object to a JSON encodable object.

        @param obj Object.
        @return JSON encodable object.
        """
        # Dispatch.
        if Serializer.isSpecial(obj):
            obj = self._dispatch(obj, "convert")
        return obj

    def _convertInstance(self, obj):
        """
        Convert a object instance type.

        @param obj Object.
        @return JSON encodable object notation dictionary.
        """
        return self._toObjDict(obj)

    def _findRegisteredKey(self, clazz):
        """
        Find the key of given class object in registered classes.

        @param clazz Class object.
        @return The key matched in registered classes.
        """
        key = None
        if clazz in self.registered:
            key = clazz
        elif isinstance(clazz, six.string_types):
            for k in self.registered:
                d = {}
                Serializer.writePythonClassMark(d, k)
                if clazz == d[Serializer.CLASS_KEY]:
                    key = k
                    break

        elif hasattr(clazz, '__bases__') and clazz.__bases__:
            for b in clazz.__bases__:
                key2 = self._findRegisteredKey(b)
                if key2:
                    # Allow derived classes use its conversion function.
                    if self.registered[key2][2]:
                        return key2

        return key

    def _toObjDict(self, obj):
        """
        Convert a object to a JSON encodable object notation dictionary.

        @param obj Object.
        @return JSON encodable object notation dictionary.
        """
        if hasattr(obj, '__class__'):
            t = obj.__class__
        else:
            t = type(obj)

        key = self._findRegisteredKey(t)
        dic = {}
        impl = self.registered.get(key, (None,))[0]
        if impl:
            dic = impl(obj)
            if isinstance(dic, dict) and not Serializer.CLASS_KEY in dic:
                Serializer.writePythonClassMark(dic, t)
        else:
            dic = self._bruteForceToObjDict(obj)

        return dic

    def accept(self, clazz, attr):
        """
        Determine to accept given attribute or not.

        @param clazz Class.
        @param attr Attribute name.
        @return True if acceptable.
        """
        if not attr.startswith('_'):
            return True

        if attr.startswith('__'):
            return False

        c = (clazz,) + clazz.__bases__
        for cc in c:
            if attr.startswith('_%s__' % cc.__name__):
                return False

        return True

    def _bruteForceToObjDict(self, obj):
        """
        Brute force way to convert a object to be a object notation dictionary.
        We extract all writable attributes in this object and construct a
        dictionary which contains these attributes.

        @param obj Object.
        @return JSON encodable object notation dictionary.
        """
        # Class object is not convertable.
        if inspect.isclass(obj):
            return {}

        clazz = obj.__class__

        # Extract all writable attributes.
        #d = dict(clazz.__dict__, **obj.__dict__)
        dic = {}
        for k, v in inspect.getmembers(obj):
            if self.accept(clazz, k) and not callable(v):
                dic[k] = self.convert(v)

        Serializer.writePythonClassMark(dic, clazz)
        return dic

    def _convertList(self, obj):
        """
        Convert a list to a JSON encodable list.

        @param obj List.
        @return JSON encodable list.
        """
        return self._instanceToObjDictInContainer(obj)

    def _convertTuple(self, obj):
        """
        Convert a tuple to a JSON encodable tuple.

        @param obj Tuple.
        @return JSON encodable tuple.
        """
        return self._instanceToObjDictInContainer(list(obj))

    def _convertDict(self, obj):
        """
        Convert a dictionary to a JSON encodable dictionary.

        @param obj Dictionary.
        @return JSON encodable dictionary.
        """
        return self._instanceToObjDictInContainer(obj)

    def _instanceToObjDictInContainer(self, container):
        """
        Convert the special types in a container to be a JSON encodable type.

        @param container List, tuple, dictionary container.
        @return A JSON encodable container.
        """
        iter = None
        if isinstance(container, dict):
            iter = list(container.items())
        else:
            iter = enumerate(container)

        for k, v in iter:
            container[k] = self.convert(v)

        return container

    def deserialize(self, json):
        """
        Deserialize a JSON string to original object.

        @param json JSON string.
        @return Object.
        """
        return self.revert(self.backend.read(json))

    def revert(self, obj):
        """
        Inverse the JSON encodable object to original object.

        @param obj JSON encodable object.
        @return Original object.
        """
        # Dispatch.
        if Serializer.isSpecial(obj):
            obj = self._dispatch(obj, "revert")

        return obj

    def _fromObjDict(self, dic):
        """
        Inverse a object notation dictionary to original object.

        @param dic Object notation dictionary.
        @return Original object.
        """
        try:
            clazz = classForName(dic[Serializer.CLASS_KEY])
        except ImportError:
            clazz = dic[Serializer.CLASS_KEY]

        key = self._findRegisteredKey(clazz)
        impl = self.registered.get(key, (None, None))[1]
        if impl:
            return impl(dic)
        else:
            return self._bruteForceFromObjDict(dic, clazz)

    def _bruteForceFromObjDict(self, dic, clazz):
        """
        Try to use brute force way to invert a object notation dictionary to
        original object.

        @param Object notation dictionary.
        @param clazz The type of this object notation dictionary which
                     represents.
        """
        # Try default constructor.
        # If no default constructor, it will raise TypeError.
        # Maybe we can try types.InstanceType() while there is no
        # default constructor available.
        obj = clazz()

        del dic[Serializer.CLASS_KEY]

        for k, v in list(dic.items()):
            value = self.revert(v)
            try:
                setattr(obj, k, value)
            except:
                continue
        return obj

    def _revertDict(self, dic):
        """
        Inverse a dictionary to original dictionary.

        @param dic Dictionary.
        @param Original dictionary.
        """
        if Serializer.CLASS_KEY in dic:
            return self._fromObjDict(dic)
        else:
            for k, v in list(dic.items()):
                dic[k] = self.revert(v)

        return dic

    def _revertList(self, list):
        """
        Inverse a list to original list.

        @param list List.
        @param Original list.
        """
        for k, v in enumerate(list):
            list[k] = self.revert(v)

        return list

from django.utils.functional import Promise

import django

if django.__version__ == '1.8':
    from django.utils.datastructures import SortedDict

class DjangoConverter(object):
    """
    It is responsible for converting django's queryset and model to JSON format.
    """

    def __init__(self, serializer):
        self.serializer = serializer

    def promiseToObjDict(self, obj):
        """
        Convert a Promise (most of time, it will be i18n's lazy proxy) to
        object notation dictionary.

        @param obj Promise object.
        @return Object notation dictionary.
        """
        return six.u(obj)

    def sortedDictToObjDict(self, obj):
        """
        Convert a SortedDict to object notation dictionary.

        @param obj SortedDict object.
        @return Object notation dictionary.
        """
        dic = {}
        dic['keyOrder'] = obj.keyOrder
        for k, v in list(obj.items()):
            dic[k] = self.serializer.convert(v)

        return dic

    def objDictToSortedDict(self, dic):
        """
        Convert a object notation dictionary of SortedDict to SortedDict.

        @param dic Object notation dictionary.
        @return SortedDict.
        """
        dd = dic
        del dd[Serializer.CLASS_KEY]
        keyOrder = dd.pop('keyOrder')

        sd = SortedDict()
        for k in keyOrder:
            sd[k] = self.serializer.revert(dic[k])

        return sd



# Instantiate global instances.
serializer = Serializer()
djangoConverter = DjangoConverter(serializer)

serializer.register(Promise,
                    djangoConverter.promiseToObjDict,
                    None, True)

if django.__version__ == '1.8':
    serializer.register(SortedDict,
                        djangoConverter.sortedDictToObjDict,
                        djangoConverter.objDictToSortedDict, True)
