#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: urls.py 11172 2018-07-16 09:31:00Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $ (last)
# $Date: 2018-07-16 17:31:00 +0800 (週一, 16 七月 2018) $
# $Revision: 11172 $

import unittest
import datetime

from dateutil import parser

from Iuno.cloud.storage import gos as os

try:
    FileNotFoundError
except NameError:
    FileNotFoundError = IOError

class GosTest(unittest.TestCase):
    TEST_FOLDER = 'CI/Tests/Iuno.cloud.storage'
    TEST_FOLDER_ID = '1kSTFKkrQAfNuZjQgN6jBHBIZzhHYtr7w'
    RESULT_FILE_DICT = {
        'id': '1ellDpAhS3XqWPav6wD0xunOpaWpxrrt3',
        'name': 'plane.jpg'
    }

    # Service.py
    def testGetFileById(self):
        fileId = self.RESULT_FILE_DICT["id"]
        result = os.getFile(fileId)

        self.assertDictEqual(result, self.RESULT_FILE_DICT)

    def testGetFileByPath(self):
        path = os.path.join(self.TEST_FOLDER, "plane.jpg")
        result = os.getFile(path, type=os.BY_PATH)

        self.assertDictEqual(result, self.RESULT_FILE_DICT)

    def testGetFileNotFound(self):
        fileId = "incorrectId"
        path = "incorrectPath"
        BY_SIZE = 3

        # why using lambda see https://stackoverflow.com/questions/6103825/
        # how-to-properly-use-unit-testings-assertraises-with-nonetype-objects
        self.assertRaises(FileNotFoundError, lambda: os.getFile(fileId))
        self.assertRaises(FileNotFoundError,
                          lambda: os.getFile(path, type=os.BY_PATH))
        self.assertRaises(AttributeError,
                          lambda: os.getFile("20MB", type=BY_SIZE))

    def testGetFiles(self):
        folderId = self.TEST_FOLDER_ID
        result = os.getFiles(folderId)

        self.assertEqual(len(result), 1)
        self.assertDictEqual(result[0], self.RESULT_FILE_DICT)

    def testListdir(self):
        result = os.listdir(self.TEST_FOLDER)

        self.assertEqual(len(result), 1)
        self.assertEqual(result[0], 'plane.jpg')

    def testMkdir(self):
        newDir = os.path.join(self.TEST_FOLDER, "NewDir")
        os.mkdir(newDir)

        result = os.listdir(self.TEST_FOLDER)
        self.assertTrue(all(
            [name in result for name in ('NewDir', 'plane.jpg')]))

        os.remove(newDir)

    def testMakedirs(self):
        os.makedirs(os.path.join(self.TEST_FOLDER, "A", "B"))

        self.assertTrue("A" in os.listdir(self.TEST_FOLDER))
        self.assertTrue("B" in os.listdir(os.path.join(self.TEST_FOLDER, "A")))

        os.remove(os.path.join(self.TEST_FOLDER, "A"))

    def testRename(self):
        newDir = os.path.join(self.TEST_FOLDER, "NewDir")

        os.makedirs(newDir)
        os.rename(os.path.join(self.TEST_FOLDER, 'plane.jpg'),
                  os.path.join(newDir, 'plane2.jpg'))

        result = os.listdir(newDir)
        self.assertEqual(len(result), 1)
        self.assertEqual(result[0], 'plane2.jpg')

        os.rename(os.path.join(newDir, 'plane2.jpg'),
                  os.path.join(self.TEST_FOLDER, 'plane.jpg'))
        os.remove(newDir)

    def testPathJoin(self):
        self.assertEqual(
            'GosTest/A/B',
            os.path.join('GosTest', 'A', 'B')
        )

    def testPathExists(self):
        self.assertTrue('GosTest/plane.jpg')

    def testPathGetctime(self):
        # photo taken time
        self.assertEqual(
            datetime.datetime.strptime(
                "2018:09:16 16:59:01",
                "%Y:%m:%d %H:%M:%S"
            ),
            os.path.getctime(
                os.path.join(self.TEST_FOLDER, 'plane.jpg'),
                property='imageMediaMetadata',
                subProperty="time",
                timeFormat="%Y:%m:%d %H:%M:%S"
            )
        )

        # creation time
        self.assertEqual(
            parser.parse("2018-12-07T06:38:00.233Z"),
            os.path.getctime(os.path.join(self.TEST_FOLDER, 'plane.jpg'))
        )


if __name__ == '__main__':
    unittest.main()
