#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: build_products.py 11932 2020-02-18 17:50:41Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2020-02-19 01:50:41 +0800 (週三, 19 二月 2020) $
# $Revision: 11932 $

import os
import sys
import shutil
import datetime
import hashlib
import zipfile
import platform

from django.core.management.base import BaseCommand

from Iuppiter.Util import run
from Zephyrus.boilerplate.management.commands import (
    PROJECT_MASHINES_DIR, PROJECT_MASHINES_ZIP, ROOT_DIR_PATH, 
    PROJECT_VIRTUALENV_DIR, PROJECT_ENV_REQUIREMENTS, PROJECT_ENV_SCRIPT)
from Zephyrus.boilerplate.management.commands import createLogger  

def createEnv(version):
    if not os.path.isdir(PROJECT_VIRTUALENV_DIR):
        os.mkdir(PROJECT_VIRTUALENV_DIR)
    
    envPath = os.path.join(PROJECT_VIRTUALENV_DIR, "Env%s" % version)

    # windows
    if 'Windows' in platform.system(): 
        pythonPath =  os.path.join(envPath, 'python.exe')
        cwdPath =  os.path.join(envPath, 'Scripts')
        scriptPath = os.path.join(PROJECT_VIRTUALENV_DIR, 'CreateEnv.bat')
    else:
        pythonPath =  os.path.join(envPath, 'bin', 'python')
        cwdPath =  os.path.join(envPath, 'bin')
        scriptPath = os.path.join(PROJECT_VIRTUALENV_DIR, 'CreateEnv.sh')

    # requirements
    requirementPath = os.path.join(
        PROJECT_VIRTUALENV_DIR, 'REQUIREMENTS_%s.txt' % version)

    if not os.path.isdir(envPath):
        # copy file
        if not os.path.isfile(scriptPath):
            shutil.copyfile(PROJECT_ENV_SCRIPT, scriptPath)

        #  windows
        if not os.path.isfile(requirementPath):
            shutil.copyfile(
                PROJECT_ENV_REQUIREMENTS.replace("[VERSION]", version), 
                requirementPath)

        # create env
        #  windows
        if 'Windows' in platform.system(): 
            run("CreateEnv.bat %s" % version, 
                shell=True, cwd=PROJECT_VIRTUALENV_DIR)
        else:
            run(". $HOME/.bashrc && chmod +x $HOME/app/templates_shop/ZephyrusEnv/CreateEnv.sh && "
                "source $HOME/app/templates_shop/ZephyrusEnv/CreateEnv.sh %s" % version, 
                shell=True, cwd=PROJECT_VIRTUALENV_DIR)

    return (envPath, pythonPath)

class Command(BaseCommand):

    help = 'Start build all zip of template to dst folder.'
    
    def add_arguments(self, parser):
        parser.add_argument('start', type=str, help="start folder")
        parser.add_argument('output', type=str, help="output folder")
        parser.add_argument(
            '-l', '--log', 
            type=str, default="BuildProducts.log",
            help="log file name.")
        parser.add_argument(
            '-t', '--templateShop', 
            action='store_true',
            help="build all template for TemplateShop.")
        parser.add_argument(
            '-c', '--cmsVersion', 
            type=str, default="3.7.0",
            help="CMS version for products.")
        

    def handle(self, *args, **options):
        startPath = options['start']
        dstPath = options['output']
        templateShop = options['templateShop']
        version = options['cmsVersion']

        logger = createLogger(__name__, options['log'])
        machineLog = createLogger('ProductMachine', "MachineError.log")

        # create env
        envPath, pythonPath = self.createEnv(version)
        
        # create project
        machineName = hashlib.md5(
            (startPath + str(datetime.datetime.now())).encode()).hexdigest()
        machinePath = self.createProject(machineName, version)
        templateDir = self.unzipTemplates(startPath, machineName, templateShop)
        
        # build products
        currentPath = os.path.abspath(".")
        os.chdir(machinePath)
        
        try:
            env = os.environ.copy()
            del env['DJANGO_SETTINGS_MODULE']

            # create env
            env['VIRTUAL_ENV'] = envPath[0]
            envPython = envPath[1].replace("\\", "/")
            
            if 'Windows' in platform.system(): 
                activateCommand = 'call activate "%s"' % (envPath,)
            else:
                activateCommand = '. /home/nuwa/.bashrc && conda activate "%s"' % (envPath,)

            if not templateShop:
                cmd = "python manage.py build_templates \"%s\" \"%s\" -o" % (
                        templateDir, dstPath)
            else:
                cmd = "python manage.py build_templates \"%s\" \"%s\"" % (
                        templateDir, dstPath)

            run("%s && %s" % (activateCommand, cmd), env=env, shell=True)
            # run("%s" % (activateCommand, cmd), env=env, shell=True)
            with open(os.path.join(machinePath, 'BuildTemplates.log')) as log:
                content = log.read()
            if content:
                for f in os.listdir(machinePath):
                    if f == 'BuildTemplates.log' or f == 'AllError.log':
                        continue
                    if f.endswith('.log'):
                        with open(os.path.join(machinePath, f)) as log:
                            content += log.read()
                with open(os.path.join(machinePath, 'AllError.log')) as err:
                    machineLog.error(content + err.read())
                content = content.strip().replace('\n', '|')
                logger.error(content)
                return 1
            else:
                return 0
        finally:
            # remove project
            os.chdir(currentPath)
            projectPath = os.path.join(PROJECT_MASHINES_DIR, machineName)
            shutil.rmtree(projectPath)

    def createProject(self, machineName, version):
        if not os.path.isdir(os.path.join(PROJECT_MASHINES_DIR)):
            os.mkdir(os.path.join(PROJECT_MASHINES_DIR))
        dirName = os.path.join(PROJECT_MASHINES_DIR, machineName)
        os.mkdir(os.path.join(dirName))
        os.mkdir(os.path.join(dirName, 'Templates'))
        machinePath = os.path.join(dirName, "ProjectMachine")
        zip = zipfile.ZipFile(
            PROJECT_MASHINES_ZIP.replace("[VERSION]", version))
        zip.extractall(machinePath)
        zip.close()
        return machinePath

    def unzipTemplates(self, startPath, machineName, templateShop):
        dirName = os.path.join(PROJECT_MASHINES_DIR, machineName)
        templateDir = os.path.join(dirName, "Templates")
        for f in os.listdir(startPath):
            if not templateShop:
                templateName = os.path.splitext(f)[0]
                zip = zipfile.ZipFile(os.path.join(startPath, f))
                zip.extractall(os.path.join(templateDir, templateName))
                zip.close()
            else:
                shutil.copytree(
                    os.path.join(startPath, f), os.path.join(templateDir, f))
        return templateDir

    def createEnv(self, version):
        return createEnv(version)
