#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: cms_plugins.py 11243 2018-10-05 09:20:04Z Lavender $
#
# Copyright (c) 2018 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $ (last)
# $Date: 2018-10-05 17:20:04 +0800 (週五, 05 十月 2018) $
# $Revision: 11243 $

from django.db.models import Count
from django.template.loader import select_template
from django.db.models import Q

from cms.plugin_base import CMSPluginBase
from cms.plugin_pool import plugin_pool
from oscar.core.loading import get_model

from Zephyrus.cms_oscar import settings 
from Zephyrus.cms_oscar.models import (
    OscarFeatured, OscarLatest, OscarOffer, OR, AND)
from Zephyrus.cms_oscar.forms import OscarFeaturedForm, OscarLatestForm


ProductRecord = get_model('analytics', 'ProductRecord')
Product = get_model('catalogue', 'Product')
ConditionalOffer = get_model('offer', 'ConditionalOffer')
Condition = get_model('offer', 'Condition')

if hasattr(Product, 'tags'):
    hasTags = True
else:
    hasTags = False

class FeaturedPlugin(CMSPluginBase):
    model = OscarFeatured
    module = "Zephyrus Oscar"
    name = "Featured Products"
    render_template = settings.ZEPHYRUS_OSCAR_FEATURED_DEFAULT_TEMPLATE
    form = OscarFeaturedForm

    def get_render_template(self, context, instance, placeholder):
        return select_template([
            instance.renderTemplate,
            settings.ZEPHYRUS_OSCAR_FEATURED_DEFAULT_TEMPLATE,
        ])

    def render(self, context, instance, placeholder):
        catagories = instance.catagory.all()
        query = None

        # category
        if catagories:
            for c in catagories:
                if query:
                    if instance.filterMethod == OR:
                        query = query | Q(product__categories__id=c.id)
                    else:
                        query = query & Q(product__categories__id=c.id)
                else:
                    query = Q(product__categories__id=c.id)

        # tag
        if hasTags:
            if instance.tag:
                tags = [int(t) for t in instance.tag.split(",")]
            else:
                tags = []

            if tags:
                for t in tags:
                    if query:
                        if instance.filterMethod == OR:
                            query = query | Q(product__tags__id=t)
                        else:
                            query = query & Q(product__tags__id=t)
                    else:
                        query = Q(product__tags__id=t)

        if query:
            queryset = ProductRecord.objects.filter(query)
        else:
            queryset = ProductRecord.objects.all()

        if queryset.count() >= instance.limit:
            topViewProducts = \
                queryset.order_by('-num_views')[:instance.limit]
        else:
            topViewProducts = queryset.order_by('-num_views')
        products = [x.product for x in topViewProducts]
        context.update({
            'products': products,
        })
        
        return context

class LatestPlugin(CMSPluginBase):
    model = OscarLatest
    module = "Zephyrus Oscar"
    name = "Latest Products"
    render_template = settings.ZEPHYRUS_OSCAR_LATEST_DEFAULT_TEMPLATE
    form = OscarLatestForm

    def get_render_template(self, context, instance, placeholder):
        return select_template([
            instance.renderTemplate,
            settings.ZEPHYRUS_OSCAR_LATEST_DEFAULT_TEMPLATE,
        ])

    def render(self, context, instance, placeholder):
        catagories = instance.catagory.all()
        query = None

        # category
        if catagories.exists():
            for c in catagories:
                if query:
                    if instance.filterMethod == OR:
                        query = query | Q(categories__id=c.id)
                    else:
                        query = query & Q(categories__id=c.id)
                else:
                    query = Q(categories__id=c.id)

        # tag
        if hasTags:
            if instance.tag:
                tags = [int(t) for t in instance.tag.split(",")]
            else:
                tags = []

            if tags:
                for t in tags:
                    if query:
                        if instance.filterMethod == OR:
                            query = query | Q(tags__id=t)
                        else:
                            query = query & Q(tags__id=t)
                    else:
                        query = Q(tags__id=t)

        if query:
            queryset = Product.objects.filter(query)
        else:
            queryset = Product.objects.all()

        if queryset.count() >= instance.limit:
            products = queryset.order_by('-date_updated')[:instance.limit]
        else:
            products = queryset.order_by('-date_updated')
        context.update({
            'products': products,
        })
        
        return context

class OfferPlugin(CMSPluginBase):
    model = OscarOffer
    module = "Zephyrus Oscar"
    name = "Offer Products"
    render_template = settings.ZEPHYRUS_OSCAR_OFFER_DEFAULT_TEMPLATE
    
    def get_render_template(self, context, instance, placeholder):
        return select_template([
            instance.renderTemplate,
            settings.ZEPHYRUS_OSCAR_OFFER_DEFAULT_TEMPLATE,
        ])

    def render(self, context, instance, placeholder):
        products = []
        for offer in instance.offer.all():
            for p in offer.products():
                if not p in products:
                    products.append(p)

        context.update({
            'products': products,
        })
        
        return context


    
plugin_pool.register_plugin(FeaturedPlugin)
plugin_pool.register_plugin(LatestPlugin)
plugin_pool.register_plugin(OfferPlugin)
