#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 11982 2020-03-19 11:39:46Z Lavender $
#
# Copyright (c) 2018 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $ (last)
# $Date: 2020-03-19 20:39:46 +0900 (週四, 19 三月 2020) $
# $Revision: 11982 $

import os
import sys
import base64 
import warnings
import json
import six
import datetime
import urllib
import hashlib

from decimal import *
from collections import OrderedDict

import twder

from Iuppiter.Encoding import utf8
from Iuppiter.Encryption import Encryptor
from Iuppiter.DjangoUtil import DJANGO_VERSION, reverse
from Iuno import warnWithout, update, require
from Iuppiter.DjangoUtil import (extendInstalledApps, 
                                 extendMiddlewareClasses, 
                                 extendTemplateContextProcessors,
                                 extendTemplateDirs, 
                                 extendDatabaseDefaults, 
                                 extendAuthenticationBackend)

from django.utils.translation import ugettext_lazy as _
from django.utils.translation import ugettext as _2

from Iuno.Patch import patterns

from Iuppiter.Logging import createLogger

logger = createLogger(__name__)

CKEDITOR_CONFIGS = {
    'default': {
        'skin': 'moono',
        # 'skin': 'office2013',
        'toolbar_Basic': [
            ['Source', '-', 'Bold', 'Italic']
        ],
        'toolbar_YourCustomToolbarConfig': [
            {'name': 'document', 'items': [
                'Source', '-', 'Save', 'NewPage', 
                'Preview', 'Print', '-', 'Templates']},
            {'name': 'clipboard', 'items': [
                'Cut', 'Copy', 'Paste', 'PasteText', 
                'PasteFromWord', '-', 'Undo', 'Redo']},
            {'name': 'editing', 'items': ['Find', 'Replace', '-', 'SelectAll']},
            {'name': 'forms',
             'items': ['Form', 'Checkbox', 'Radio', 'TextField', 
                       'Textarea', 'Select', 'Button', 'ImageButton',
                       'HiddenField']},
            '/',
            {'name': 'basicstyles',
             'items': ['Bold', 'Italic', 'Underline', 'Strike', 
                       'Subscript', 'Superscript', '-', 'RemoveFormat']},
            {'name': 'paragraph',
             'items': ['NumberedList', 'BulletedList', '-', 'Outdent', 
                       'Indent', '-', 'Blockquote', 'CreateDiv', '-',
                       'JustifyLeft', 'JustifyCenter', 'JustifyRight', 
                       'JustifyBlock', '-', 'BidiLtr', 'BidiRtl',
                       'Language']},
            {'name': 'links', 'items': ['Link', 'Unlink', 'Anchor']},
            {'name': 'insert',
             'items': ['Image', 'Flash', 'Table', 'HorizontalRule', 
                       'Smiley', 'SpecialChar', 'PageBreak', 'Iframe']},
            '/',
            {'name': 'styles', 'items': [
                'Styles', 'Format', 'Font', 'FontSize']},
            {'name': 'colors', 'items': ['TextColor', 'BGColor']},
            {'name': 'tools', 'items': ['Maximize', 'ShowBlocks']},
            {'name': 'about', 'items': ['About']},
            '/',  # put this to force next toolbar on new line
            {'name': 'yourcustomtools', 'items': [
                # put the name of your editor.ui.addButton here
                'Preview',
                'Maximize',

            ]},
        ],
        'toolbar': 'YourCustomToolbarConfig',  
        'tabSpaces': 4,
        'extraPlugins': ','.join([
            'uploadimage', # the upload image feature
            # your extra plugins here
            'div',
            'autolink',
            'autoembed',
            'embedsemantic',
            'autogrow',
            # 'devtools',
            'widget',
            'lineutils',
            'clipboard',
            'dialog',
            'dialogui',
            'elementspath'
        ]),
    }
}

def attachSettings(settingsDict, enablePaypal=False):
    """
    Attach default settings to settings.py.
    It must be called in settings by passing locals().

    @param settingsDict locals() in settings.py.
    @param enablePaypal enable pay with paypal or not
    """
    from oscar import VERSION
    
    #=========================================================================
    # Django plugins.
    extraApps = []
    
    if 'IUNO_SHOP_EXTRA_APPS' in settingsDict:
        extraApps = settingsDict['IUNO_SHOP_EXTRA_APPS']

    if VERSION >= (2, 0): # 大於版本 2.0
        from oscar import INSTALLED_APPS
        
        def get_core_apps(overrides=None):
            """
            Return a list of oscar's apps amended with any passed overrides
            """
            if not overrides:
                return INSTALLED_APPS

            # Conservative import to ensure that this file can be loaded
            # without the presence Django.
            from django.utils import six
            if isinstance(overrides, six.string_types):
                raise ValueError(
                    "get_core_apps expects a list or tuple of apps "
                    "to override")

            def get_app_label(app_label, overrides):
                pattern = app_label.replace('oscar.apps.', '')
                for override in overrides:
                    if override.endswith(pattern):
                        if 'dashboard' in override and 'dashboard' not in pattern:
                            continue
                        return override
                return app_label

            apps = []
            for app_label in INSTALLED_APPS:
                apps.append(get_app_label(app_label, overrides))
            return apps
            
        ADDING_INSTALLED_APPS = (
            'django.contrib.sites',
            'django.contrib.flatpages',
            'django.contrib.humanize',
            'widget_tweaks',
            'sorl.thumbnail',
        ) + tuple(get_core_apps(extraApps))
    else:
        from oscar import get_core_apps
        
        ADDING_INSTALLED_APPS = (
            'django.contrib.sites',
            'django.contrib.flatpages',
            'widget_tweaks',
        ) + tuple(get_core_apps(extraApps))
    
    
    ADDING_INSTALLED_APPS += (
        'Iuno.shop',
        'Iuno.shop.promotions',
        'Iuno.shop.dashboard_promotions',
        'ckeditor',
        'ckeditor_uploader',
    )
    
    ADDING_MIDDLEWARE_CLASSES = (
        'oscar.apps.basket.middleware.BasketMiddleware',
        'django.contrib.flatpages.middleware.FlatpageFallbackMiddleware',
    )

    installed = settingsDict['INSTALLED_APPS']
    keys = [k for k in installed if k.startswith('django.contrib')]
    extendInstalledApps(settingsDict, ADDING_INSTALLED_APPS,
                        key=list(keys)[-1])

    extendMiddlewareClasses(settingsDict, ADDING_MIDDLEWARE_CLASSES,
                            key='django.middleware.security.SecurityMiddleware')
    #=========================================================================
    
    #=========================================================================
    # Email.
    emailDefaults = {
        'ACCOUNT_EMAIL_CONFIRMATION_REQUIRED' : True,
    }

    update(settingsDict, emailDefaults)
    settingsDict['OSCAR_FROM_EMAIL'] = settingsDict['DEFAULT_FROM_EMAIL']
    #=========================================================================
    
    #=========================================================================
    # authentication backends.
    authenticationBackends = [            
        'django.contrib.auth.backends.ModelBackend',
        'oscar.apps.customer.auth_backends.EmailBackend',
    ]

    authenticationBackends = tuple(authenticationBackends)
    
    extendAuthenticationBackend(settingsDict, authenticationBackends)
    #=========================================================================
    
    #=========================================================================
    #  Django templates.
    #  Context processors
    ADDING_TEMPLATES_CONTEXT_PROCESSORS = (
        'oscar.apps.search.context_processors.search_form',
        'oscar.apps.checkout.context_processors.checkout',
        'oscar.apps.customer.notifications.context_processors.notifications',
        'oscar.core.context_processors.metadata',                       
    )
    
    if not VERSION >= (2, 0):
        ADDING_TEMPLATES_CONTEXT_PROCESSORS += (
            'oscar.apps.promotions.context_processors.promotions',
        )
    
    extendTemplateContextProcessors(settingsDict, 
                                    ADDING_TEMPLATES_CONTEXT_PROCESSORS)

    overrideTemplateDir = os.path.join(
        os.path.dirname(os.path.abspath(__file__)), 
        os.path.join("templates", "shop", "override"))
        
    # Template tags
    if VERSION >= (2, 0):
        from oscar import config
        OSCAR_MAIN_TEMPLATE_DIR = os.path.join(
            os.path.dirname(
                os.path.abspath(config.__file__)), 'templates/oscar')
    else:
        from oscar import OSCAR_MAIN_TEMPLATE_DIR
        
    ADDING_TEMPLATES_DIRS = [
        os.path.join(settingsDict['BASE_DIR'], 'templates'),
        overrideTemplateDir,
        OSCAR_MAIN_TEMPLATE_DIR,
    ]
    extendTemplateDirs(settingsDict, ADDING_TEMPLATES_DIRS)
    #=========================================================================
    # Django database default values.
    ADDING_DATABASES_DEFAULTS = {
        'ATOMIC_REQUESTS': True
    }
    extendDatabaseDefaults(settingsDict, ADDING_DATABASES_DEFAULTS)
    #=========================================================================
    # Django haystack connections.
    BASE_DIR = settingsDict['BASE_DIR']
    if settingsDict['DEBUG'] or sys.platform.startswith("win32"):
        ADDING_HAYSTACK_CONNECTIONS = {
            'HAYSTACK_CONNECTIONS': {
                'default': {
                    'ENGINE': (
                        'haystack.backends.whoosh_backend.WhooshEngine'),
                    'PATH': os.path.join(BASE_DIR, 'whoosh_index'),
                },
            },
        }
    else:
        ADDING_HAYSTACK_CONNECTIONS = {
            'HAYSTACK_CONNECTIONS': {
                'default': {
                    'ENGINE': 'xapian_backend.XapianEngine',
                    'PATH': os.path.join(BASE_DIR, 'xapian_index'),
                },
            },
        }
        
    update(settingsDict, ADDING_HAYSTACK_CONNECTIONS)
    #=========================================================================

    #=========================================================================
    # Static files setting for Iuno.shop
    location = lambda x: os.path.join(
        os.path.dirname(os.path.realpath(settingsDict['__file__'])), x)

    if settingsDict['DEBUG']:
        addStaticRoot = {
            'STATIC_ROOT': location('static')
        }
        update(settingsDict, addStaticRoot)
    elif settingsDict['DEBUG'] == False:
        ADD_STATICFILES_FINDERS = {
            'STATICFILES_FINDERS': [
                'django.contrib.staticfiles.finders.FileSystemFinder',
                'django.contrib.staticfiles.finders.AppDirectoriesFinder',
            ]
        }
        update(settingsDict, ADD_STATICFILES_FINDERS)

    #=========================================================================
    if 'OSCAR_DEFAULT_CURRENCY' not in settingsDict:
        settingsDict['OSCAR_DEFAULT_CURRENCY'] = 'USD'
    #=========================================================================
    # Oscar default settings 
    # __import__ 模組的用法與官方文件有落差，可能是使用時帶入參數有誤，
    # 先以目前的寫法帶過
    # see #2784
    addOscarDefaultsPackages = {}
    _level = -1 if six.PY2 else 0
    oscarDefaultsPackages = __import__(
            'oscar.defaults', globals(), locals(), [], _level
        ).__dict__['defaults']
        
    for attr in dir(oscarDefaultsPackages):
        addOscarDefaultsPackages[attr] = getattr(
            oscarDefaultsPackages, attr)
    update(settingsDict, addOscarDefaultsPackages)
    #=========================================================================

    #=========================================================================
    # payment
    defaultSettings = {
        'SESSION_COOKIE_SAMESITE' : None,
    }

    update(settingsDict, defaultSettings)
    
    ecpayEnable = False
    if 'IUNO_SHOP_ECPAY_ENABLE' in settingsDict:
        ecpayEnable = settingsDict['IUNO_SHOP_ECPAY_ENABLE']

    allpayEnable = False
    if 'IUNO_SHOP_ALLPAY_ENABLE' in settingsDict:
        allpayEnable = settingsDict['IUNO_SHOP_ALLPAY_ENABLE']

    spgatewayEnable = False
    if 'IUNO_SHOP_SPGATEWAY_ENABLE' in settingsDict:
        spgatewayEnable = settingsDict['IUNO_SHOP_SPGATEWAY_ENABLE']

    cathaybkEnable = False
    if 'IUNO_SHOP_CATHAYBK_ENABLE' in settingsDict:
        cathaybkEnable = settingsDict['IUNO_SHOP_CATHAYBK_ENABLE']
        
    atmEnable = False
    if hasattr(settings, 'IUNO_SHOP_ATM_ENABLE'):
        atmEnable = settings.IUNO_SHOP_ATM_ENABLE
        
    if atmEnable:
        require('IUNO_SHOP_ATM_EXPIRED_HOURS', settingsDict)

    ADDING_OSCAR_DASHBOARD_NAVIGATION = {
        'label': _('Payment'),
        'icon': 'icon-globe',
        'children': [
           
        ]
    }
    
    def accessFunction(user, urlName, urlArgs, urlKwargs):
        return user.is_staff


    if enablePaypal:
        require('PAYPAL_API_USERNAME', settingsDict)
        require('PAYPAL_API_PASSWORD', settingsDict)
        require('PAYPAL_API_SIGNATURE', settingsDict)

        ADDING_PAYPAL_INSTALLED_APPS = (
            'paypal',
        )
        extendInstalledApps(settingsDict, ADDING_PAYPAL_INSTALLED_APPS,
                        key=list(keys)[-1])


        ADDING_OSCAR_SHOP_TAGLINE = {
            'OSCAR_SHOP_TAGLINE' : 'Paypal'
        }
        update(settingsDict, ADDING_OSCAR_SHOP_TAGLINE)

        ADDING_OSCAR_DASHBOARD_NAVIGATION['children'].append({
            'label': _('Paypal transactions'),
            'url_name': 'paypal-express-list',
            'access_fn': accessFunction,
        })
        
    if ecpayEnable:
        ADDING_OSCAR_DASHBOARD_NAVIGATION['children'].append({
            'label': _('ECPay transactions'),
            'url_name': 'ecpayTransaction',
            'access_fn': accessFunction,
        })
        ADDING_OSCAR_DASHBOARD_NAVIGATION['children'].append({
            'label': _('ECPay logistic transactions'),
            'url_name': 'ecpayLogisticTransaction',
            'access_fn': accessFunction,
        })
    if allpayEnable:
        ADDING_OSCAR_DASHBOARD_NAVIGATION['children'].append({
            'label': _('AllPay transactions'),
            'url_name': 'allpayTransaction',
            'access_fn': accessFunction,
        })
    if spgatewayEnable:
        ADDING_OSCAR_DASHBOARD_NAVIGATION['children'].append({
            'label': _('Spgateway transactions'),
            'url_name': 'spgatewayTransaction',
            'access_fn': accessFunction,
        })
    if cathaybkEnable:
        ADDING_OSCAR_DASHBOARD_NAVIGATION['children'].append({
            'label': _('Cathay credit transactions'),
            'url_name': 'cathaybkCreditTransaction',
            'access_fn': accessFunction,
        })

    if enablePaypal or ecpayEnable or allpayEnable or spgatewayEnable:
        if 'OSCAR_DASHBOARD_NAVIGATION' in settingsDict:
            settingsDict['OSCAR_DASHBOARD_NAVIGATION'].append(
                ADDING_OSCAR_DASHBOARD_NAVIGATION)
        else:
            settingsDict['OSCAR_DASHBOARD_NAVIGATION'] = \
                                            ADDING_OSCAR_DASHBOARD_NAVIGATION
                                            
    # bonus
    for s in settingsDict['OSCAR_DASHBOARD_NAVIGATION']:
        addUrl = False
        for child in s.get('children', []):
            if child.get('url_name') == 'dashboard:voucher-list':
                addUrl = True
        if addUrl:
            s['children'].append({
                'label': _('Bonus settings'),
                'url_name': 'bonusSettings',
                'access_fn': accessFunction,
            })
    #=========================================================================
    
    #=========================================================================
    # Django ckeditor settings
    DJANGO_CKEDITOR_CONF = {
        'CKEDITOR_UPLOAD_PATH': "uploads/",
        'CKEDITOR_CONFIGS': CKEDITOR_CONFIGS,
    }
    update(settingsDict, DJANGO_CKEDITOR_CONF)
    #=========================================================================
    
    #=========================================================================
    # Oscar other settings OSCAR_SLUG_ALLOW_UNICODE
    if DJANGO_VERSION >= 11100: # django >= 1.11
        settingsDict['OSCAR_SLUG_ALLOW_UNICODE'] = True
    #=========================================================================

    #=========================================================================
    # Django order status settings
    allSettings = [
        'OSCAR_INITIAL_ORDER_STATUS',
        'OSCAR_INITIAL_LINE_STATUS',
        'OSCAR_ORDER_STATUS_PIPELINE',
        'OSCAR_PROCESSING_SEND_EMAIL_STATUS',
        'OSCAR_ARRIVE_SEND_EMAIL_STATUS',

        'OSCAR_ORDER_RETURN_STATUS',
        'OSCAR_ORDER_PROCESSING_STATUS',
        'OSCAR_ORDER_COMPLETE_STATUS',
        'OSCAR_ORDER_ARRIVE_STATUS',
    ]

    # add default settings
    needShipping = False
    if 'IUNO_SHOP_NEED_SHIPPING' in settingsDict:
        needShipping = settingsDict['IUNO_SHOP_NEED_SHIPPING']
    
    hasConfig = False
    for s in allSettings:
        if s in settingsDict:
             hasConfig = True
             break

    if not hasConfig:
        if needShipping:
            ADDING_OSCAR_ORDER_STATUS = {
                'OSCAR_PROCESSING_SEND_EMAIL_STATUS': u'商品送貨中',
                'OSCAR_ARRIVE_SEND_EMAIL_STATUS': u'商品已到貨',
                'OSCAR_INITIAL_ORDER_STATUS': u'等待處理中',
                'OSCAR_INITIAL_LINE_STATUS': '-',
                'OSCAR_ORDER_RETURN_STATUS': u'商品退貨中',
                'OSCAR_ORDER_PROCESSING_STATUS': u'商品送貨中',
                'OSCAR_ORDER_COMPLETE_STATUS': u'完成訂單',
                'OSCAR_ORDER_ARRIVE_STATUS': u'商品已到貨',
                'OSCAR_ORDER_STATUS_PIPELINE': {
                    u'等待處理中': \
                        (u'商品送貨中', u'訂單取消', u'測試用訂單'),
                    u'商品送貨中': \
                        (u'完成訂單', u'商品已到貨', 
                         u'訂單取消', u'測試用訂單'),
                    u'商品已到貨': (u'完成訂單', u'訂單取消', u'測試用訂單',),
                    u'訂單取消': (u'測試用訂單', ),
                    u'完成訂單': (u'測試用訂單', ),

                    u'商品退貨中': (u'完成退貨', u'測試用訂單'),
                    u'完成退貨': (u'測試用訂單',),
                    u'測試用訂單': (u'等待處理中',),
                },
                'OSCAR_ORDER_STATUS': [
                    (u'等待處理中', u'等待處理中'),
                    (u'商品送貨中', u'商品送貨中'),
                    (u'商品已到貨', u'商品已到貨'),
                    (u'訂單取消', u'訂單取消'),
                    (u'完成訂單', u'完成訂單'),
                    (u'商品退貨中', u'商品退貨中'),
                    (u'完成退貨', u'完成退貨'),
                    (u'測試用訂單', u'測試用訂單'),
                ],
            }
        else:
            ADDING_OSCAR_ORDER_STATUS = {
                'OSCAR_INITIAL_ORDER_STATUS': u'完成訂單',
                'OSCAR_ORDER_COMPLETE_STATUS': u'完成訂單',
                'OSCAR_INITIAL_LINE_STATUS': '-',
                'OSCAR_ORDER_STATUS_PIPELINE': {
                    u'完成訂單': (u'測試用訂單', ),
                    u'測試用訂單': (u'完成訂單',),
                },
                'OSCAR_ORDER_STATUS': [
                    (u'完成訂單', u'完成訂單'),
                    (u'測試用訂單', u'測試用訂單'),
                ],
            }

        update(settingsDict, ADDING_OSCAR_ORDER_STATUS)
    else:
        # check settings:
        unsettings = [s for s in allSettings if not s in settingsDict]
        if unsettings:
            raise KeyError("You should also set '%s' value." % str(unsettings))
    
    #=========================================================================
    
    #=========================================================================
    # oscar ckeditor dashboard
    DEFAULT_CKEDITOR_SETTINGS = {
        'IUNO_SHOP_PRODUCT_DESCRIPTION_CKEDITOR': True,
        'IUNO_SHOP_CONTENT_BLOCK_CKEDITOR': True,
        'IUNO_SHOP_PAGE_CKEDITOR': True,
    }
    update(settingsDict, DEFAULT_CKEDITOR_SETTINGS)
    
    #=========================================================================
    
    #=========================================================================
    # promotions
    DEFAULT_PROMOTION_SETTINGS = {
        'OSCAR_PROMOTION_FOLDER': 'images/promotions/',
        'OSCAR_PROMOTION_POSITIONS': (
            ('page', 'Page'),
            ('right', 'Right-hand sidebar'),
            ('left', 'Left-hand sidebar')
        ),
        'OSCAR_PROMOTIONS_ENABLED': True,
    }
    update(settingsDict, DEFAULT_PROMOTION_SETTINGS)

    # 用來加 dashboard 頁面，把 promotion 加到跟 page 同個 Tab 下
    for s in settingsDict['OSCAR_DASHBOARD_NAVIGATION']:
        addUrl = False
        for child in s.get('children', []):
            if child.get('url_name') == 'dashboard:page-list':
                addUrl = True
        if addUrl:
            s['children'].append({
                'label': _('Content blocks'),
                'url_name': 'promotion-list',
            })
            s['children'].append({
                'label': _('Content blocks by page'),
                'url_name': 'promotion-list-by-page',
            })
    
    

def attachShopURLs(settings, urlpatterns):
    """
    Attach shop related urlpatterns.
    It must be called in urls.py by passing urlpatterns.
 
    @param settings Django project's settings module.
    @param urlpatterns urlpatterns in urls.py.
    """
    from Iuno.shop import views

    from django.conf.urls import url, include
    from django.contrib.admin.views.decorators import staff_member_required
    from oscar import VERSION       
    
    urlpatterns += patterns('',
        # the i18n is the default url settings of django oscar, but generally
        # we are not used.
        url(r'^i18n/', include('django.conf.urls.i18n')),
        url(r'^checkout/payment/redirect/', 
            views.sendPaymentForm, name="formRedirect"),
        url(r'^shop/dashboard/bonus/', 
            staff_member_required(views.BonusSettingsView.as_view()), 
            name="bonusSettings"),
        url(r'^shop/dashboard/users/(?P<pk>-?\d+)/bonus/update/$',
            staff_member_required(views.updateBonus),
            name='updateBonus'),
        url(r'^shop/basket/bonus',
            views.useBonus, name="useBonus"),
        url(r'^shop/customer/bonus',
            views.CustomerBonusListView.as_view(), name="customerBonusList"),
        url(r'^shop/send_confirm_code/$', views.sendConfirmCode, 
            name='sendConfirmCode'),
    )
    if VERSION >= (2, 0, 0):
        from django.apps import apps
        urlpatterns += patterns('',
            url(r'^shop/', include(apps.get_app_config('oscar').urls[0])),
            url(r'^shop/dashboard/catalogue'
                r'/products/(?P<pk>\d+)/(?P<action>.*)/$', 
                staff_member_required(views.ProductPublicView.as_view()),
                name='dashboardProductPublic'),
            # promotions
            url(r'^shop/', include(apps.get_app_config('promotions').urls[0])),
            
            # dashboard_promotions
            url(r'^shop/dashboard/content-blocks/', include(apps.get_app_config('dashboard_promotions').urls[0])),
        )
    else:
        from oscar.app import application as shop
        urlpatterns += patterns('',
            url(r'^shop/', shop.urls),
        )
    
    ckeditorEnable = False
    if hasattr(settings, 'IUNO_SHOP_PRODUCT_DESCRIPTION_CKEDITOR'):
        ckeditorEnable = settings.IUNO_SHOP_PRODUCT_DESCRIPTION_CKEDITOR
    
    if ckeditorEnable:
        urlpatterns += patterns('',
            url(r'^ckeditor/', include('ckeditor_uploader.urls')),
        )

    paypalEnable = False
    if hasattr(settings, 'IUNO_SHOP_PAYPAL_ENABLE'):
        paypalEnable = settings.IUNO_SHOP_PAYPAL_ENABLE
    
    ecpayEnable = False
    if hasattr(settings, 'IUNO_SHOP_ECPAY_ENABLE'):
        ecpayEnable = settings.IUNO_SHOP_ECPAY_ENABLE

    allpayEnable = False
    if hasattr(settings, 'IUNO_SHOP_ALLPAY_ENABLE'):
        allpayEnable = settings.IUNO_SHOP_ALLPAY_ENABLE

    spgatewayEnable = False
    if hasattr(settings, 'IUNO_SHOP_SPGATEWAY_ENABLE'):
        spgatewayEnable = settings.IUNO_SHOP_SPGATEWAY_ENABLE

    cathaybkEnable = False
    if hasattr(settings, 'IUNO_SHOP_CATHAYBK_ENABLE'):
        cathaybkEnable = settings.IUNO_SHOP_CATHAYBK_ENABLE
        
    atmEnable = False
    if hasattr(settings, 'IUNO_SHOP_ATM_ENABLE'):
        atmEnable = settings.IUNO_SHOP_ATM_ENABLE

    if paypalEnable:
        from paypal.express.dashboard import views as expressViews
        urlpatterns += patterns('',          
            url(r'^shop/dashboard/paypal/express/transactions/$', 
                expressViews.TransactionListView.as_view(),
                name='paypal-express-list'),
            url(r'^shop/dashboard/paypal/express/transactions/(?P<pk>\d+)/$', 
                expressViews.TransactionDetailView.as_view(),
                name='paypal-express-detail'),
            url(r'^checkout/paypal/place-order/(?P<basket_id>\d+)/$',
                views.SuccessResponseView.as_view(), 
                name='paypal-place-order'),
            url(r'^checkout/paypal/', include('paypal.express.urls')),
        )

    if cathaybkEnable:
        urlpatterns += patterns('',
            url(r'^cathaybk-credit/checkxml/', views.returnCathaybkCreditXml),
        )

    if (ecpayEnable or allpayEnable or spgatewayEnable or 
        cathaybkEnable or atmEnable):
        urlpatterns += patterns('',
            url(r'^(?P<name>.*)/preview/', 
                views.SuccessResponseView2.as_view(), name='paymentPreview'),
            url(r'^shop/orders/(?P<orderNumber>[\w-]*)/refund/', 
                views.RefundView.as_view(),
                name='refund'),
        )
    if ecpayEnable:
        urlpatterns += patterns('',
            url(r'^dashboard/ecpay/transaction/', 
                staff_member_required(
                    views.ECPayTransactionListView.as_view()),
                name='ecpayTransaction'),
            url(r'^dashboard/ecpaylogistic/transaction/', 
                staff_member_required(
                    views.ECPayLogisticTransactionListView.as_view()),
                name='ecpayLogisticTransaction'),
            url(r'^shop/ecpaylogistic/reply/', 
                views.ECPayServerReplyView.as_view(),
                name='ecpayLogisticReply'),
        )
    if allpayEnable:
        urlpatterns += patterns('',
            url(r'^dashboard/allpay/transaction/', 
                staff_member_required(
                    views.AllPayTransactionListView.as_view()),
                name='allpayTransaction'),
        )
    if spgatewayEnable:
        urlpatterns += patterns('',
            url(r'^dashboard/spgateway/transaction/', 
                staff_member_required(
                    views.SpgatewayTransactionListView.as_view()),
                name='spgatewayTransaction'),
        )

    if cathaybkEnable:
        urlpatterns += patterns('',
            url(r'^dashboard/spgateway/transaction/', 
                staff_member_required(
                    views.CathaybkCreditTransactionListView.as_view()),
                name='cathaybkCreditTransaction'),
        )
        
    if atmEnable:
        urlpatterns += patterns('',
            url(r'^shop/atm/(?P<orderNumber>[\w-]*)/fill/', 
                views.ATMFillView.as_view(),
                name='atmFill'),
        )

# ------------------------------------------------------------------------------
import hashlib
import time
import datetime

from django.conf import settings
from django.template.loader import get_template
from django.template import Context
from django.utils.translation import ugettext_lazy as _
from django.middleware.csrf import get_token
from pyallpay import AllPay

from Iuppiter.Encoding import utf8
from Iuno.shop.Util import Pay2GoMPGAPI, CathaybkAPI, toBytes

ECPAY_SANDBOX_SERVICE_URL = \
                    'https://payment-stage.ecpay.com.tw/Cashier/AioCheckOut/V2'
ECPAY_SERVICE_URL = 'https://payment.ecpay.com.tw/Cashier/AioCheckOut/V2'

ALLPAY_SANDBOX_SERVICE_URL = \
                    'http://payment-stage.allpay.com.tw/Cashier/AioCheckOut'
ALLPAY_SERVICE_URL = 'https://payment.allpay.com.tw/Cashier/AioCheckOut'

SPGATEWAY = 'spgateway'
ALLPAY = 'allpay'
ECPAY = 'ecpay'
ECPAY_MAP = 'ecpay_map'
ECPAY_LOGISTIC = 'ecpay_logistic'
CATHAYBK = 'cathaybk'
ATM = 'ATM'

class AllPay2(AllPay):
    def __init__(self, paymentConf, serviceMethod='post'):
        if six.PY2:
            # super() argument 1 must be type, not classobj
            AllPay.__init__(self, paymentConf, service_method=serviceMethod)
        else:
            super(AllPay2, self).__init__(
                paymentConf, service_method=serviceMethod)
        if 'Language' in paymentConf:
            self.url_dict['Language'] = paymentConf['Language']

    def gen_check_out_form(self, dict_url, auto_send=True):
        """
        Generate The Form Submission
        :param dict_url:
        :return: the html of the form
        """
        form_html = utf8(
            ('<form id="allPay-Form" name="allPayForm" method="post" '
             'target="_self" action="%s" style="display: none;">') % 
             self.service_url)

        for i, val in enumerate(dict_url):
            form_html += utf8(
                "<input type='hidden' name='%s' value='%s' />" % (
                    val, dict_url[val]))

        form_html  += utf8(
            '<input type="submit" class="large" '
            'id="payment-btn" value="BUY" /></form>')
        if auto_send:
            form_html += utf8("<script>document.allPayForm.submit();</script>")
        return form_html

def buildAPIForm(submission, request, apiName, extra={}):
    scheme = request.scheme
    domain = request.META['HTTP_HOST']
    
    from django.conf import settings
    
    ecpayEnable = False
    if hasattr(settings, 'IUNO_SHOP_ECPAY_ENABLE'):
        ecpayEnable = settings.IUNO_SHOP_ECPAY_ENABLE

    allpayEnable = False
    if hasattr(settings, 'IUNO_SHOP_ALLPAY_ENABLE'):
        allpayEnable = settings.IUNO_SHOP_ALLPAY_ENABLE

    spgatewayEnable = False
    if hasattr(settings, 'IUNO_SHOP_SPGATEWAY_ENABLE'):
        spgatewayEnable = settings.IUNO_SHOP_SPGATEWAY_ENABLE

    cathaybkEnable = False
    if hasattr(settings, 'IUNO_SHOP_CATHAYBK_ENABLE'):
        cathaybkEnable = settings.IUNO_SHOP_CATHAYBK_ENABLE
        
    atmEnable = False
    if hasattr(settings, 'IUNO_SHOP_ATM_ENABLE'):
        atmEnable = settings.IUNO_SHOP_ATM_ENABLE
        
    enable = {
        SPGATEWAY: spgatewayEnable,
        ALLPAY: allpayEnable,
        ECPAY: ecpayEnable,
        CATHAYBK: cathaybkEnable,
        ECPAY_LOGISTIC: ecpayEnable,
        ECPAY_MAP: ecpayEnable,
        ATM: atmEnable,
    }

    if not apiName in [SPGATEWAY, ALLPAY, ECPAY, 
                       CATHAYBK, ECPAY_LOGISTIC, ECPAY_MAP, ATM]:
        raise ValueError("Not pay api name: %s" % apiName)
        
    if not enable[apiName]:
        logger.error("The project not enable the payment: %s" % apiName)
        return ''

    MERCHANT_TRADE_NO_LEN = 20
    
    basket = submission['basket']

    if not apiName == ECPAY_MAP:
        total = submission['order_total'].excl_tax

        # USD
        if submission['order_total'].currency == 'USD':
            total = total * Decimal(twder.now('USD')[2]) # 當前匯率
        
        # item name
        itemName = utf8('')
        for index, line in enumerate(basket.all_lines()):
            product = line.product
            
            productName = product.get_title()
            itemName += utf8(productName)
            
            itemName += utf8(_(' %(dollar)d dollar X %(number)d#') % {
                        'dollar': line.price_excl_tax, 'number': line.quantity})
        itemName += utf8(_("shipping charge %(dollar)d dollar X 1") % {
                         'dollar': submission['shipping_charge'].excl_tax})

        itemName = utf8(itemName)
        
    # merchantTradeNo
    basketId = basket.id
    tradeNo = 'BASKETID%d' % basketId
    hashCode = hashlib.sha224(
        toBytes(str(datetime.datetime.now()))).hexdigest().upper()
    lenNo = (MERCHANT_TRADE_NO_LEN - len(tradeNo))
    merchantTradeNo = str(hashCode[0:lenNo]) + tradeNo
    

    # return url
    RETURN_URL = 'http://localhost:8000/%s/preview/' % apiName
    CLIENT_BACK_URL = 'http://localhost:8000/'
    PAYMENT_INFO_URL = 'http://localhost:8000/feedback'

    returnUrl = RETURN_URL.replace(
                           'localhost:8000', domain).replace('http', scheme)
    backUrl = CLIENT_BACK_URL.replace(
                        'localhost:8000', domain).replace('http', scheme)

    form = None
    # build form data
    if apiName == SPGATEWAY:
        data = {
            'TimeStamp': time.time(),
            'RespondType': 'JSON',
            'Version': '1.2',
            'LangType': 'zh-tw',
            'MerchantID': settings.SPGATEWAY_API_MERCHANT_ID,
            'MerchantOrderNo': merchantTradeNo,
            'Amt': int(total),
            'ItemDesc': itemName.replace(toBytes('#'), toBytes('|')),
            'LoginType': 0,
        }
        
        if six.PY3:
            data['ItemDesc'] = data['ItemDesc'].decode("utf-8")

        mpg = Pay2GoMPGAPI(
            hashKey=settings.SPGATEWAY_API_HASH_KEY,
            hashIv=settings.SPGATEWAY_API_HASH_IV,
            sandbox=settings.SPGATEWAY_SANDBOX_MODE,
        )

        payment = mpg.createPayment(data)
        if DJANGO_VERSION >= 11100: # django >= 1.11
            context = {'spgateway': payment,}
        else:
            context = Context({'spgateway': payment,})
        temp = get_template('shop/payment/Spgateway.html')
        form = temp.render(context)
       
    if apiName in [ECPAY, ALLPAY]:
        data = {
            'TotalAmount': int(total), 
            'ChoosePayment': 'ALL', 
            'MerchantTradeNo': merchantTradeNo, 
            'ItemName': itemName,
            'TradeDesc' : settings.OSCAR_SHOP_NAME, 
            'Remark' : settings.OSCAR_SHOP_NAME,
        }
        
        if apiName == ECPAY and submission['order_total'].currency == 'USD':
            data['Language'] = 'ENG'
            
        if apiName == ECPAY and hasattr(settings, 'ECPAY_LANGUAGE'):
            data['Language'] = settings.ECPAY_LANGUAGE
        
        if six.PY3:
            data['ItemName'] = data['ItemName'].decode("utf-8")

        ap = AllPay2(data)

        if apiName == ECPAY:
            if settings.ECPAY_SANDBOX_MODE:
                ap.service_url = ECPAY_SANDBOX_SERVICE_URL
            else:
                ap.service_url = ECPAY_SERVICE_URL

            ap.url_dict['MerchantID'] = settings.ECPAY_API_MERCHANT_ID
            ap.HASH_KEY = settings.ECPAY_API_HASH_KEY
            ap.HASH_IV = settings.ECPAY_API_HASH_IV
        else:
            if settings.ALLPAY_SANDBOX_MODE:
                ap.service_url = ALLPAY_SANDBOX_SERVICE_URL
            else:
                ap.service_url = ALLPAY_SERVICE_URL

            ap.url_dict['MerchantID'] = settings.ALLPAY_API_MERCHANT_ID
            ap.HASH_KEY = settings.ALLPAY_API_HASH_KEY
            ap.HASH_IV = settings.ALLPAY_API_HASH_IV


        # modify return Url
        ap.url_dict['ReturnURL'] = returnUrl
        ap.url_dict['OrderResultURL'] = returnUrl
        ap.url_dict['ClientBackURL'] = backUrl

        dictUrl = ap.check_out()

        if apiName == ECPAY:
            form = ap.gen_check_out_form(dictUrl, 
                                    settings.ECPAY_AUTO_SEND_FORM)
                                    
            form = form.replace(toBytes('allPay-Form'), toBytes('ecpayForm'))
            form = form.replace(toBytes('allPayForm'), toBytes('ecpayForm'))
        else:
            form = ap.gen_check_out_form(dictUrl, 
                                    settings.ALLPAY_AUTO_SEND_FORM)
            form = form.replace(toBytes('allPay-Form'), toBytes('allpayForm'))
            form = form.replace(toBytes('allPayForm'), toBytes('allpayForm'))
       
    if apiName == CATHAYBK:
        mpg = CathaybkAPI(
            merchantId=settings.CATHAYBK_API_MERCHANT_ID,
            cubkey=settings.CATHAYBK_API_CUBKEY,
        )

        data = {
            'ORDERNUMBER': merchantTradeNo,
            'AMOUNT': int(total),
        }

        form = mpg.buildForm(data)
        
    if apiName == ECPAY_MAP:
        temp = get_template('shop/payment/EcpayMap.html')

        if settings.ECPAY_SANDBOX_MODE:
            apiUrl = "https://logistics-stage.ecpay.com.tw/Express/map"
        else:
            apiUrl = "https://logistics.ecpay.com.tw/Express/map"

        if hasattr(settings, "ECPAY_API_LOGISTIC_SUB_TYPES"):
            subTypes = settings.ECPAY_API_LOGISTIC_SUB_TYPES
        else:
            subTypes = {
                "FAMI": u"全家便利商店",
                "UNIMART": u"統一7-11超商" ,
                "HILIFE": u"萊爾富",
            }

        data = {
            'TradeDesc' : settings.OSCAR_SHOP_NAME, 
            'Remark' : settings.OSCAR_SHOP_NAME,
        }

        if submission['shipping_address']:
            needExtraData = False
        else:
            needExtraData = True

        if DJANGO_VERSION >= 11100: # django >= 1.11
            context = {
                'apiUrl': apiUrl,
                'chantID': settings.ECPAY_API_MERCHANT_ID,
                'subTypes': subTypes,
                'returnUrl': returnUrl,
                'needExtraData': needExtraData,
                'extraData': json.dumps(
                    data, ensure_ascii=False, 
                    default=str).replace("\"", "&quot;"),
            }
        else:
            context = Context({
                'apiUrl': apiUrl,
                'chantID': settings.ECPAY_API_MERCHANT_ID,
                'subTypes': subTypes,
                'returnUrl': returnUrl,
                'needExtraData': needExtraData,
                'extraData': json.dumps(
                    data, ensure_ascii=False, 
                    default=str).replace("\"", "&quot;"),
            })
        form = temp.render(context)
        
        return form
        
    if apiName == ECPAY_LOGISTIC and request.session.get('logisticMap'):
        mapData = json.loads(request.session['logisticMap'])
        
        if settings.ECPAY_SANDBOX_MODE:
            apiUrl = "https://logistics-stage.ecpay.com.tw/Express/Create"
        else:
            apiUrl = "https://logistics.ecpay.com.tw/Express/Create"

        if hasattr(settings, "OSCAR_SHOP_NAME_FOR_ECPAY"):
            shopName = settings.OSCAR_SHOP_NAME_FOR_ECPAY
        else:
            shopName = settings.OSCAR_SHOP_NAME

        if hasattr(settings, "OSCAR_SHOP_PHONE"):
            shopPhone = settings.OSCAR_SHOP_PHONE 
        else:
            shopPhone = ''

        if "isCollection" in extra:
            isCollection = extra["isCollection"]
        else:
            isCollection = "Y"

        data = {
            "MerchantID": settings.ECPAY_API_MERCHANT_ID,
            "MerchantTradeNo": merchantTradeNo,
            "MerchantTradeDate": datetime.datetime.now().strftime(
                                    "%Y/%m/%d %H:%M:%S"),

            "LogisticsType": "CVS",
            "LogisticsSubType": mapData['storeType'],
            "IsCollection": isCollection,
            "ReceiverStoreID": mapData['storeId'],
            
            "GoodsAmount": int(total),
            "CollectionAmount": int(total),
           
            
            # 必須 len(SendName) <= 10 個字元 
            "SenderName": shopName,
            "SenderCellPhone": shopPhone,

            "ReceiverName": mapData['receiverName'], 
            "ReceiverCellPhone": mapData['receiverPhone'], 
            
            # 隨時接訂單資訊的 URL
            "ServerReplyURL": backUrl[:-1] + reverse("ecpayLogisticReply"), 
            "ClientReplyURL": returnUrl, 
        }
        
        if six.PY3:
            data["GoodsName"] = itemName.replace(
                "#".encode(), ' '.encode()).replace(
                "(".encode(), ''.encode()).replace(
                ")".encode(), ''.encode()).decode('utf8')
        else:
            data["GoodsName"] = itemName.replace(
                "#", ' ').replace("(", '').replace(")", '').decode('utf8')
            
        if mapData['storeType'] == "UNIMARTC2C":
            data["LogisticsC2CReplyURL"] = returnUrl
        
        data = OrderedDict(data)
        
        if six.PY3:
            data = OrderedDict(sorted(data.items()))
        else:
            data = OrderedDict(sorted(data.iteritems()))

        orderedDict = OrderedDict()
        orderedDict['HashKey'] = settings.ECPAY_API_HASH_KEY
        for field in data:
            orderedDict[field] = data[field]

        orderedDict['HashIV'] = settings.ECPAY_API_HASH_IV

        dataList = []
        for k, v in orderedDict.items():
            dataList.append("%s=%s" % (k, v))
        dataStr = u"&".join(dataList)

        if six.PY3:
            encodeStr = urllib.parse.urlencode({'data': utf8(dataStr),})[5:]
            checkValue = hashlib.md5(
                encodeStr.lower().encode()).hexdigest().upper()
        else:
            encodeStr = urllib.urlencode({'data': utf8(dataStr),})[5:]
            checkValue = hashlib.md5(encodeStr.lower()).hexdigest().upper()
        data["CheckMacValue"] = checkValue

        temp = get_template('shop/payment/EcpayLogistic.html')
        if DJANGO_VERSION >= 11100: # django >= 1.11
            context = {
                'apiUrl': apiUrl,
                'data': data,
            }
        else:
            context = Context({
                'apiUrl': apiUrl,
                'data': data,
            })
        form = temp.render(context)
        
        return form
        
    if apiName == ATM:
        temp = get_template('shop/payment/ATM.html')
        context = {
            'data': {
                'total': int(total),
                'basketId': request.basket.id,
            },
        }
        form = temp.render(context)
        
    if not form:
        return None
    else:             
        if six.PY3:
            if not isinstance(form, bytes):
                form = str(form).encode()
            encryptor = Encryptor(settings.SECRET_KEY.encode())
            formData = encryptor.encrypt(form)
        else:
            encryptor = Encryptor(settings.SECRET_KEY)
            formData = encryptor.encrypt(form)
        
        form = base64.b64encode(formData)
        
        if six.PY3:
            form = form.decode()
            
        temp = get_template('shop/payment/PaymentForm.html')
        if DJANGO_VERSION >= 11100: # django >= 1.11
            context = {
                "csrf": get_token(request),
                "formId": "%sForm" % apiName,
                'formData': form,
            }
        else:
            context = Context({
                "csrf": get_token(request),
                "formId": "%sForm" % apiName,
                'formData': form,
            })
        form = temp.render(context)
        
        return form