#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: admin.py 11069 2018-05-21 10:18:52Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2018-05-21 18:18:52 +0800 (週一, 21 五月 2018) $
# $Revision: 11069 $

import datetime
import logging
import urllib
import six
import hashlib
import pdfkit
import os

from bs4 import BeautifulSoup
from unidecode import unidecode

from django.contrib import auth
from django.contrib.sites.models import Site
from django.conf import settings
from django.conf.urls import url
from django.core.mail import send_mail
from django.core.validators import validate_email
from django.core.exceptions import ValidationError
from django.shortcuts import redirect, get_object_or_404, render
from django.template.loader import get_template
from django.template.defaultfilters import slugify
from django.utils.http import urlquote  as django_urlquote
from django.views.generic.detail import DetailView
from django.views.generic import View
from django.views.static import serve
from django.utils.translation import ugettext_lazy as _
from django.urls import reverse
from django.http import (
    HttpResponseRedirect,
    HttpResponsePermanentRedirect,
)

from djangocms_forms.models import FormDefinition
from djangocms_forms.utils import hashid_to_int, int_to_hashid
from djangocms_forms.forms import FormBuilder
from djangocms_text_ckeditor.models import Text
from djangocms_picture.models import Picture

from cms import cms_toolbars
from cms.models import CMSPlugin

from menus.utils import set_language_changer

from aldryn_newsblog import views
from aldryn_newsblog.views import (
    ArticleDetail, CategoryArticleList, TagArticleList)
from aldryn_newsblog.models import Article

from Zephyrus.landing import ErrorCode
from Zephyrus.landing.models import Email
from Zephyrus.blog_extension.models import CallToActionModel, CallToActionStatistics
from Zephyrus.blog_extension import settings as _settings

logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)
logger.addHandler(logging.StreamHandler())

PDF_ROOT = 'pdf/'


def renderArticle(request, articleId):
    article = get_object_or_404(Article, pk=hashid_to_int(articleId))
    domain = Site.objects.get_current().domain
    context = {'article': article, 'filename': f'{domain}-{article.slug}'}
    return render(request, _settings.CALL_TO_ACTION_PDF_BASE_TEMPLATE , context)

class CallToActionEmailView(View):
    def post(self, request, *args, **kwargs):
        email = request.POST.get('email')
        fromUrl = request.POST.get('fromUrl')
        redirectTo = request.POST.get('redirectTo')
        # formId = request.POST.get('form_id')
        actionId = request.POST.get('callToActionId')
        return sendAndRedirect(request, email, fromUrl, redirectTo, actionId)
    def get(self, request, *args, **kwargs):
        return redirect('/')


def sendAndRedirect(request, email, fromUrl, redirectTo, actionId, formId=None, actionFormId=None):
    errorCode = ErrorCode.NORMAL

    # this is for djangocms form default
    if 'form_id' in request.POST:
        formId = hashid_to_int(request.POST.get('form_id'))
        landingForm = get_object_or_404(FormDefinition, pk=formId)
        form = FormBuilder(landingForm, data=request.POST, files=request.FILES)
        if not form.is_valid():
            return redirect(fromUrl)

        order = landingForm.fields.all()
        excludedFieldTypes = ['MathCaptcha', 'ReCaptcha']
        order = [field for field in order
                    if field.field_type not in excludedFieldTypes]

        dataDict = {}
        for field in order:
            key = slugify(unidecode(field.label).replace('-', '_'))
            value = form.cleaned_data.get(key, '(no input)')
            dataDict[key] = {'value': value, 'field': field,}

    elif formId:
        formId = hashid_to_int(formId)
        landingForm = get_object_or_404(FormDefinition, pk=formId)
        dataDict = {}
        form = True

    # this is for only social login
    else:
        landingForm = None
        form = None

    # for embed file, external link or PDF
    action = get_object_or_404(
        CallToActionModel,
        cmsplugin_ptr_id=actionId
    )
    m = hashlib.md5()
    data = f"{email}{actionId}"
    m.update(data.encode("utf-8"))
    token = m.hexdigest()
    actionRecord, created = CallToActionStatistics.objects.get_or_create(
        action=action,
        email=email,
        url=fromUrl,
        defaults={'token':token, 'count':0}
    )
    # if not created:
    #     actionRecord.count = actionRecord.count + 1
    actionRecord.save()
    domain = Site.objects.get_current().domain

    hashedActionId = int_to_hashid(int(actionId))
    link = f"{settings.META_SITE_PROTOCOL}://{domain}{reverse('showTargetFile')}?key={hashedActionId}"
    fileData = {'link': link, 'token': token}


    try:
        validate_email(email)

        # Save data to db
        try:
            if not form:
                email = Email.objects.create(
                    email=email,
                    fromUrl=fromUrl,
                )
                email.save()
            else:
                data = {}
                for key in dataDict:
                    value = dataDict[key]['value']
                    data[key] = value
                email = Email.objects.create(
                    email=email,
                    fromUrl=fromUrl,
                    data=data,
                )
                email.save()
        except Exception as e:
            errorCode = ErrorCode.SAVE_EMAIL
            logger.error('Not successfully save email: %s' % str(e))

        # Send email
        try:
            message = (
                "Customer id : %d\nCustomer email : %s\n" %
                (email.id, email.email))
            message += "Date : %s\n" % datetime.datetime.now()
            message += "From : %s\n" % fromUrl

            if not form:
                title = (
                    '[Call To Action %s] - customer use Call To Action Plugin' %
                    request.META['HTTP_HOST'])
            else:
                title = (
                    '[Call To Action %s] - %s' % (
                    request.META['HTTP_HOST'], landingForm.name))

            if not form:
                send_mail(
                    title,
                    message,
                    # settings.EMAIL_HOST_USER,
                    settings.DEFAULT_FROM_EMAIL,
                    settings.ZEPHYRUS_LANDING_SEND_TO,
                    fail_silently=False
                )
            else:

                message += "Other Data : \n"

                for key in dataDict:
                    value = dataDict[key]['value']
                    field = dataDict[key]['field']
                    message += "\t%s : %s\n" % (field.label, value)
                send_mail(
                    title,
                    message,
                    settings.DEFAULT_FROM_EMAIL,
                    landingForm.email_to.split(','),
                    fail_silently=False
                )

            replaceStr = {
                'email': email.email,
            }
            replaceStr.update(fileData)

            if not form:
                title = _settings.CALL_TO_ACTION_EMAIL_SUBJECT

            else:
                title = landingForm.email_subject

            mail = get_template('blog_extension/call_to_action/FileConfirm.txt')
            message = mail.render(replaceStr)
            send_mail(
                title,
                message,
                settings.DEFAULT_FROM_EMAIL,
                [email.email,],
                fail_silently=False
            )

        except Exception as e:
            errorCode = ErrorCode.SEND_EMAIL
            logger.error('Not successfully send email: %s' % str(e))
    except ValidationError as e:
        errorCode = ErrorCode.EMAIL_FORMAT
        logger.error('Email format error: %s' % str(e))

    # Redirect
    if redirectTo:
        if six.PY3:
            url = urllib.parse.quote(redirectTo)
        else:
            url = urllib.quote(redirectTo)
    else:
        if hasattr(settings, "ZEPHYRUS_LANDING_THANK_URL"):
            url = fromUrl + settings.ZEPHYRUS_LANDING_THANK_URL
        else:
            url = fromUrl + "thank"

    url = url + "?from_url=%s&error=%d" % (fromUrl, errorCode)


    if actionFormId:
        landingForm = get_object_or_404(FormDefinition, pk=actionFormId)

    if (not form) and (not actionFormId):
        return redirect(url)

    if landingForm:

        # landingForm.success_redirect
        if landingForm.page_redirect:
            # url = landingForm.page_redirect.get_public_url() + \'
            url = landingForm.page_redirect.get_absolute_url() + \
                "?from_url=%s&error=%d" % (fromUrl, errorCode)

        # landingForm.redirect_url
        elif landingForm.external_redirect:
            url = landingForm.external_redirect + \
                "?from_url=%s&error=%d" % (fromUrl, errorCode)

        else:
            pass

    return redirect(url)

class CallToActionSocialView(View):
    def get(self, request, *args, **kwargs):
        user = request.user
        email = user.email
        fromUrl = request.GET.get('fromUrl')
        redirectTo = request.GET.get('redirectTo')
        formId = request.GET.get('formId')
        actionId = request.GET.get('callToActionId')
        actionFormId = hashid_to_int(request.GET.get('actionFormId'))
        auth.logout(request)
        return sendAndRedirect(
            request, email, fromUrl, redirectTo, actionId, formId=formId, actionFormId=actionFormId)


def showTargetFile(request):

    def returnTargetFile(request, callToActionStatistics):
        if callToActionStatistics.action.targetFile:

            path = settings.BASE_DIR + \
                settings.MEDIA_URL + \
                str(callToActionStatistics.action.targetFile.file)

            response = serve(request, path, document_root=settings.BASE_DIR)
            return response

        elif callToActionStatistics.action.linkUrl:
            return render(
                request,
                'blog_extension/call_to_action/ExternalFile.html',
                {'externalLink': callToActionStatistics.action.linkUrl}
            )

        else:
            domain = Site.objects.get_current().domain
            
            slug = callToActionStatistics.url.split('/')[-2]
            article = Article.objects.filter(translations__slug=slug)
            if article:
                article = article[0]
                articleId = int_to_hashid(article.pk)
                fileName = f'[{domain}]{article.slug}.pdf'
                path = settings.MEDIA_URL[1:] + PDF_ROOT + fileName
                try:
                    pdfkit.from_url(settings.META_SITE_PROTOCOL + '://' + domain + reverse('renderArticle', kwargs={'articleId': articleId}), path)

                # might occur as the https is at risk
                # the file(img or any file on site) is loaded successfully
                # with network status code 2 and http status code 0 - Connection closed
                # Warning: SSL error ignored
                # Done but Exit with code 1 due to network error: RemoteHostClosedError
                except Exception as e:
                    pass


            else:
            
                # make PDF directly
                
                resp = requests.get(f"{settings.META_SITE_PROTOCOL}://{domain}{callToActionStatistics.url}")
                soup = BeautifulSoup(resp.text, 'html.parser')
                title = soup.find('title')
                title = title.text.replace(' ', '')
                title = title.replace('\n', '-')

                fileName = f"{title}.pdf"


                path = settings.MEDIA_URL[1:] + PDF_ROOT + fileName


                pdfkit.from_url(
                    f"{settings.META_SITE_PROTOCOL}://{domain}{callToActionStatistics.url}",
                    path,
                )
            response = serve(request, path, document_root=settings.BASE_DIR)

            return response

    key = request.GET.get('key')
    if request.POST.get('token'):
        actionRecord = CallToActionStatistics.objects.filter(
            action_id=hashid_to_int(key),
            token=request.POST.get('token'),
        )
        if actionRecord.exists():
            actionRecord = actionRecord[0]
            actionRecord.count = actionRecord.count + 1
            actionRecord.save()
            return returnTargetFile(request, actionRecord)
        else:
            return render(
                request,
                'blog_extension/call_to_action/FileVerification.html',
                {'key': key, 'msg': 'Invalid token'}
            )



    else:
        return render(
            request,
            'blog_extension/call_to_action/FileVerification.html',
            {'key': key}
        )

class ArticleDetail2(ArticleDetail):
    def get(self, request, *args, **kwargs):
        if not hasattr(self, 'object'):
            self.object = self.get_object()
            
        set_language_changer(request, self.object.get_absolute_url)
        url = self.object.get_absolute_url()
       
        if (self.config.non_permalink_handling == 200 or 
            django_urlquote(request.path) == url):
            return super(ArticleDetail, self).get(request, *args, **kwargs)

        if self.config.non_permalink_handling == 302:
            return HttpResponseRedirect(url)
        elif self.config.non_permalink_handling == 301:
            return HttpResponsePermanentRedirect(url)
        else:
            raise Http404('This is not the canonical uri of this object.')
            
    # 以下兩個一定要貼，不然會 maximum recursion depth exceeded in cmp
    def get_object(self, queryset=None):
        if queryset is None:
            queryset = self.get_queryset()

        slug = self.kwargs.get(self.slug_url_kwarg, None)
        pk = self.kwargs.get(self.pk_url_kwarg, None)
        
        if pk is not None:
            return DetailView.get_object(self, queryset=queryset)
        elif slug is not None:
            return super(ArticleDetail, self).get_object(queryset=queryset)

        raise AttributeError('ArticleDetail view must be called with either '
                             'an object pk or a slug')
    
    def get_context_data(self, **kwargs):
        context = super(ArticleDetail, self).get_context_data(**kwargs)
        context['prev_article'] = self.get_prev_object(
            self.queryset, self.object)
        context['next_article'] = self.get_next_object(
            self.queryset, self.object)
        return context
    
views.ArticleDetail = ArticleDetail2

def get_queryset(self):
    categories = [self.category,]
    stack = [self.category,]

    while stack:
        category = stack.pop()
        children = category.get_children()
        for child in children:
            categories.append(child)
            stack.append(child)

    return super(CategoryArticleList, self).get_queryset().filter(
        categories__in=categories
    ).distinct()

CategoryArticleList.get_queryset = get_queryset

class NoCategoryArticleList(CategoryArticleList):
    def get_queryset(self):
        return super(CategoryArticleList, self).get_queryset().filter(
            categories=None
        )
    
    def get(self, request):
        return super(CategoryArticleList, self).get(request)

    def get_context_data(self, **kwargs):
        kwargs['newsblog_category'] = _("No category")
        ctx = super(CategoryArticleList, self).get_context_data(**kwargs)
        ctx['newsblog_category'] = _("No category")
        return ctx

class NoTagArticleList(TagArticleList):
    def get_queryset(self):
        return super(TagArticleList, self).get_queryset().filter(
            tags=None
        )

    def get(self, request):
        return super(TagArticleList, self).get(request)

    def get_context_data(self, **kwargs):
        kwargs['newsblog_tag'] = _("No tag")
        return super(TagArticleList, self).get_context_data(**kwargs)

from aldryn_newsblog.urls import urlpatterns
from aldryn_newsblog.feeds import TagFeed, CategoryFeed

urlpatterns += [
    url(r'^(?P<slug>.*)/$',
        ArticleDetail2.as_view(), name='article-detail'),
    url(r'^(?P<year>\d{4})/(?P<slug>.*)/$',
        ArticleDetail2.as_view(), name='article-detail'),
    url(r'^(?P<year>\d{4})/(?P<month>\d{1,2})/(?P<slug>.*)/$',
        ArticleDetail2.as_view(), name='article-detail'),
    url(r'^(?P<year>\d{4})/(?P<month>\d{1,2})/(?P<day>\d{1,2})/(?P<slug>.*)/$',
        ArticleDetail2.as_view(), name='article-detail'),

    url(r'^category/(?P<category>.*)/$',
        CategoryArticleList.as_view(), name='article-list-by-category'),
    url(r'^category/(?P<category>.*)/feed/$',
        CategoryFeed(), name='article-list-by-category-feed'),

    url(r'^tag/(?P<tag>.*)/$',
        TagArticleList.as_view(), name='article-list-by-tag'),
    url(r'^tag/(?P<tag>.*)/feed/$',
        TagFeed(), name='article-list-by-tag-feed'),
]

urlpatterns.insert(0, 
    url(r'^category/no_category/$',
        NoCategoryArticleList.as_view(), name='article-list-by-no-category')
)

urlpatterns.insert(0, 
    url(r'^tag/no_tag/$',
        NoTagArticleList.as_view(), name='article-list-by-no-tag')
)