#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: forms.py 11802 2019-12-17 16:04:02Z Lavender $
#
# Copyright (c) 2018 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $ (last)
# $Date: 2019-12-18 01:04:02 +0900 (週三, 18 十二月 2019) $
# $Revision: 11802 $

import os
import re
import hashlib
import datetime

from time import gmtime, strftime

from django import forms
from django.conf import settings
from django.utils.translation import gettext_lazy as _
from django.contrib import messages
from django.core.files.storage import default_storage
from django.core.files.base import ContentFile
from django.conf import settings
from django.db.models import Q

from oscar import VERSION
from oscar.core.loading import get_model, get_class, get_classes
from oscar.forms.widgets import DatePickerInput
from oscar.core.utils import datetime_combine
from ckeditor_uploader.widgets import CKEditorUploadingWidget

from Iuno.shop.models import OrdererAddress, BonusSettings

(RelatedFieldWidgetWrapper,
 ) = get_classes(
    'dashboard.widgets',
    ('RelatedFieldWidgetWrapper',))

# product ckeditor
catalogueViews = get_class('dashboard.catalogue', 'views')

Product = get_model('catalogue', 'Product')
_ProductForm = get_class('dashboard.catalogue.forms', 'ProductForm')

class ProductForm(_ProductForm):
    class Meta(_ProductForm.Meta):
        
        if hasattr(settings, 'IUNO_SHOP_PRODUCT_DESCRIPTION_CKEDITOR'):
            if settings.IUNO_SHOP_PRODUCT_DESCRIPTION_CKEDITOR:
                widgets = {
                    'structure': forms.HiddenInput(),
                    'description': CKEditorUploadingWidget(),
                }
            else:
                widgets = {
                    'structure': forms.HiddenInput(),
                    'description': forms.Textarea(attrs={'class': 'wysiwyg'}),
                }
        else:
            widgets = {
                'structure': forms.HiddenInput(),
                'description': forms.Textarea(attrs={'class': 'wysiwyg'}),
            }
            
catalogueViews.ProductCreateUpdateView.form_class = ProductForm

# Product Search
_ProductSearchForm = get_class('dashboard.catalogue.forms', 'ProductSearchForm')
Category = get_model('catalogue', 'Category')
ProductClass = get_model('catalogue', 'ProductClass')

class ProductSearchForm(_ProductSearchForm):
    productClass = forms.ModelChoiceField(
        label=_('Product type'),
        queryset=ProductClass.objects.all(), required=False
    )
    category = forms.ModelChoiceField(
        queryset=Category.objects.all(), 
        label=_("Categories"), required=False
    )
    dateFrom = forms.DateField(
        required=False, label=_("Date from"), widget=DatePickerInput)
    dateTo = forms.DateField(
        required=False, label=_("Date to"), widget=DatePickerInput)
   
    if VERSION >= (2, 0, 0):
        notPublic = forms.BooleanField(
            label=_('Not public'), required=False, initial=True)
    
_apply_search = catalogueViews.ProductListView.apply_search
def apply_search(self, queryset):
    self.form = self.form_class(self.request.GET)
    if not self.form.is_valid():
        return queryset
    data = self.form.cleaned_data
    
    queryset = _apply_search(self, queryset)
    
    if data.get('category'):
        queryset = queryset.filter(categories__in=[data['category'],])
    if data.get('productClass'):
        queryset = queryset.filter(product_class=data['productClass'])
    if data.get('notPublic'):
        queryset = queryset.filter(is_public=False)
    if data['dateFrom'] and data['dateTo']:
        dateTo = datetime_combine(data['dateTo'], datetime.time.max)
        dateFrom = datetime_combine(data['dateFrom'], datetime.time.min)
        queryset = queryset.filter(
            date_updated__gte=dateFrom, date_updated__lt=dateTo)
    elif data['dateFrom']:
        dateFrom = datetime_combine(data['dateFrom'], datetime.time.min)
        queryset = queryset.filter(date_updated__gte=dateFrom)
    elif data['dateTo']:
        dateTo = datetime_combine(data['dateTo'], datetime.time.max)
        queryset = queryset.filter(date_updated__lt=dateTo)
    
    return queryset
    
catalogueViews.ProductListView.form_class = ProductSearchForm
catalogueViews.ProductListView.apply_search = apply_search
catalogueViews.ProductListView.template_name = \
    'shop/dashboard/catalogue/product_list.html'
    
# customer search form
_UserSearchForm = get_class('dashboard.users.forms', 'UserSearchForm')
usersViews = get_class('dashboard.users', 'views')

class UserSearchForm(_UserSearchForm):
    notActive = forms.BooleanField(
        label=_('Not active'), required=False, initial=False)
    notStaff = forms.BooleanField(
        label=_('Not staff status'), required=False, initial=False)
    dateFrom = forms.DateField(
        required=False, label=_("Date from"), widget=DatePickerInput)
    dateTo = forms.DateField(
        required=False, label=_("Date to"), widget=DatePickerInput)
    
_apply_search2 = usersViews.IndexView.apply_search
def apply_search2(self, queryset):
    self.form = self.form_class(self.request.GET)
    if not self.form.is_valid():
        return queryset
    data = self.form.cleaned_data
    queryset = _apply_search2(self, queryset)
    
    if data.get('notActive'):
        queryset = queryset.filter(is_active=False)
    
    if data.get('notStaff'):
        queryset = queryset.filter(is_staff=False)
    
    
    if data['dateFrom'] and data['dateTo']:
        dateTo = datetime_combine(data['dateTo'], datetime.time.max)
        dateFrom = datetime_combine(data['dateFrom'], datetime.time.min)
        queryset = queryset.filter(
            date_joined__gte=dateFrom, ddate_joined__lt=dateTo)
    elif data['dateFrom']:
        dateFrom = datetime_combine(data['dateFrom'], datetime.time.min)
        queryset = queryset.filter(date_joined__gte=dateFrom)
    elif data['dateTo']:
        dateTo = datetime_combine(data['dateTo'], datetime.time.max)
        queryset = queryset.filter(date_joined__lt=dateTo)
    
    return queryset

usersViews.IndexView.form_class = UserSearchForm
usersViews.IndexView.apply_search = apply_search2
usersViews.IndexView.template_name = 'shop/dashboard/users/index.html'

# https://code.nuwainfo.com/trac/Mercurius/ticket/3278#ticket
_ProductRecommendationForm = get_class(
    'dashboard.catalogue.forms', 'ProductRecommendationForm')
ProductSelect = get_class('dashboard.catalogue.widgets', 'ProductSelect')

_lookup_filter = catalogueViews.ProductLookupView.lookup_filter
def lookup_filter(self, qs, term):
    return qs.filter(Q(title__icontains=term)
                     | Q(parent__title__icontains=term)
                     | Q(upc__icontains=term))
def format_object(self, obj):
    return {
        'id': obj.pk,
        'text': u"%s (%s)" % (obj, obj.upc),
    }
                     
catalogueViews.ProductLookupView.lookup_filter = lookup_filter
catalogueViews.ProductLookupView.format_object = format_object

# https://code.nuwainfo.com/trac/Mercurius/ticket/3272#ticket
def process_all_forms(self, form):
    if self.creating and form.is_valid():
        self.object = form.save(commit=False)
    
    postCopy = self.request.POST.copy()
    for k, v in postCopy.items():
        if 'options-' in k and '-option' in k and v:
            postCopy[k] = u"%s" % hashlib.md5(
                str("%s%s" % (v, datetime.datetime.now())).encode()).hexdigest()
    
    attribute_option_formset = self.attribute_option_formset(
        self.request.POST, self.request.FILES, instance=self.object)
    attribute_option_formset2 = self.attribute_option_formset(
        postCopy, self.request.FILES, instance=self.object)

    is_valid = form.is_valid() and attribute_option_formset2.is_valid()

    if is_valid:
        attribute_option_formset2.save()
        return self.forms_valid(form, attribute_option_formset)
    else:
        return self.forms_invalid(form, attribute_option_formset)
        
catalogueViews.AttributeOptionGroupCreateUpdateView.process_all_forms = \
    process_all_forms
catalogueViews.AttributeOptionGroupCreateUpdateView.form_valid = \
    process_all_forms
catalogueViews.AttributeOptionGroupCreateUpdateView.form_invalid = \
    process_all_forms
    
# oderer address form
_ShippingAddressForm = get_class(
    'checkout.forms', 'ShippingAddressForm')
Country = get_model('address', 'Country')
    
class ShippingAddressForm2(_ShippingAddressForm):
    ordererFirstName = forms.CharField(
        label=_('Orderer first name'), required=True)
    ordererLastName = forms.CharField(
        label=_('Orderer last name'), required=True)
    ordererLine1 = forms.CharField(
        label=_('Orderer first line of address'), required=True)
    ordererLine4 = forms.CharField(
        label=_('Orderer city'), required=True)
    ordererCountry = forms.ModelChoiceField(
        label=_('Orderer country'),
        queryset=Country.objects.all(), required=True,
    )
    ordererPostcode = forms.CharField(
        label=_('Orderer postcode'), required=True)
        
    class Meta(_ShippingAddressForm.Meta):
        fields = [
            'ordererFirstName', 'ordererLastName', 'ordererLine1', 
            'ordererLine4', 'ordererCountry', 'ordererPostcode',
        ] + _ShippingAddressForm.Meta.fields
  
    def clean_odererPostcode(self):
        data = self.cleaned_data['ordererPostcode']
        ordererCountry = self.cleaned_data['ordererCountry']
        
        if (not data 
            and self.Meta.model.POSTCODE_REQUIRED and ordererCountry):
            countryCode = ordererCountry.iso_3166_1_a2
            regex = self.Meta.model.POSTCODES_REGEX.get(countryCode, None)
            if regex:
                msg = _("Addresses in %(country)s require a valid postcode") \
                    % {'country': self.country}
                raise exceptions.ValidationError(msg)

        if data and ordererCountry:
            # Ensure postcodes are always uppercase
            postcode = data.upper().replace(' ', '')
            countryCode = ordererCountry.iso_3166_1_a2
            regex = self.Meta.model.POSTCODES_REGEX.get(countryCode, None)

            # Validate postcode against regex for the country if available
            if regex and not re.match(regex, postcode):
                msg = _("The postcode '%(postcode)s' is not valid "
                        "for %(country)s") \
                    % {'postcode': self.postcode,
                       'country': self.country}
                raise exceptions.ValidationError(
                    {'postcode': [msg]})
                    
        return data     
            
from oscar.apps.checkout import views as checkoutViews

_getForm =  checkoutViews.ShippingAddressView.get_form
def get_form(self, **kwargs):
    form = _getForm(self, **kwargs)
    if (self.request.user.is_authenticated and 
        not OrdererAddress.objects.filter(user=self.request.user).exists()):
        return ShippingAddressForm2(**self.get_form_kwargs())
    return form
 
_getValid =  checkoutViews.ShippingAddressView.form_valid
UserAddress = get_model('address', 'UserAddress')
def form_valid(self, form):
    rt = _getValid(self, form)
    
    if (self.request.user.is_authenticated and 
        not OrdererAddress.objects.filter(user=self.request.user).exists()):
        address = UserAddress.objects.create(
            user=self.request.user,
            first_name=form.cleaned_data['ordererFirstName'],
            last_name=form.cleaned_data['ordererLastName'],
            line1=form.cleaned_data['ordererLine1'],
            line4=form.cleaned_data['ordererLine4'],
            country=form.cleaned_data['ordererCountry'],
            postcode=form.cleaned_data['ordererPostcode'],
        )
        ordererAddress, created = OrdererAddress.objects.get_or_create(
            user=self.request.user,
            defaults={
                'address': address,
            }
        )
    
    return rt

checkoutViews.ShippingAddressView.get_form = get_form
checkoutViews.ShippingAddressView.form_valid = form_valid
    
# content block ckeditor
if not VERSION >= (2, 0, 0):
    # RawHTMLForm
    promotionsViews = get_class('dashboard.promotions', 'views')

    RawHTML = get_model('promotions', 'RawHTML')
    _RawHTMLForm = get_class('dashboard.promotions.forms', 'RawHTMLForm')

    class RawHTMLForm(_RawHTMLForm):
        class Meta:
            model = RawHTML
            fields = ['name', 'body']
            
            if hasattr(settings, 'IUNO_SHOP_CONTENT_BLOCK_CKEDITOR'):
                if settings.IUNO_SHOP_CONTENT_BLOCK_CKEDITOR:
                    widgets = {
                        'body': CKEditorUploadingWidget(),
                    }
                else:
                    widgets = {
                        'body': forms.Textarea(attrs={'class': 'wysiwyg'}),
                    }
            else:
                widgets = {
                    'body': forms.Textarea(attrs={'class': 'wysiwyg'}),
                }
                
    promotionsViews.CreateRawHTMLView.form_class = RawHTMLForm
    promotionsViews.UpdateRawHTMLView.form_class = RawHTMLForm
else:
    # RawHTMLForm
    pass
    
# Pages
pagesViews = get_class('dashboard.pages', 'views')

FlatPage = get_model('flatpages', 'FlatPage')
_PageUpdateForm = get_class('dashboard.pages.forms', 'PageUpdateForm')

class PageUpdateForm(_PageUpdateForm):
    class Meta(_PageUpdateForm.Meta):
        model = FlatPage
        fields = ('title', 'url', 'content')
        
        if hasattr(settings, 'IUNO_SHOP_PAGE_CKEDITOR'):
            if settings.IUNO_SHOP_CONTENT_BLOCK_CKEDITOR:
                widgets = {
                    'content': CKEditorUploadingWidget(),
                }
            else:
                widgets = {
                    'content': forms.Textarea(attrs={'class': 'wysiwyg'}),
                }
        else:
            widgets = {
                'content': forms.Textarea(attrs={'class': 'wysiwyg'}),
            }
            
pagesViews.PageCreateView.form_class = PageUpdateForm
pagesViews.PageUpdateView.form_class = PageUpdateForm


if VERSION >= (2, 0, 0):
    # option type
    from Iuno.shop.models import (
        OptionType, TYPE_CHOICES, TEXT, INTEGER, 
        BOOLEAN, FLOAT, DATE, DATETIME, OPTION, RICHTEXT, FILE, IMAGE
    )
    from Iuppiter.extension.Widgets import FileDragDropWidget

    Option = get_model('catalogue', 'Option')
    ProductAttribute = get_model('catalogue', 'ProductAttribute')
    AttributeOptionGroup = get_model('catalogue', 'AttributeOptionGroup')
    _OptionForm = get_class('dashboard.catalogue.forms', 'OptionForm')

    class OptionForm(_OptionForm):

        optionType = forms.ChoiceField(
            choices=TYPE_CHOICES, label=_("Option type"))
        optionGroup = forms.ModelChoiceField(
            queryset=AttributeOptionGroup.objects.all(), 
            label=_("Option group")
        )
        
        def __init__(self, *args, **kwargs):
            if "instance" in kwargs:
                instance = kwargs["instance"]
                if instance:
                    ot, created = OptionType.objects.get_or_create(
                        option=instance,
                        defaults={
                            'type': TEXT, 
                            'optionGroup': None,
                        }
                    )
                    kwargs["initial"] = {
                        "optionType": ot.type,
                        "optionGroup": ot.optionGroup,
                    }
               
            super().__init__(*args, **kwargs)
            
            self.fields["optionGroup"].required = False
            self.fields["optionGroup"].help_text = _("Select an option group")
            
            remote_field = ProductAttribute._meta.get_field(
                'option_group').remote_field
            self.fields["optionGroup"].widget = RelatedFieldWidgetWrapper(
                self.fields["optionGroup"].widget, remote_field)
                
        def save(self, commit=True):
            instance = super(OptionForm, self).save(commit=commit)
            type = self.cleaned_data['optionType']
            optionGroup = self.cleaned_data['optionGroup']
            
            ot, created = OptionType.objects.update_or_create(
                option=instance,
                defaults={
                    'type': type, 
                    'optionGroup': optionGroup,
                }
            )
            if commit:
                instance.save()
            return instance
                
        class Meta:
            model = Option
            fields = ['name', 'type']

    catalogueViews.OptionCreateUpdateView.form_class = OptionForm

    from oscar.apps.basket import views as basketViews
    from oscar.templatetags import basket_tags
    from oscar.forms.widgets import DatePickerInput, DateTimePickerInput

    _AddToBasketForm = get_class('basket.forms', 'AddToBasketForm')

    class AddToBasketForm(_AddToBasketForm):

        def _add_option_field(self, product, option):
            ot, created = OptionType.objects.get_or_create(
                option=option,
                defaults={
                    'type': TEXT, 
                    'optionGroup': None,
                }
            )
            kwargs = {
                'required': option.is_required,
                'label': option.name,
            }
            
            if ot.type == OPTION:
                # OPTION
                choice = []
                if ot.optionGroup:
                    choiceOptions = [
                        o.option 
                        for o in ot.optionGroup.options.all().order_by('id')]
                    
                    for c in choiceOptions:
                        choice.append((c, c))
                    
                kwargs.update({
                    'widget': forms.Select,
                    'choices': choice,
                })
                self.fields[option.code] = forms.ChoiceField(**kwargs)
            elif ot.type == INTEGER:
                # INTEGER
                self.fields[option.code] = forms.IntegerField(**kwargs)
            elif ot.type == BOOLEAN:
                # Boolean
                self.fields[option.code] = forms.BooleanField(**kwargs)
            elif ot.type == FLOAT:
                # FLOAT
                self.fields[option.code] = forms.FloatField(**kwargs)
            elif ot.type == RICHTEXT:
                # RICHTEXT
                kwargs.update({
                    'widget': forms.Textarea,
                })
                self.fields[option.code] = forms.CharField(**kwargs)
            elif ot.type == DATE:
                # DATE
                kwargs.update({
                    'widget': DatePickerInput,
                })
                self.fields[option.code] = forms.DateField(**kwargs)
            elif ot.type == DATETIME:
                # DATETIME
                kwargs.update({
                    'widget': DateTimePickerInput,
                })
                self.fields[option.code] = forms.DateTimeField(**kwargs)
            elif ot.type == FILE:
                # FILE
                kwargs.update({
                    'widget': forms.FileInput,
                })
                self.fields[option.code] = forms.FileField(**kwargs)
            elif ot.type == IMAGE:
                # IMAGE
                kwargs.update({
                    'widget': FileDragDropWidget
                })
                self.fields[option.code] = forms.ImageField(**kwargs)
            else:
                # TEXT
                self.fields[option.code] = forms.CharField(**kwargs)
                
    class SimpleAddToBasketForm(AddToBasketForm):
        def __init__(self, *args, **kwargs):
            super().__init__(*args, **kwargs)
            if 'quantity' in self.fields:
                self.fields['quantity'].initial = 1
                self.fields['quantity'].widget = forms.HiddenInput()
    
    BasketMessageGenerator = get_class('basket.utils', 'BasketMessageGenerator')
    
    class BasketAddView(basketViews.BasketAddView):
    
        form_class = AddToBasketForm 
        
        def form_valid(self, form): 
            # save file
            options = form.cleaned_options()
            
            for optionData in options:
                option = optionData['option']
                ot, created = OptionType.objects.get_or_create(
                    option=option,
                    defaults={
                        'type': TEXT, 
                        'optionGroup': None,
                    }
                )
                if ot.type in [FILE, IMAGE]:
                    f = self.request.FILES.get(option.code, None)
                    if f:
                        pathId = hashlib.md5(
                            ("%s%s" % (
                                str(f),
                                datetime.datetime.now(),
                            )).encode()).hexdigest()
                        mediaPath = os.path.join(
                            'OptionUpload', pathId, str(f))
                        uploadPath = os.path.join(
                            settings.MEDIA_ROOT, mediaPath)
                        if os.path.exists(uploadPath):
                            os.remove(uploadPath)
                        
                        path = default_storage.save(
                            mediaPath, 
                            ContentFile(f.read())
                        )
                        
                        optionData['value'] = mediaPath
                    else:
                        optionData['value'] = ''
                    
            # 以下複製的，因為要修改其中一段
            offers_before = self.request.basket.applied_offers()

            line, created = self.request.basket.add_product(
                form.product, form.cleaned_data['quantity'],
                options)

            messages.success(self.request, self.get_success_message(form),
                             extra_tags='safe noicon')

            # Check for additional offer messages
            BasketMessageGenerator().apply_messages(self.request, offers_before)

            # Send signal for basket addition
            self.add_signal.send(
                sender=self, product=form.product, user=self.request.user,
                request=self.request)

            return super(basketViews.BasketAddView, self).form_valid(form)
      
    #from oscar.apps.basket import apps
    basketViews.BasketAddView.form_class = AddToBasketForm 
    basketViews.BasketAddView.form_valid = BasketAddView.form_valid   
    #apps.BasketConfig.add_view = BasketAddView
    basket_tags.AddToBasketForm = AddToBasketForm
    basket_tags.SimpleAddToBasketForm = SimpleAddToBasketForm
    
# https://code.nuwainfo.com/trac/Mercurius/ticket/3285#ticket
class BonusSettingsForm(forms.ModelForm):
    class Meta:
        model = BonusSettings
        fields = ['consumption', 'percent', 'discount', 
                  'limit', 'receiveStatus', 'includeRange', 'excludeRange']
        
class BonusForm(forms.Form):
    quantity = forms.IntegerField(
        label=_("Quantity"), min_value=0)
        
    def __init__(self):
        super().__init__()
 
# https://code.nuwainfo.com/trac/Mercurius/ticket/3289#ticket 
from oscar.apps.customer import views as customerViews

_EmailUserCreationForm = get_class("customer.forms", "EmailUserCreationForm")
        
class EmailUserCreationForm(_EmailUserCreationForm):
    timeStamp = forms.CharField(
        label=_("Time Stamp"),
        widget=forms.HiddenInput,
        required=True
    )

    confirmationCode = forms.CharField(
        label=_("Confirm Code"),
        max_length=30,
        widget=forms.TextInput(attrs={'class' : 'form-control'}),
        required=True
    )
    
    def __init__(self, *args, **kwargs):
        super(EmailUserCreationForm, self).__init__(*args, **kwargs)
        if 'initial' in kwargs:
            self.fields['email'].initial = kwargs['initial']['email']

        self.fields['timeStamp'].initial = strftime(
            "%Y-%m-%d %H:%M:%S", gmtime())

    def clean_confirmationCode(self):
        """
        Checks for whether confirm code is correct.
        """
        confirmationCode = self.cleaned_data['confirmationCode']

        m = hashlib.md5()
        m.update(self.cleaned_data.get('email', '').encode())
        m.update(self.cleaned_data.get('timeStamp', '').encode())
        code = m.hexdigest()[5:10]

        if not code == confirmationCode:
            raise forms.ValidationError(
                _("Confirm code doesn't match."))
        return confirmationCode
        
customerViews.AccountRegistrationView.form_class = EmailUserCreationForm

# https://code.nuwainfo.com/trac/Mercurius/ticket/3290#ticket

_OrderNoteForm = get_class("dashboard.orders.forms", "OrderNoteForm")

class OrderNoteForm(_OrderNoteForm):

    CHOICES = [
        ('Normal', _('Normal')),
        ('Display for customer', _('Display for customer')),
    ]

    def __init__(self, order, user, *args, **kwargs):
        super().__init__(order, user, *args, **kwargs)
        self.fields['note_type'].widget.choices = self.CHOICES

    class Meta(_OrderNoteForm.Meta):
        fields = ['note_type', 'message',]
        widgets = {
            'note_type': forms.Select,
        }
        
    
